/*
 * pdsswave.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */


#include <pds/pdsswave.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsSWave                                               ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsSWave *pds_swave_new(PdsDsReal A,PdsDsReal Teta)
 *  \brief Crea una estructura de tipo PdsSWave.
 * 
 *  El primer valor de la estructura es y[0]=A*sen(Teta).
 *  \param[in] A Es la amplitud de onda sinusoidal.
 *  \param[in] Teta \f$\theta\f$ Es la frecuencia normalizada a 2pi para Fsampling, en la
 *  onda sinusoidal.
 *  \return Un puntero a una estructura de tipo PdsSWave.
 *  \ingroup PdsSWaveGroup
 */
PdsSWave *pds_swave_new(PdsDsReal A,PdsDsReal Teta)
{
	PdsSWave *WAVE=NULL;

	WAVE=(PdsSWave*)calloc(1,sizeof(PdsSWave));
	if(WAVE==NULL)	return NULL;

	WAVE->A=A;
	WAVE->Teta=Teta;
	WAVE->a=2.0*cos(Teta);

	WAVE->Y[0]=A*sin(Teta);
	WAVE->Y[1]=0;
	WAVE->Y[2]=0;

	return WAVE;
}


/** \fn PdsSWave *pds_swave_new_nos(PdsDsReal A,PdsDsNatural N)
 *  \brief Crea una estructura de tipo PdsSWave.
 * 
 *  El primer valor de la estructura es y[0]=A*sen(Teta).
 *  \param[in] A Es la amplitud de onda sinusoidal.
 *  \param[in] N Número de muestras por periodo en la onda sinusoidal.
 *  \return Un puntero a una estructura de tipo PdsSWave.
 *  \ingroup PdsSWaveGroup
 */
PdsSWave *pds_swave_new_nos(PdsDsReal A,PdsDsNatural N)
{
	PdsSWave *WAVE=NULL;

	WAVE=(PdsSWave*)calloc(1,sizeof(PdsSWave));
	if(WAVE==NULL)	return NULL;

	WAVE->A=A;
	WAVE->Teta=2*M_PI/N;
	WAVE->a=2.0*cos(WAVE->Teta);

	WAVE->Y[0]=A*sin(WAVE->Teta);
	WAVE->Y[1]=0;
	WAVE->Y[2]=0;

	return WAVE;
}


/** \fn int pds_swave_get_value(PdsSWave *WAVE,PdsDsReal *y)
 *  \brief Obtiene un valor en la salida e itera el generador sinusoidal.
 *  \param[in,out] WAVE La estructura de PdsSWave a consultar.
 *  \param[out] y En esta variable se carga el  nuevo valor a la salida de generador.
 *  \return TRUE si todo fue bien o FALSE si no. Ejemplo WAVE==NULL.
 *  \ingroup PdsSWaveGroup
 */
int pds_swave_get_value(PdsSWave *WAVE,PdsDsReal *y)
{
	if(WAVE==NULL)	return FALSE;

	*y        =WAVE->Y[0];

	WAVE->Y[2]=WAVE->Y[1];
	WAVE->Y[1]=WAVE->Y[0];
	WAVE->Y[0]=WAVE->a*WAVE->Y[1]-WAVE->Y[2];

	return TRUE;
}


/** \fn int pds_swave_get_last_value(PdsSWave *WAVE,PdsDsReal *y)
 *  \brief Obtiene el último valor a la salida del generador sinusoidal.
 *  \param[in,out] WAVE La estructura de PdsSWave a consultar.
 *  \param[out] y En esta variable se carga el último valor a la salida de generador.
 *  \return TRUE si todo fue bien o FALSE si no. Ejemplo WAVE==NULL.
 *  \ingroup PdsSWaveGroup
 */
int pds_swave_get_last_value(PdsSWave *WAVE,PdsDsReal *y)
{
	if(WAVE==NULL)	return FALSE;

	*y  =WAVE->Y[1];

	return TRUE;
}


/** \fn int pds_swave_get_vector(PdsSWave *WAVE,PdsVector *Vector)
 *  \brief Carga un vector con valores obtenidos en la salida del 
 *  generador sinusoidal. Realiza una iteración para cada elemento del vector.
 *  \param[in,out] WAVE La estructura de PdsSWave a consultar.
 *  \param[out] Vector En este vector se cargará los valores a la salida de generador.
 *  \return TRUE si todo fue bien o FALSE si no. Ejemplo WAVE==NULL, Vector==NULL.
 *  \ingroup PdsSWaveGroup
 */
int pds_swave_get_vector(PdsSWave *WAVE,PdsVector *Vector)
{
	PdsRaNatural i;
	if(WAVE==NULL)		return FALSE;
	if(Vector==NULL)	return FALSE;

	for(i=0;i<Vector->Nel;i++)
	{
		Vector->V[i]=WAVE->Y[0];

		WAVE->Y[2]=WAVE->Y[1];
		WAVE->Y[1]=WAVE->Y[0];
		WAVE->Y[0]=WAVE->a*WAVE->Y[1]-WAVE->Y[2];
	}

	return TRUE;
}

/** \fn void pds_swave_free(PdsSWave *WAVE)
 *  \brief Libera una estructura de tipo puntero PdsSWave.
 *  \param[in,out] WAVE La estructura de PdsSWave a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsSWaveGroup
 */
void pds_swave_free(PdsSWave *WAVE)
{
	if(WAVE!=NULL)
	{
		free(WAVE);
	}
}


/** \fn void pds_swave_destroy(PdsSWave **WAVE)
 *  \brief Libera una estructura de tipo puntero PdsSWave, y lo carga con NULL.
 *  \param[in,out] WAVE La estructura de PdsSWave a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsSWaveGroup
 */
void pds_swave_destroy(PdsSWave **WAVE)
{
	if((*WAVE)!=NULL)
	{
		free(*WAVE);
		(*WAVE)=NULL;
	}
}


