/*
 * pdsuniformquantizer.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsuniformquantizer.h
 *  \author Fernando Pujaico Rivera
 *  \date 02-11-2012
 *  \brief Funciones que trabajan con un cuantizador uniforme de entrada uniforme.
 *  
 *  <br><br>
 *  \image html UniformQ.png "Cuantizador para una p.d.f de entrada Uniforme."
 */

#ifndef __PDSUNIFORMQUANTIZER_H__
#define __PDSUNIFORMQUANTIZER_H__

#ifdef __cplusplus
extern "C" {
#endif 

#include <pds/pdsdsglobal.h>
#include <stdlib.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

/** \defgroup PdsUniformQuantizerGroup Funciones del módulo PdsUniformQuantizer.
 *
 *  <br><br>
 *  Se genera un cuantizador no uniforme, optimizado para trabajar con una señal de
 *  de entrada con distribución de probabilidad uniforme entre los valores A y B.
 *  Los intervalos de decisión del cuantizador están repartidos de modo 
 *  que todos los índices sean equiprobables.
 *  <br>
 *  \image html UniformQ.png "Cuantizador para una p.d.f de entrada Uniforma."
 * @{
 */

/*! \struct PdsUniformQuantizer
 *  \brief La estructura tipo  PdsUniformQuantizer.
 *
 *  Esta estructura crea un cuantizador uniforme de entrada de p.d.f. uniforme.
 *  Para usar incluir pds/pdsds.h.
 *  <br><br>
 *  \image html UniformQ.png "Cuantizador para una p.d.f de entrada Uniforme."
 *  \ingroup PdsUniformQuantizerGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct
{
	/*! Valor inferior de la entrada del cuantizador. */
	PdsDsReal A;
	/*! Valor superior de la entrada del cuantizador. */
	PdsDsReal B;
	/*! Numero de bits. */
	PdsDsNatural Bits;
	/*! Numero de niveles de cuantificación. N=2^Bits */
	PdsDsNatural N;
	/*! Probabilidad de cada índice. P=1/N. */
	PdsDsReal P;
	/*! Extremos de los niveles de decisión del cuantizador. */
	PdsDsReal *X;
	/*! Valor representativo de los niveles de cuantificación*/
	PdsDsReal *Y;

}PdsUniformQuantizer;


/** \fn PdsUniformQuantizer *pds_uniform_quantizer_new(PdsDsReal A, PdsDsReal B, PdsDsNatural Bits)
 *  \brief Crea una estructura de tipo PdsUniformQuantizer.
 * 
 *  Crea un cuantizador uniforme, optimizado para trabajar con una señal de
 *  de entrada con distribución de probabilidad uniforme entre los valores A y B. 
 *  Los intervalos de decisión del cuantizador están repartidos de modo 
 *  que todos los índices sean equiprobables.
 *  \param[in] A Es el valor inferior esperado de la señal de entrada del cuantizador.
 *  \param[in] B Es el valor superior esperado de la señal de entrada del cuantizador.
 *  \param[in] Bits Es el número de bits del cuantizador.
 *  \return Un puntero a una estructura de tipo PdsUniformQuantizer.
 *  \ingroup PdsUniformQuantizerGroup
 */
PdsUniformQuantizer *pds_uniform_quantizer_new(PdsDsReal A, PdsDsReal B, PdsDsNatural Bits);


/** \fn int pds_uniform_quantizer_get_id(const PdsUniformQuantizer *Q, PdsDsReal Valor, PdsDsNatural *Id)
 *  \brief Devuelve el índice correspondiente al nivel de cuantización del Valor.
 * 
 *  Esta función devuelve un índice entre 0 y Q->N-1.
 *  \param[in] Q Cuantizador uniforme para una distribución uniforma.
 *  \param[in] Valor Valor de entrada del cuantizador.
 *  \param[out] Id Índice correspondiente al Valor de entrada del cuantizador.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsUniformQuantizerGroup
 */
int pds_uniform_quantizer_get_id(const PdsUniformQuantizer *Q, PdsDsReal Valor, PdsDsNatural *Id);


/** \fn int pds_uniform_quantizer_get_value(const PdsUniformQuantizer *Q, PdsDsNatural Id, PdsDsReal *Valor)
 *  \brief Devuelve un valor representativo correspondiente al nivel de cuantificación  del Id.
 * 
 *  Esta función devuelve un valor representativo al índice Id.
 *  \param[in] Q Cuantizador uniforme para una distribución uniforme.
 *  \param[in] Id Índice correspondiente a consultar.
 *  \param[out] Valor Valor correspondiente en la entrada del cuantizador.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsUniformQuantizerGroup
 */
int pds_uniform_quantizer_get_value(const PdsUniformQuantizer *Q, PdsDsNatural Id, PdsDsReal *Valor);


/** \fn void pds_uniform_quantizer_free(PdsUniformQuantizer *Q)
 *  \brief Libera la memoria correspondiente al cuantizador Q.
 *
 *  \param[in] Q Cuantizador uniforme a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsUniformQuantizerGroup
 */
void pds_uniform_quantizer_free(PdsUniformQuantizer *Q);


/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif /* __PDSUNIFORMQUANTIZER_H__ */
