/*
 * pdsgaussianquantizer.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsgaussianquantizer.h
 *  \author Fernando Pujaico Rivera
 *  \date 02-11-2012
 *  \brief Funciones que trabajan con un cuantizador no uniforme de entrada gaussiana.
 *  
 *  <br><br>
 *  \image html GaussianQ.png "Cuantizador para una p.d.f de entrada Gaussiana."
 */

#ifndef __PDSGAUSSIANQUANTIZER_H__
#define __PDSGAUSSIANQUANTIZER_H__

#ifdef __cplusplus
extern "C" {
#endif 

#include <pds/pdsdsglobal.h>
#include <stdlib.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

/** \defgroup PdsGaussianQuantizerGroup Funciones del módulo PdsGaussianQuantizer.
 *
 *  <br><br>
 *  Se genera un cuantizador no uniforme, optimizado para trabajar con una señal de
 *  de entrada con distribución de probabilidad gaussiana con media Media y varianza
 *  Sigma^2. Los intervalos de decisión del cuantizador están repartidos de modo 
 *  que todos los índices sean equiprobables.
 *  <br>
 *  \image html GaussianQ.png "Cuantizador para una p.d.f de entrada Gaussiana."
 * @{
 */

/*! \struct PdsGaussianQuantizer
 *  \brief La estructura tipo  PdsGaussianQuantizer.
 *
 *  Esta estructura crea un cuantizador no uniforme de entrada de p.d.f. gaussiana.
 *  Para usar incluir pds/pdsds.h.
 *  <br><br>
 *  \image html GaussianQ.png "Cuantizador para una p.d.f de entrada Gaussiana."
 *  \ingroup PdsGaussianQuantizerGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct
{
	/*! Media. */
	PdsDsReal Media;
	/*! Desvio padron. */
	PdsDsReal Sigma;
	/*! Numero de bits. */
	PdsDsNatural Bits;
	/*! Numero de niveles de cuantificación. N=2^Bits */
	PdsDsNatural N;
	/*! Probabilidad de cada índice. P=1/N. */
	PdsDsReal P;
	/*! Extremos de los niveles de decisión del cuantizador. */
	PdsDsReal *X;
	/*! Valor representativo de los niveles de cuantificación*/
	PdsDsReal *Y;

}PdsGaussianQuantizer;

/** \fn PdsGaussianQuantizer *pds_gaussian_quantizer_new(PdsDsReal Media, PdsDsReal Sigma, PdsDsNatural Bits)
 *  \brief Crea una estructura de tipo PdsGaussianQuantizer.
 * 
 *  Crea un cuantizador no uniforme, optimizado para trabajar con una señal de
 *  de entrada con distribución de probabilidad gaussiana con media Media y varianza
 *  Sigma^2. Los intervalos de decisión del cuantizador están repartidos de modo 
 *  que todos los índices sean equiprobables.
 *  Ejecutar esta orden puede tardar un poco a partir de 8 bits en adelante.
 *  \param[in] Media Es el valor medio esperado de la señal de entrada del cuantizador.
 *  \param[in] Sigma Es el desvío padrón de la señal de entrada del cuantizador.
 *  \param[in] Bits Es el número de bits del cuantizador.
 *  \return Un puntero a una estructura de tipo PdsGaussianQuantizer.
 *  \ingroup PdsGaussianQuantizerGroup
 */
PdsGaussianQuantizer *pds_gaussian_quantizer_new(PdsDsReal Media, PdsDsReal Sigma, PdsDsNatural Bits);


/** \fn PdsGaussianQuantizer *pds_gaussian_quantizer_new_quantizer(PdsGaussianQuantizer *Q0)
 *  \brief Crea una estructura de tipo PdsGaussianQuantizer como una copia de otra.
 * 
 *  Crea un cuantizador no uniforme, copiando los parámetros de otro cuantizador 
 *  uniforme. Esta función es rápida comparado a hacer pds_gaussian_quantizer_new.
 *  \param[in] Q0 Cuantizador al que se le copiaran los parámetros.
 *  \return Un puntero a una estructura de tipo PdsGaussianQuantizer.
 *  \ingroup PdsGaussianQuantizerGroup
 */
PdsGaussianQuantizer *pds_gaussian_quantizer_new_quantizer(PdsGaussianQuantizer *Q0);


/** \fn int pds_gaussian_quantizer_get_id(const PdsGaussianQuantizer *Q, PdsDsReal Valor, PdsDsNatural *Id)
 *  \brief Devuelve el índice correspondiente al nivel de cuantización del Valor.
 * 
 *  Esta función devuelve un índice entre 0 y Q->N-1.
 *  \param[in] Q Cuantizador no uniforme para una distribución gaussiana.
 *  \param[in] Valor Valor de entrada del cuantizador.
 *  \param[out] Id Índice correspondiente al Valor de entrada del cuantizador.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsGaussianQuantizerGroup
 */
int pds_gaussian_quantizer_get_id(const PdsGaussianQuantizer *Q, PdsDsReal Valor, PdsDsNatural *Id);


/** \fn int pds_gaussian_quantizer_get_value(const PdsGaussianQuantizer *Q, PdsDsNatural Id, PdsDsReal *Valor)
 *  \brief Devuelve un valor representativo correspondiente al nivel de cuantización  del Id.
 * 
 *  Esta función devuelve un valor representativo al índice Id.
 *  \param[in] Q Cuantizador no uniforme para una distribución gaussiana.
 *  \param[in] Id Índice correspondiente a consultar.
 *  \param[out] Valor Valor correspondiente en la entrada del cuantizador.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsGaussianQuantizerGroup
 */
int pds_gaussian_quantizer_get_value(const PdsGaussianQuantizer *Q, PdsDsNatural Id, PdsDsReal *Valor);


/** \fn void pds_gaussian_quantizer_free(PdsGaussianQuantizer *Q)
 *  \brief Libera la memoria correspondiente al cuantizador Q.
 *
 *  \param[in] Q Cuantizador no uniforme a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsGaussianQuantizerGroup
 */
void pds_gaussian_quantizer_free(PdsGaussianQuantizer *Q);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif /* __PDSGAUSSIANQUANTIZER_H__ */
