/*
 * pdsuniformquantizer.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */


#include <stdlib.h>
#include <math.h>
#include <pds/pdsuniformquantizer.h>

////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsUniformQuantizer                                   ////
////////////////////////////////////////////////////////////////////////////////

/** \fn PdsUniformQuantizer *pds_uniform_quantizer_new(PdsDsReal A, PdsDsReal B, PdsDsNatural Bits)
 *  \brief Crea una estructura de tipo PdsUniformQuantizer.
 * 
 *  Crea un cuantizador uniforme, optimizado para trabajar con una señal de
 *  de entrada con distribución de probabilidad uniforme entre los valores A y B. 
 *  Los intervalos de decisión del cuantizador están repartidos de modo 
 *  que todos los índices sean equiprobables.
 *  \param[in] A Es el valor inferior esperado de la señal de entrada del cuantizador.
 *  \param[in] B Es el valor superior esperado de la señal de entrada del cuantizador.
 *  \param[in] Bits Es el número de bits del cuantizador.
 *  \return Un puntero a una estructura de tipo PdsUniformQuantizer.
 *  \ingroup PdsUniformQuantizerGroup
 */
PdsUniformQuantizer *pds_uniform_quantizer_new(PdsDsReal A, PdsDsReal B, PdsDsNatural Bits)
{
	PdsUniformQuantizer *Q=NULL;
	PdsDsNatural i;
	PdsDsReal d;

	if(A>=B)	return NULL;

	if(Bits>(8*sizeof(PdsDsNatural)))	return NULL;

	Q=(PdsUniformQuantizer *)calloc(1,sizeof(PdsUniformQuantizer));
	if(Q==NULL)	return NULL;
	////////////////////////////////////////////////////////////////////////
	Q->A=A;
	Q->B=B;
	Q->Bits=Bits;
	Q->N=(PdsDsNatural)pow(2.0,Bits);
	Q->P=1.0/Q->N;
	////////////////////////////////////////////////////////////////////////
	Q->X=(PdsDsReal *)calloc(Q->N,sizeof(PdsDsReal));
	if(Q->X==NULL)
	{	
		free(Q);
		return NULL;
	}
	////////////////////////////////////////////////////////////////////////
	Q->Y=(PdsDsReal *)calloc(Q->N,sizeof(PdsDsReal));
	if(Q->Y==NULL)
	{	
		free(Q->X);
		free(Q);
		return NULL;
	}

	d=(Q->B-Q->A)/Q->N;
	////////////////////////////////////////////////////////////////////////
	for(i=0;i<=(Q->N-2);i++)
	{
		Q->X[i]=Q->A+d*(1.0+i);
	}

	////////////////////////////////////////////////////////////////////////
	for(i=0;i<=(Q->N-1);i++)
	{
		Q->Y[i]=Q->A+d*(1.0+2.0*i)/2.0;
	}
	return Q;
}


/** \fn int pds_uniform_quantizer_get_id(const PdsUniformQuantizer *Q, PdsDsReal Valor, PdsDsNatural *Id)
 *  \brief Devuelve el índice correspondiente al nivel de cuantización del Valor.
 * 
 *  Esta función devuelve un índice entre 0 y Q->N-1.
 *  \param[in] Q Cuantizador uniforme para una distribución uniforma.
 *  \param[in] Valor Valor de entrada del cuantizador.
 *  \param[out] Id Índice correspondiente al Valor de entrada del cuantizador.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsUniformQuantizerGroup
 */
int pds_uniform_quantizer_get_id(const PdsUniformQuantizer *Q, PdsDsReal Valor, PdsDsNatural *Id)
{
	PdsDsNatural i;

	if(Q==NULL)	return FALSE;

	if(Valor<=Q->X[0])	
	{
		*Id=0;
	}
	else if(Valor>Q->X[Q->N-2])
	{
		*Id=Q->N-1;
	}
	else
	{
		for(i=0;i<Q->N-2;i++)
		{
			if(( Valor>Q->X[i] )&&( Valor<=Q->X[i+1] ))
			{
				*Id=i+1;
			}
		}
	}
	return TRUE;
}

/** \fn int pds_uniform_quantizer_get_value(const PdsUniformQuantizer *Q, PdsDsNatural Id, PdsDsReal *Valor)
 *  \brief Devuelve un valor representativo correspondiente al nivel de cuantificación  del Id.
 * 
 *  Esta función devuelve un valor representativo al índice Id.
 *  \param[in] Q Cuantizador uniforme para una distribución uniforme.
 *  \param[in] Id Índice correspondiente a consultar.
 *  \param[out] Valor Valor correspondiente en la entrada del cuantizador.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsUniformQuantizerGroup
 */
int pds_uniform_quantizer_get_value(const PdsUniformQuantizer *Q, PdsDsNatural Id, PdsDsReal *Valor)
{
	if(Q==NULL)	return FALSE;
	if(Id>=Q->N)
	{
		*Valor=Q->Y[Q->N-1];
		return FALSE;
	}
	else
	{
		*Valor=Q->Y[Id];
		return TRUE;
	}
}

/** \fn void pds_uniform_quantizer_free(PdsUniformQuantizer *Q)
 *  \brief Libera la memoria correspondiente al cuantizador Q.
 *
 *  \param[in] Q Cuantizador uniforme a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsUniformQuantizerGroup
 */
void pds_uniform_quantizer_free(PdsUniformQuantizer *Q)
{
	if(Q!=NULL)
	{
		free(Q->X);
		free(Q->Y);
		free(Q);
	}
}

