/*
 * pdsiqwave.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */


#include <pds/pdsiqwave.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsIQWave                                              ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsIQWave *pds_iqwave_new(PdsDsReal A,PdsDsReal Teta)
 *  \brief Crea una estructura de tipo PdsIQWave.
 * 
 *  El primer valor de la estructura es yc[0]=A*cos(Teta) ys[0]=A*sin(Teta).
 *  \param[in] A Es la amplitud de onda IQ sinusoidal.
 *  \param[in] Teta \f$\theta\f$ Es la frecuencia normalizada a 2pi para Fsampling, en la
 *  onda IQ sinusoidal.
 *  \return Un puntero a una estructura de tipo PdsIQWave.
 *  \ingroup PdsIQWaveGroup
 */
PdsIQWave *pds_iqwave_new(PdsDsReal A,PdsDsReal Teta)
{
	PdsIQWave *WAVE=NULL;

	WAVE=(PdsIQWave*)calloc(1,sizeof(PdsIQWave));
	if(WAVE==NULL)	return NULL;

	WAVE->A=A;
	WAVE->Teta=Teta;
	WAVE->a=cos(Teta);
	WAVE->b=sin(Teta);

	WAVE->Yc[0]=A*cos(Teta);	WAVE->Yc[1]=A;
	WAVE->Ys[0]=A*sin(Teta);	WAVE->Ys[1]=0.0;

	return WAVE;
}


/** \fn PdsIQWave *pds_iqwave_new_nos(PdsDsReal A,PdsDsNatural N)
 *  \brief Crea una estructura de tipo PdsIQWave.
 * 
 *  El primer valor de la estructura es yc[0]=A*cos(Teta) ys[0]=A*sin(Teta).
 *  \param[in] A Es la amplitud de onda IQ sinusoidal.
 *  \param[in] N Número de muestras por periodo en la onda IQ sinusoidal.
 *  \return Un puntero a una estructura de tipo PdsIQWave.
 *  \ingroup PdsIQWaveGroup
 */
PdsIQWave *pds_iqwave_new_nos(PdsDsReal A,PdsDsNatural N)
{
	PdsIQWave *WAVE=NULL;

	WAVE=(PdsIQWave*)calloc(1,sizeof(PdsIQWave));
	if(WAVE==NULL)	return NULL;

	WAVE->A=A;
	WAVE->Teta=2*M_PI/N;
	WAVE->a=cos(WAVE->Teta);
	WAVE->b=sin(WAVE->Teta);

	WAVE->Yc[0]=A*cos(WAVE->Teta);	WAVE->Yc[1]=A;
	WAVE->Ys[0]=A*sin(WAVE->Teta);	WAVE->Ys[1]=0.0;

	return WAVE;
}


/** \fn int pds_iqwave_get_value(PdsIQWave *WAVE,PdsDsReal *yc,PdsDsReal *ys)
 *  \brief Obtiene un valor de yc e ys en la salida e itera el generador IQ sinusoidal.
 *  \param[in,out] WAVE La estructura de PdsIQWave a consultar.
 *  \param[out] yc En esta variable se carga el  nuevo valor a la salida de generador coseno.
 *  \param[out] ys En esta variable se carga el  nuevo valor a la salida de generador seno.
 *  \return TRUE si todo fue bien o FALSE si no. Ejemplo WAVE==NULL.
 *  \ingroup PdsIQWaveGroup
 */
int pds_iqwave_get_value(PdsIQWave *WAVE,PdsDsReal *yc,PdsDsReal *ys)
{
	if(WAVE==NULL)	return FALSE;

	*yc        =WAVE->Yc[0];
	*ys        =WAVE->Ys[0];

	WAVE->Yc[1]=WAVE->Yc[0];
	WAVE->Ys[1]=WAVE->Ys[0];

	WAVE->Yc[0]=WAVE->a*WAVE->Yc[1]-WAVE->b*WAVE->Ys[0];
	WAVE->Ys[0]=WAVE->b*WAVE->Yc[1]+WAVE->a*WAVE->Ys[0];

	return TRUE;
}


/** \fn int pds_iqwave_get_last_value(PdsIQWave *WAVE,PdsDsReal *yc,PdsDsReal *ys)
 *  \brief Obtiene el último valor a la salida del generador IQ sinusoidal.
 *  \param[in,out] WAVE La estructura de PdsIQWave a consultar.
 *  \param[out] yc En esta variable se carga el  último valor a la salida de generador coseno.
 *  \param[out] ys En esta variable se carga el  último valor a la salida de generador seno.
 *  \return TRUE si todo fue bien o FALSE si no. Ejemplo WAVE==NULL.
 *  \ingroup PdsIQWaveGroup
 */
int pds_iqwave_get_last_value(PdsIQWave *WAVE,PdsDsReal *yc,PdsDsReal *ys)
{
	if(WAVE==NULL)	return FALSE;

	*yc        =WAVE->Yc[0];
	*ys        =WAVE->Ys[0];

	return TRUE;
}


/** \fn int pds_iqwave_get_vector(PdsIQWave *WAVE,PdsVector *VectorC,PdsVector *VectorS)
 *  \brief Carga un par de vectores con valores obtenidos en la salida del 
 *  generador IQ. Realiza una iteración para cada elemento "i" de los vectores.
 *  \param[in,out] WAVE La estructura de PdsIQWave a consultar.
 *  \param[out] VectorC En este vector se cargará los valores a la salida coseno de generador.
 *  \param[out] VectorS En este vector se cargará los valores a la salida seno de generador.
 *  \return TRUE si todo fue bien o FALSE si no. Ejemplo WAVE==NULL, Vector==NULL o
 *  los tamaños de los vectores no coinciden.
 *  \ingroup PdsIQWaveGroup
 */
int pds_iqwave_get_vector(PdsIQWave *WAVE,PdsVector *VectorC,PdsVector *VectorS)
{
	PdsRaNatural i;

	if(WAVE==NULL)		return FALSE;
	if(VectorC==NULL)	return FALSE;
	if(VectorS==NULL)	return FALSE;
	if(VectorS->Nel!=VectorC->Nel)	return FALSE;

	for(i=0;i<VectorC->Nel;i++)
	{
		VectorC->V[i]=WAVE->Yc[0];
		VectorS->V[i]=WAVE->Ys[0];

		WAVE->Yc[1]=WAVE->Yc[0];
		WAVE->Ys[1]=WAVE->Ys[0];

		WAVE->Yc[0]=WAVE->a*WAVE->Yc[1]-WAVE->b*WAVE->Ys[0];
		WAVE->Ys[0]=WAVE->b*WAVE->Yc[1]+WAVE->a*WAVE->Ys[0];
	}

	return TRUE;
}

/** \fn void pds_iqwave_free(PdsIQWave *WAVE)
 *  \brief Libera una estructura de tipo puntero PdsIQWave.
 *  \param[in,out] WAVE La estructura de PdsIQWave a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsIQWaveGroup
 */
void pds_iqwave_free(PdsIQWave *WAVE)
{
	if(WAVE!=NULL)
	{
		free(WAVE);
	}
}


/** \fn void pds_iqwave_destroy(PdsIQWave **WAVE)
 *  \brief Libera una estructura de tipo puntero PdsIQWave, y lo carga con NULL.
 *  \param[in,out] WAVE La estructura de PdsIQWave a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsIQWaveGroup
 */
void pds_iqwave_destroy(PdsIQWave **WAVE)
{
	if((*WAVE)!=NULL)
	{
		free(*WAVE);
		(*WAVE)=NULL;
	}
}


