/*
 * pdsgaussianquantizer.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */


#include <stdlib.h>
#include <math.h>
#include <pds/pdsmath.h>
#include <pds/pdsgaussianquantizer.h>

////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsGaussianQuantizer                                   ////
////////////////////////////////////////////////////////////////////////////////

/** \fn PdsGaussianQuantizer *pds_gaussian_quantizer_new(PdsDsReal Media, PdsDsReal Sigma, PdsDsNatural Bits)
 *  \brief Crea una estructura de tipo PdsGaussianQuantizer.
 * 
 *  Crea un cuantizador no uniforme, optimizado para trabajar con una señal de
 *  de entrada con distribución de probabilidad gaussiana con media Media y varianza
 *  Sigma^2. Los intervalos de decisión del cuantizador están repartidos de modo 
 *  que todos los índices sean equiprobables.
 *  Ejecutar esta orden puede tardar un poco a partir de 8 bits en adelante.
 *  \param[in] Media Es el valor medio esperado de la señal de entrada del cuantizador.
 *  \param[in] Sigma Es el desvío padrón de la señal de entrada del cuantizador.
 *  \param[in] Bits Es el número de bits del cuantizador.
 *  \return Un puntero a una estructura de tipo PdsGaussianQuantizer.
 *  \ingroup PdsGaussianQuantizerGroup
 */
PdsGaussianQuantizer *pds_gaussian_quantizer_new(PdsDsReal Media, PdsDsReal Sigma, PdsDsNatural Bits)
{
	PdsGaussianQuantizer *Q=NULL;
	PdsDsNatural i;
	
	if(Bits>(8*sizeof(PdsDsNatural)))	return NULL;

	Q=(PdsGaussianQuantizer *)calloc(1,sizeof(PdsGaussianQuantizer));
	if(Q==NULL)	return NULL;
	////////////////////////////////////////////////////////////////////////
	Q->Media=Media;
	Q->Sigma=Sigma;
	Q->Bits=Bits;
	Q->N=(PdsDsNatural)pow(2.0,Bits);
	Q->P=1.0/Q->N;
	////////////////////////////////////////////////////////////////////////
	Q->X=(PdsDsReal *)calloc(Q->N,sizeof(PdsDsReal));
	if(Q->X==NULL)
	{	
		free(Q);
		return NULL;
	}
	////////////////////////////////////////////////////////////////////////
	Q->Y=(PdsDsReal *)calloc(Q->N,sizeof(PdsDsReal));
	if(Q->Y==NULL)
	{	
		free(Q->X);
		free(Q);
		return NULL;
	}

	////////////////////////////////////////////////////////////////////////
	for(i=1;i<=(Q->N/2-1);i++)
	{
		Q->X[Q->N-1-i]=Q->Media+Q->Sigma*pds_qfuncinv(i*Q->P);
	}
	Q->X[Q->N-1-i]=Q->Media;
	for(i=(Q->N/2+1);i<=(Q->N-1);i++)
	{
		Q->X[Q->N-1-i]=2.0*Q->Media-Q->X[i-1];
	}
	////////////////////////////////////////////////////////////////////////


	Q->Y[0]=Q->Media+(Q->Sigma/(sqrt(M_PI*2.0)*Q->P))*(-exp(-((Q->X[0]-Q->Media)*(Q->X[0]-Q->Media))/(2.0*Q->Sigma*Q->Sigma)));
	for(i=0;i<Q->N-2;i++)	Q->Y[i+1]=Q->Media+(Q->Sigma/(sqrt(M_PI*2)*Q->P))*(exp(-((Q->X[i]-Q->Media)*(Q->X[i]-Q->Media))/(2.0*Q->Sigma*Q->Sigma))-exp(-(Q->X[i+1]-Q->Media)*(Q->X[i+1]-Q->Media)/(2.0*Q->Sigma*Q->Sigma)));
	Q->Y[Q->N-1]=Q->Media+(Q->Sigma/(sqrt(M_PI*2.0)*Q->P))*(exp(-((Q->X[Q->N-2]-Q->Media)*(Q->X[Q->N-2]-Q->Media))/(2.0*Q->Sigma*Q->Sigma)));

	return Q;
}

/** \fn PdsGaussianQuantizer *pds_gaussian_quantizer_new_quantizer(PdsGaussianQuantizer *Q0)
 *  \brief Crea una estructura de tipo PdsGaussianQuantizer como una copia de otra.
 * 
 *  Crea un cuantizador no uniforme, copiando los parámetros de otro cuantizador 
 *  uniforme. Esta función es rápida comparado a hacer pds_gaussian_quantizer_new.
 *  \param[in] Q0 Cuantizador al que se le copiaran los parámetros.
 *  \return Un puntero a una estructura de tipo PdsGaussianQuantizer.
 *  \ingroup PdsGaussianQuantizerGroup
 */
PdsGaussianQuantizer *pds_gaussian_quantizer_new_quantizer(PdsGaussianQuantizer *Q0)
{
	PdsGaussianQuantizer *Q=NULL;
	PdsDsNatural i;

	if(Q0==NULL)	return NULL;

	if(Q0->Bits>(8*sizeof(PdsDsNatural)))	return NULL;

	Q=(PdsGaussianQuantizer *)calloc(1,sizeof(PdsGaussianQuantizer));
	if(Q==NULL)	return NULL;
	////////////////////////////////////////////////////////////////////////
	Q->Media=Q0->Media;
	Q->Sigma=Q0->Sigma;
	Q->Bits=Q0->Bits;
	Q->N=Q0->N;
	Q->P=Q0->P;
	////////////////////////////////////////////////////////////////////////
	Q->X=(PdsDsReal *)calloc(Q->N,sizeof(PdsDsReal));
	if(Q->X==NULL)
	{	
		free(Q);
		return NULL;
	}

	////////////////////////////////////////////////////////////////////////
	Q->Y=(PdsDsReal *)calloc(Q->N,sizeof(PdsDsReal));
	if(Q->Y==NULL)
	{	
		free(Q->X);
		free(Q);
		return NULL;
	}

	////////////////////////////////////////////////////////////////////////
	for(i=1;i<Q->N;i++)	Q->X[i]=Q0->X[i];
	for(i=1;i<Q->N;i++)	Q->Y[i]=Q0->Y[i];

	return Q;
}


/** \fn int pds_gaussian_quantizer_get_id(const PdsGaussianQuantizer *Q, PdsDsReal Valor, PdsDsNatural *Id)
 *  \brief Devuelve el índice correspondiente al nivel de cuantización del Valor.
 * 
 *  Esta función devuelve un índice entre 0 y Q->N-1.
 *  \param[in] Q Cuantizador no uniforme para una distribución gaussiana.
 *  \param[in] Valor Valor de entrada del cuantizador.
 *  \param[out] Id Índice correspondiente al Valor de entrada del cuantizador.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsGaussianQuantizerGroup
 */
int pds_gaussian_quantizer_get_id(const PdsGaussianQuantizer *Q, PdsDsReal Valor, PdsDsNatural *Id)
{
	PdsDsNatural i;

	if(Q==NULL)	return FALSE;

	if(Valor<=Q->X[0])	
	{
		*Id=0;
	}
	else if(Valor>Q->X[Q->N-2])
	{
		*Id=Q->N-1;
	}
	else
	{
		for(i=0;i<Q->N-2;i++)
		{
			if(( Valor>Q->X[i] )&&( Valor<=Q->X[i+1] ))
			{
				*Id=i+1;
			}
		}
	}
	return TRUE;
}

/** \fn int pds_gaussian_quantizer_get_value(const PdsGaussianQuantizer *Q, PdsDsNatural Id, PdsDsReal *Valor)
 *  \brief Devuelve un valor representativo correspondiente al nivel de cuantificación  del Id.
 * 
 *  Esta función devuelve un valor representativo al índice Id.
 *  \param[in] Q Cuantizador no uniforme para una distribución gaussiana.
 *  \param[in] Id Índice correspondiente a consultar.
 *  \param[out] Valor Valor correspondiente en la entrada del cuantizador.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsGaussianQuantizerGroup
 */
int pds_gaussian_quantizer_get_value(const PdsGaussianQuantizer *Q, PdsDsNatural Id, PdsDsReal *Valor)
{
	if(Q==NULL)	return FALSE;
	if(Id>=Q->N)
	{
		*Valor=Q->Y[Q->N-1];
		return FALSE;
	}
	else
	{
		*Valor=Q->Y[Id];
		return TRUE;
	}
}

/** \fn void pds_gaussian_quantizer_free(PdsGaussianQuantizer *Q)
 *  \brief Libera la memoria correspondiente al cuantizador Q.
 *
 *  \param[in] Q Cuantizador no uniforme a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsGaussianQuantizerGroup
 */
void pds_gaussian_quantizer_free(PdsGaussianQuantizer *Q)
{
	if(Q!=NULL)
	{
		free(Q->X);
		free(Q->Y);
		free(Q);
	}
}

