/*
 * pdsdic2d.c
 * 
 * Copyright 2018 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <pds/pdsdic2d.h>
#include <pds/pdsra.h>
#include <math.h>


/** \fn PdsDic2D *pds_dic2d_new(PdsRaNatural Nlin,PdsRaNatural Ncol)
 *  \brief Crea una estructura de tipo PdsDic2D, internamente la estructura 
 *  contiene dos matrices con Nlin lineas y Ncol columnas, con valores iniciados 
 *  con ceros.
 *  \param[in] Nlin Es el número de lineas de las matrices.
 *  \param[in] Ncol Es el número de columnas de las matrices.
 *  \return Un puntero a la estructura de tipo PdsDic2D o NULL en caso de error.
 *  \ingroup PdsDic2DGroup
 */
PdsDic2D *pds_dic2d_new(PdsRaNatural Nlin,PdsRaNatural Ncol)
{
    PdsDic2D *DIC=NULL;

    DIC=(PdsDic2D *)calloc(1,sizeof(PdsDic2D));
    if(DIC==NULL)    return NULL;

    DIC->M0=pds_matrix_new(Nlin,Ncol);
    if(DIC->M0==NULL)
    {
        free(DIC);
        return NULL;
    }

    DIC->M1=pds_matrix_new(Nlin,Ncol);
    if(DIC->M1==NULL)
    {
        free(DIC->M0);
        free(DIC);
        return NULL;
    }

    DIC->zero_std_search=PDS_WRONG;
    DIC->match_threshold=PDS_DIC2D_MATCH_THRESHOLD;
    DIC->no_moved_threshold=PDS_DIC2D_NO_MOVED_THRESHOLD;
    DIC->search_step_size=PDS_DIC2D_SEARCH_PIXEL_BY_PIXEL;

    DIC->search_max_length=(int)(PDS_MIN(DIC->M0->Nlin,DIC->M0->Ncol)/4);
    DIC->debug=1;
    DIC->last_match_corr=0;

    return DIC;
}

/** \fn PdsDic2D *pds_dic2d_new_from_matrices(const PdsMatrix *Mat0,const PdsMatrix *Mat1)
 *  \brief Crea una estructura de tipo PdsDic2D, internamente la estructura 
 *  contiene una copia de las dos matrices de entrada.
 *  Los tamanho de las matrices deben ser iguales.
 *  \param[in] Mat0 Primera matriz, matriz de fuente.
 *  \param[in] Mat1 Segunda matriz, matriz de búsqueda.
 *  \return Un puntero a la estructura de tipo PdsDic2D o NULL en caso de error.
 *  \ingroup PdsDic2DGroup
 */
PdsDic2D *pds_dic2d_new_from_matrices(const PdsMatrix *Mat0,const PdsMatrix *Mat1)
{
    PdsDic2D *DIC=NULL;
    
    if(Mat0==NULL)  return NULL;
    if(Mat1==NULL)  return NULL;

    

    DIC=(PdsDic2D *)calloc(1,sizeof(PdsDic2D));
    if(DIC==NULL)    return NULL;

    DIC->M0=pds_matrix_new_matrix(Mat0);
    if(DIC->M0==NULL)
    {
        free(DIC);
        return NULL;
    }

    DIC->M1=pds_matrix_new_matrix(Mat1);
    if(DIC->M1==NULL)
    {
        free(DIC->M0);
        free(DIC);
        return NULL;
    }

    DIC->zero_std_search=PDS_WRONG;
    DIC->match_threshold=PDS_DIC2D_MATCH_THRESHOLD;
    DIC->no_moved_threshold=PDS_DIC2D_NO_MOVED_THRESHOLD;
    DIC->search_step_size=PDS_DIC2D_SEARCH_PIXEL_BY_PIXEL;

    DIC->search_max_length=(int)(PDS_MIN(DIC->M0->Nlin,DIC->M0->Ncol)/4);
    DIC->debug=1;
    DIC->last_match_corr=0;

    return DIC;
}

/** \fn int pds_dic2d_push_matrix(PdsDic2D *DIC,const PdsMatrix *Mat)
 *  \brief Actualiza los valores de las matrices en la estructura de tipo PdsDic2D, 
 *  internamente la matriz M1->M0 y mat->M1.
 *  Los tamanho de las matrices deben ser iguales.
 *  \param[in] DIC La estructura en consulta.
 *  \param[in] Mat Matriz a copiar.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ejemplo: DIC==NULL MAT==NULL). 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_push_matrix(PdsDic2D *DIC,const PdsMatrix *Mat)
{    
    if(DIC==NULL)  return PDS_WRONG;
    if(Mat==NULL)  return PDS_WRONG;


    if(Mat->Nlin!=DIC->M1->Nlin)    return PDS_WRONG;
    if(Mat->Ncol!=DIC->M1->Ncol)    return PDS_WRONG;

    PdsMatrix *tmp=pds_matrix_new_matrix(Mat);
    if(tmp==NULL)  return PDS_WRONG;

    pds_matrix_free(DIC->M0);

    DIC->M0=DIC->M1;
    DIC->M1=tmp;

    return PDS_OK;
}

/** \fn int pds_dic2d_get_nlines(const PdsDic2D *DIC, PdsRaNatural *Nlines)
 *  \brief Devuelve el número de lineas en las matrices dentro de la estructura DIC.
 *  \param[in] DIC La estructura en consulta.
 *  \param[out] Nlines En donde se guardará el número de lineas de las matrices.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: DIC==NULL). 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_get_nlines(const PdsDic2D *DIC, PdsRaNatural *Nlines)
{
    if(DIC   ==NULL)  return PDS_WRONG;
    if(Nlines==NULL)  return PDS_WRONG;

    if(DIC->M1==NULL)   return PDS_WRONG;
    if(DIC->M0==NULL)   return PDS_WRONG;

    if(DIC->M0->Nlin!=DIC->M1->Nlin)    return PDS_WRONG;
    else                                *Nlines=DIC->M0->Nlin;

    return PDS_OK;
}

/** \fn int pds_dic2d_get_ncolumns(const PdsDic2D *DIC, PdsRaNatural *Ncolumns)
 *  \brief Devuelve el número de columnas en las matrices dentro de la estructura DIC.
 *  \param[in] DIC La estructura en consulta.
 *  \param[out] Ncolumns En donde se guardará el número de columnas de las matrices.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: DIC==NULL). 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_get_ncolumns(const PdsDic2D *DIC, PdsRaNatural *Ncolumns)
{
    if(DIC     ==NULL)  return PDS_WRONG;
    if(Ncolumns==NULL)  return PDS_WRONG;

    if(DIC->M1==NULL)   return PDS_WRONG;
    if(DIC->M0==NULL)   return PDS_WRONG;

    if(DIC->M0->Ncol!=DIC->M1->Ncol)    return PDS_WRONG;
    else                                *Ncolumns=DIC->M0->Ncol;

    return PDS_OK;
}


/** \fn int pds_dic2d_get_no_moved_threshold(const PdsDic2D *DIC, PdsRaReal *no_moved_threshold)
 *  \brief Devuelve el umbral para aceptar que el objeto no se ha movido.
 *  \param[in] DIC La estructura en consulta.
 *  \param[out] no_moved_threshold Umbral de coincidencia.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: DIC==NULL). 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_get_no_moved_threshold(const PdsDic2D *DIC, PdsRaReal *no_moved_threshold)
{
    if(DIC               ==NULL)  return PDS_WRONG;
    if(no_moved_threshold==NULL)  return PDS_WRONG;

    *no_moved_threshold=DIC->no_moved_threshold;

    return PDS_OK;
}

/** \fn int pds_dic2d_get_match_threshold(const PdsDic2D *DIC, PdsRaReal *match_threshold)
 *  \brief Devuelve el umbral para aceptar una coincidencia en un match.
 *  \param[in] DIC La estructura en consulta.
 *  \param[out] match_threshold Umbral de coincidencia.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: DIC==NULL). 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_get_match_threshold(const PdsDic2D *DIC, PdsRaReal *match_threshold)
{
    if(DIC            ==NULL)  return PDS_WRONG;
    if(match_threshold==NULL)  return PDS_WRONG;

    *match_threshold=DIC->match_threshold;

    return PDS_OK;
}

/** \fn int pds_dic2d_get_search_step_size(const PdsDic2D *DIC, PdsRaNatural *search_step_size)
 *  \brief Devuelve el paso en pixels, para la búsqueda de regiones coincidentes.
 *  Por defecto es PDS_DIC2D_SEARCH_PIXEL_BY_PIXEL.
 *  \param[in] DIC La estructura en consulta.
 *  \param[out] search_step_size Paso en pixels, para la búsqueda de regiones coincidentes.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: DIC==NULL). 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_get_search_step_size(const PdsDic2D *DIC, PdsRaNatural *search_step_size)
{
    if(DIC              ==NULL)  return PDS_WRONG;
    if(search_step_size==NULL)  return PDS_WRONG;

    *search_step_size=DIC->search_step_size;

    return PDS_OK;
}

/** \fn int pds_dic2d_get_search_max_length(const PdsDic2D *DIC, PdsRaNatural *search_max_length)
 *  \brief Devuelve la distancia máxima en pixels, para la búsqueda de regiones coincidentes.
 *  \param[in] DIC La estructura en consulta.
 *  \param[out] search_max_length La distancia maxima en pixels, para la búsqueda de regiones coincidentes.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: DIC==NULL). 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_get_search_max_length(const PdsDic2D *DIC, PdsRaNatural *search_max_length)
{
    if(DIC              ==NULL)  return PDS_WRONG;
    if(search_max_length==NULL)  return PDS_WRONG;

    *search_max_length=DIC->search_max_length;

    return PDS_OK;
}


/** \fn int pds_dic2d_get_zero_std_search(const PdsDic2D *DIC, int *zero_std_search)
 *  \brief Devuelve si esta habilitado que una región con desvío padrón cero sea buscada.
 *  Por defecto está variable está deshabilitada con PDS_WRONG, se habilita con
 *  PDS_OK.
 *  \param[in] DIC La estructura en consulta. 
 *  \param[out] zero_std_search Si se busca regiones con desvío padrón cero.
 *  Este valor es PDS_OK o PDS_WRONG.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: DIC==NULL o zero_std_search==NULL). 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_get_zero_std_search(const PdsDic2D *DIC, int *zero_std_search)
{
    if(DIC            ==NULL)  return PDS_WRONG;
    if(zero_std_search==NULL)  return PDS_WRONG;

    *zero_std_search=DIC->zero_std_search;

    return PDS_OK;
}

/** \fn int pds_dic2d_set_zero_std_search(PdsDic2D *DIC, int zero_std_search)
 *  \brief Establece si una región con desvío padrón cero serán buscadas.
 *  Por defecto está variable está deshabilitada con PDS_WRONG, se habilita con
 *  PDS_OK.
 *  \param[in] DIC La estructura en consulta. 
 *  \param[in] zero_std_search Habilitador, des-habilitador, si se busca regiones
 *  con desvío padrón cero.
 *  Este valor no es aceptado si es diferente de PDS_OK o PDS_WRONG.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: DIC==NULL o parámetro fuera de rango). 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_set_zero_std_search(PdsDic2D *DIC, int zero_std_search)
{
    if(DIC==NULL)  return PDS_WRONG;

    if     (zero_std_search==PDS_OK)    DIC->zero_std_search=zero_std_search;
    else if(zero_std_search==PDS_WRONG) DIC->zero_std_search=zero_std_search;
    else                                return PDS_WRONG;

    return PDS_OK;
}

/** \fn int pds_dic2d_set_no_moved_threshold(PdsDic2D *DIC, PdsRaReal no_moved_threshold)
 *  \brief Establece el umbral para aceptar que la región no se ha movido.
 *  Por defecto este umbral es PDS_DIC2D_NO_MOVED_THRESHOLD.
 *  \param[in] DIC La estructura en consulta. 
 *  \param[in] no_moved_threshold Umbral para aceptar que la región no se ha movido.
 *  Este valor no es aceptado si es mayor que 1.0 o menor de 0.0.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: DIC==NULL o parámetro fuera de rango). 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_set_no_moved_threshold(PdsDic2D *DIC, PdsRaReal no_moved_threshold)
{
    if(DIC==NULL)  return PDS_WRONG;

    if(no_moved_threshold > +1.0)    return  PDS_WRONG;
    if(no_moved_threshold <  0.0)    return  PDS_WRONG;

    DIC->no_moved_threshold=no_moved_threshold;

    return PDS_OK;
}

/** \fn int pds_dic2d_set_match_threshold(PdsDic2D *DIC, PdsRaReal match_threshold)
 *  \brief Establece el umbral para aceptar una coincidencia en un match.
 *  Por defecto este umbral es PDS_DIC2D_MATCH_THRESHOLD.
 *  \param[in] DIC La estructura en consulta.
 *  \param[in] match_threshold Umbral de coincidencia.
 *  Este valor no puede ser mayor que 1.0 o menor de -1.0.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: DIC==NULL). 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_set_match_threshold(PdsDic2D *DIC, PdsRaReal match_threshold)
{
    if(DIC==NULL)  return PDS_WRONG;

    if(match_threshold > +1.0)    return  PDS_WRONG;
    if(match_threshold < -1.0)    return  PDS_WRONG;

    DIC->match_threshold=match_threshold;

    return PDS_OK;
}

/** \fn int pds_dic2d_set_search_step_size(PdsDic2D *DIC, PdsRaNatural search_step_size)
 *  \brief Establece el paso en pixels, para la búsqueda de regiones coincidentes.
 *  Por defecto es PDS_DIC2D_SEARCH_PIXEL_BY_PIXEL.
 *  \param[in] DIC La estructura en consulta.
 *  \param[in] search_step_size Paso en pixels, para la búsqueda de regiones coincidentes.
 *  El valor cero no es aceptado.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: DIC==NULL). 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_set_search_step_size(PdsDic2D *DIC, PdsRaNatural search_step_size)
{
    if(DIC==NULL)  return PDS_WRONG;

    if(search_step_size==0)    return  PDS_WRONG;

    DIC->search_step_size=search_step_size;

    return PDS_OK;
}

/** \fn int pds_dic2d_set_search_max_length(PdsDic2D *DIC, PdsRaNatural search_max_length)
 *  \brief Establece la distancia máxima en pixels, para la búsqueda de regiones coincidentes.
 *  \param[in] DIC La estructura en consulta.
 *  \param[in] search_max_length La distancia máxima en pixels, para la búsqueda de regiones coincidentes.
 *  El valor cero no es aceptado.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: DIC==NULL). 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_set_search_max_length(PdsDic2D *DIC, PdsRaNatural search_max_length)
{
    if(DIC==NULL)  return PDS_WRONG;

    if(search_max_length==0)    return  PDS_WRONG;

    DIC->search_max_length=search_max_length;

    return PDS_OK;
}


/** \fn int pds_dic2d_set_debug(PdsDic2D *DIC, unsigned char debug)
 *  \brief Establece se  debug esta habilitado, se habilitado se mostrara mas
 *  infrmacion por pantalla.
 *  \param[in] DIC La estructura en consulta.
 *  \param[in] debug Habilita con valor 1, deshabilita con valor diferente de 1.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: DIC==NULL). 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_set_debug(PdsDic2D *DIC, unsigned char debug)
{
    if(DIC==NULL)  return PDS_WRONG;

    DIC->debug=debug;

    return PDS_OK;
}

/** \fn int pds_dic2d_set_last_match_corr(PdsDic2D *DIC, PdsRaReal last_match_corr)
 *  \brief Establece el valor de la correlación en el ultimo match.
 *  \param[in] DIC La estructura en consulta.
 *  \param[in] last_match_corr Valor de la correlación en el ultimo match.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no (ej: DIC==NULL). 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_set_last_match_corr(PdsDic2D *DIC, PdsRaReal last_match_corr)
{
    if(DIC==NULL)  return PDS_WRONG;

    DIC->last_match_corr=last_match_corr;

    return PDS_OK;
}

/** \fn void pds_dic2d_free(PdsDic2D *DIC)
 *  \brief Libera la estructura de tipo puntero PdsDic2D.
 *  \param[in,out] DIC La estructura a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsDic2DGroup
 */
void pds_dic2d_free(PdsDic2D *DIC)
{
    if(DIC!=NULL)
    {
        pds_matrix_free(DIC->M0);
        pds_matrix_free(DIC->M1);
        free(DIC);
    }
}

/** \fn void pds_dic2d_destroy(PdsDic2D **DIC)
 *  \brief Libera la estructura de tipo puntero PdsDic2D, y limpia el puntero con NULL.
 *  \param[in,out] DIC La estructura a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsDic2DGroup
 */
void pds_dic2d_destroy(PdsDic2D **DIC)
{
    if(DIC!=NULL)
    if((*DIC)!=NULL)
    {
        pds_matrix_free((*DIC)->M0);
        pds_matrix_free((*DIC)->M1);
        free((*DIC));
        (*DIC)=NULL;
    }
}


////////////////////////////////////////////////////////////////////////////////

/** \fn int pds_dic2d_tracking_region( PdsDic2D *DIC, PdsRegionRect Rin, PdsRegionRect *Rout)
 *  \brief Devuelve la región coincidente en la segunda matriz.
 *  \param[in] DIC La estructura en consulta.
 *  \param[in] Rin La región de la primera matriz que se buscara en la segunda.
 *  \param[out] Rout La región encontrada en la segunda matriz.
 *  Esta variable no es alterada si el objeto no fue encontrado.
 *  \return Puede retornar:
 *  PDS_DIC2D_ERROR cuando hubo un error lectura de memoria.
 *  PDS_DIC2D_FOUND Si el objeto fue encontrado.
 *  PDS_DIC2D_NOFOUND Si el objeto no fue encontrado.
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_tracking_region( PdsDic2D *DIC, PdsRegionRect Rin, PdsRegionRect *Rout)
{
    if( (DIC ==NULL)||(Rout==NULL) )
    {
        if(DIC->debug==1)   fprintf(stderr,"ERROR: Line %d of file \"%s\".\n", __LINE__, __FILE__);
        if(DIC->debug==1)   fprintf(stderr,"ERROR: Checking input parameters.\n");
        return PDS_DIC2D_ERROR;
    }

    if((Rin.Nlin==0)&&(Rin.Ncol==0))  return PDS_DIC2D_NOFOUND;

    if(DIC->zero_std_search==PDS_WRONG)
    {   
        PdsRaReal std;
        pds_matrix_std_of_region(DIC->M0,Rin,&std);
        if(std==0)  return PDS_DIC2D_NOFOUND;
    }

    PdsRaNatural s0 = DIC->search_step_size;
    PdsRaNatural L  = DIC->search_max_length;
    PdsRaNatural N  = L/s0;

    // Verifico si la region se ha movido
    if(pds_dic2d_is_the_region_moved(DIC,Rin)!=PDS_OK)
    {
        (*Rout)=Rin;
        return PDS_DIC2D_FOUND;
    }

    // Si se ha movido busco a donde
    PdsMatrix *PCC=pds_dic2d_pearson_correlations_matrix(DIC,Rin);
    if(PCC==NULL)   
    {
        if(DIC->debug==1)   fprintf(stderr,"ERROR: Line %d of file \"%s\".\n", __LINE__, __FILE__);
        if(DIC->debug==1)   fprintf(stderr,"ERROR: Checking memory allocation.\n");
        return PDS_DIC2D_ERROR;
    }

    PdsRaReal max_corr;
    PdsRaNatural lin,col;
    pds_matrix_get_max_value_and_pos(PCC,&max_corr,&lin,&col);

    // Si encuentro la región devuelvo esta y retorno que la he encontrado.
    if(max_corr >= DIC->match_threshold)
    {
		Rout->L0=(PdsRaNatural)(Rin.L0+((int)(lin)-(int)(N))*s0);
		Rout->C0=(PdsRaNatural)(Rin.C0+((int)(col)-(int)(N))*s0);
        Rout->Nlin=Rin.Nlin;
        Rout->Ncol=Rin.Ncol;

        pds_dic2d_set_last_match_corr(DIC,max_corr);

        return PDS_DIC2D_FOUND;
    }
    else
    {
        if(DIC->debug==1)   fprintf(stdout,"Ponto no encontrado. Maxima correlation:%f \n",max_corr);
        return PDS_DIC2D_NOFOUND;
    }

    return PDS_DIC2D_FOUND;
}

/** \fn int pds_dic2d_is_the_region_moved(PdsDic2D *DIC, PdsRegionRect Ro)
 *  \brief Indica si la region se ha movido. 
 *
 *  Es considerado que hubo movimiento si la correlación de la regiones en M0 y M1
 *  es menor que no_moved_threshold.
 *  \param[in] DIC La estructura en consulta.
 *  \param[in] Ro La region de la matriz M0.
 *  \return PDS_OK si se movimiento o PDS_WRONG si no se movimiento o si hubo algún
 *  error en la consulta. 
 *  \ingroup PdsDic2DGroup
 */
int pds_dic2d_is_the_region_moved(PdsDic2D *DIC, PdsRegionRect Ro)
{
    PdsRaReal corr;
    if(DIC ==NULL)  return PDS_WRONG;

    if((Ro.Nlin==0)&&(Ro.Ncol==0))  return PDS_WRONG;

    pds_matrix_corr_matrix_of_region(DIC->M0,DIC->M1,Ro,Ro,&corr);
    
    if(corr < (DIC->no_moved_threshold))    return PDS_OK;
    else
    {
        pds_dic2d_set_last_match_corr(DIC,corr);
        return PDS_WRONG;
    }
}


/** \fn PdsMatrix *pds_dic2d_pearson_correlations_matrix(const PdsDic2D *DIC, PdsRegionRect Ro)
 *  \brief Retorna una matriz con las correlaciones de la región Ro en la matriz M0 de DIC,
 *  con una vecindad rectangular con brazo de acción search_max_length y un paso 
 *  de búsqueda de search_step_size, en la matriz M1.
 *
 *  Así, la función retorna una matriz cuadrada de 1+2(search_max_length/search_step_size)
 *  elementos por lado.
 *  \param[in] DIC La estructura en consulta.
 *  \param[in] Ro La región de la matriz M0.
 *  \return A matriz de correlaciones si todo fue bien o NULL si no (ej: DIC==NULL). 
 *  \ingroup PdsDic2DGroup
 */
PdsMatrix *pds_dic2d_pearson_correlations_matrix(const PdsDic2D *DIC, PdsRegionRect Ro)
{
    PdsRaReal corr;
    PdsMatrix *PCC=NULL;

    if(DIC ==NULL)  return NULL;

    if((Ro.Nlin==0)&&(Ro.Ncol==0))  return NULL;

    PdsRaNatural s0 = DIC->search_step_size;
    PdsRaNatural L  = DIC->search_max_length;
    PdsRaNatural N  = L/s0;
    int lin,col;
    int ID;



    PCC=pds_matrix_new(2*N+1,2*N+1);
    if(PCC==NULL)   return NULL;

    PdsRegionRect R=pds_region_rect(0,0,Ro.Nlin,Ro.Ncol);

    PdsRegionRect Rm1;
    ID=pds_matrix_get_region_rect(DIC->M1,&Rm1);


    for( lin=-((int)N);lin<=((int)N);lin=lin+1) 
    for( col=-((int)N);col<=((int)N);col=col+1)
	{

		R.L0=(PdsRaInteger)(Ro.L0+lin*s0);
		R.C0=(PdsRaInteger)(Ro.C0+col*s0);
            
        if(pds_region_rect_is_inside(R,Rm1)==PDS_OK)
        {   
            corr=0;
            ID=pds_matrix_corr_matrix_of_region(DIC->M0,DIC->M1,Ro,R,&corr);
            if(ID==PDS_OK)
            {  
                pds_matrix_set_value(PCC,lin+N,col+N,corr);
            }
            else
            {
                pds_matrix_free(PCC);
                return NULL;
            }
        }

    }
    
    return PCC;
}

