/*
 * testprog2.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */


/** \example testprog2.c
 *  \author Fernando Pujaico Rivera
 *  \date 21-05-2011
 *  \brief Test del Filtro Fir con cero en pi/4 para dos señales sinusoidales 
 *  en pi/32 y pi/4. 
 *
 *  <br>
 *  Ademas la señal de entrada tiene una constante igual 1. La frecuencia 
 *  de muestro es 2*pi.
 *  <br>
 *  Test de un filtro FIR.
 *  \f[x[n]=1+sin(\frac{\pi}{32}n)+sin(\frac{\pi}{4}n+\frac{\pi}{16})\f]
 *  \f[H(z)=\frac{1}{2-2cos(\frac{\pi}{4})}(1-2cos(\frac{\pi}{4})z^{-1}+z^{-2}) \f]
 *  <br>
 *  \image html test2grafico1.png "Señal de entrada x[n]."
 *  \image html test2grafico2.png "Señal de salida y[n]."
 */

#include <config.h>
#include <stdlib.h>
#include <pds/pdsdf.h>
#include <pds/pdsvector.h>
#include "extras.h"
#include <math.h>

int main(int argc, char** argv)
{
	///////////////////////////////////////////////////////////////
	PdsFir *FIR1=NULL;
	PdsVector *h=NULL;
	PdsVector *X=NULL;
	PdsVector *Y1=NULL;
	PdsVector *Y2=NULL;
	PdsDfReal x,y;
	PdsDfNatural n,N;
	FILE *fd=NULL;
	char DATAFILE[]="datos.txt";
	char OCTAVEFILE[]="plotinoctave.m";

	N=128;

	h=pds_vector_new(3);
	X=pds_vector_new(N);
	Y1=pds_vector_new(N);
	Y2=pds_vector_new(N);
	
	/* h=[   1.70710678118655  -2.41421356237310   1.70710678118655] */
	pds_vector_set_value(h,0,1.70710678118655);	
	pds_vector_set_value(h,1,-2.41421356237310);
	pds_vector_set_value(h,2,1.70710678118655);


	///////////////////////////////////////////////////////////////////////
	FIR1=pds_fir_new(h);
	for(n=0;n<N;n++)
	{
		x=1.0+sin(M_PI*n/32.0)+sin(M_PI*n/4.0+M_PI/16.0);
		pds_fir_evaluate_value(FIR1,x,&y);
		
		X->V[n]=x;
		Y1->V[n]=y;
	}
	pds_fir_free(FIR1);
	///////////////////////////////////////////////////////////////////////

	///////////////////////////////////////////////////////////////////////
	FIR1=pds_fir_new(h);
	pds_fir_evaluate_vector(FIR1,X,Y2);
	pds_fir_free(FIR1);
	///////////////////////////////////////////////////////////////////////


	fd=fopen(DATAFILE,"w");
	if(fd==NULL)	return EXIT_FAILURE;
	pds_vector_fprintf(X,fd);
	pds_vector_fprintf(Y1,fd);
	pds_vector_fprintf(Y2,fd);
	fclose(fd);

	pds_octave_plot_lin(OCTAVEFILE,DATAFILE);

	pds_vector_free(Y2);	
	pds_vector_free(Y1);	
	pds_vector_free(X);	
	pds_vector_free(h);	


    FIR1=pds_fir_get_new_bandstop(40,0.3,0.7,NULL);
    if(FIR1!=NULL)
    { 
        printf("H:\n");
        pds_vector_printf(pds_fir_get_h_vector(FIR1));
    }

    PdsVector* Hw=pds_vector_new(200);
	pds_vector_fir_frequency_response(FIR1->h,Hw);
	pds_octplot_vector_in_png(Hw,"k","|H(k)|^2","testprog2a.m","testprog2a.png");

	return EXIT_SUCCESS;
}
