/*
 * pdsiir1hp.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsiir1hp.h
 *  \author Fernando Pujaico Rivera
 *  \date 21-05-2011
 *  \brief Funciones que trabajan con estructuras Filtro Iir pasa alto de primer orden.
 *  
 *  <br>
 *  \image html iir1hp-filter.png "Filtro IIR First-Order"
 *  \f[ b=\frac{a+1}{2} \f].
 *  \f[ y[n]=a~y[n-i] + b~x[n] - b~x[n-1] \f].
 *  \f[ \frac{Y[z]}{X[z]}=H[z]=\frac{b(1-Z^{-1})}{1-a~Z^{-1}} \f].
 */

#ifndef __PDSIIR1HP_H__
#define __PDSIIR1HP_H__


#ifdef __cplusplus
extern "C" {
#endif 

#include <pds/pdsdfglobal.h>
#include <pds/pdsvector.h>

#include <stdio.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

/** \defgroup PdsIir1HPGroup Funciones del módulo PdsIir1HP.
 *
 *  <br>
 *  \image html iir1hp-filter.png "Filtro IIR First-Order"
 *  \f[ b=\frac{a+1}{2} \f].
 *  \f[ y[n]=a~y[n-i] + b~x[n] - b~x[n-1] \f].
 *  \f[ \frac{Y[z]}{X[z]}=H[z]=\frac{b(1-Z^{-1})}{1-a~Z^{-1}} \f].
 * @{
 */


/*! \struct PdsIir1HP
 *  \brief Una estructura tipo  PdsIir1HP .
 *
 *  Esta estructura genera un filtro IIR pasa alto de primer orden.
 *  Para usar incluir pds/pdsdf.h.
 *  <br>
 *  \image html iir1hp-filter.png "Filtro IIR First-Order"
 *  \f[ b=\frac{a+1}{2} \f].
 *  \f[ y[n]=a~y[n-i] + b~x[n] - b~x[n-1] \f].
 *  \f[ \frac{Y[z]}{X[z]}=H[z]=\frac{b(1-Z^{-1})}{1-a~Z^{-1}} \f].
 *  \ingroup PdsIir1HPGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! Factor del numerador. */
	PdsDfReal b;
	/*! Posicion del polo del filtro. */
	PdsDfReal a;
	/*! Es el antiguo valor de la entrada x. */
	PdsDfReal xold;
	/*! Es el antiguo valor de la salida y. */
	PdsDfReal yold;
}PdsIir1HP;

/** \fn PdsIir1HP *pds_iir1hp_new(PdsDfReal a)
 *  \brief Crea un filtro IIR pasa alto de primer orden.
 * 
 *  \f[ b=\frac{a+1}{2} \f].
 *  \f[ y[n]=a~y[n-i] + b~x[n] - b~x[n-1] \f].
 *  \f[ \frac{Y[z]}{X[z]}=H[z]=\frac{b(1-Z^{-1})}{1-a~Z^{-1}} \f].
 *  El parámetro a nunca debe ser 1.0 ni muy próximo.
 *  \param[in] a Polo del filtro IIR pasa alto primer orden.
 *  \return Un puntero a una estructura de tipo PdsIir1HP. O NULL en caso de error.
 *  Si a es mayor o igual a 1.0 o menor que 0.0 la función da error y retorna NULL.
 *  \ingroup PdsIir1HPGroup
 */
PdsIir1HP *pds_iir1hp_new(PdsDfReal a);


/** \fn int pds_iir1hp_evaluate_value(PdsIir1HP *IIR1HP,PdsDfReal x,PdsDfReal *y)
 *  \brief Evalúa el filtro IIR pasa alto con el valor de entrada x, el resultado
 *  es cargado en y.
 * 
 *  \param[in,out] IIR1HP El filtro IIR pasa alto a usar.
 *  \param[in] x El valor de entrada del filtro.
 *  \param[out] y El valor de salida del filtro.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsIir1HPGroup
 */
int pds_iir1hp_evaluate_value(PdsIir1HP *IIR1HP,PdsDfReal x,PdsDfReal *y);


/** \fn int pds_iir1hp_evaluate_vector(PdsIir1HP *IIR1HP,const PdsVector *x,PdsVector *y)
 *  \brief Evalua el filtro IIR pasa alto con el vector de entrada x, el resultado
 *  es cargado en el vector y.
 * 
 *  \param[in,out] IIR1HP El filtro IIR pasa alto a usar.
 *  \param[in] x El vector de entrada del filtro.
 *  \param[out] y El vector de salida del filtro.
 *  \return TRUE si todo fue bien o FALSE si no. Ejemplo x o y son NULL ó de 
 *  distinto tamaño.
 *  \ingroup PdsIir1HPGroup
 */
int pds_iir1hp_evaluate_vector(PdsIir1HP *IIR1HP,const PdsVector *x,PdsVector *y);


/** \fn void pds_iir1hp_free(PdsIir1HP *IIR1HP)
 *  \brief Libera el filtro pasa alto de tipo PdsIir1HP.
 *  \param[in] IIR1HP El filtro a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsIir1HPGroup
 */
void pds_iir1hp_free(PdsIir1HP *IIR1HP);


/** \fn void pds_iir1hp_destroy(PdsIir1HP **IIR1HP)
 *  \brief Libera el filtro pasa alto de tipo PdsIir1HP. y carga la variable con NULL.
 *  \param[in] IIR1HP El filtro a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsIir1HPGroup
 */
void pds_iir1hp_destroy(PdsIir1HP **IIR1HP);


/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif	/* __PDSIIR1HP_H__ */ 

