/*
 * example1a.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */


/** \example example1a.c
 *  \author Fernando Pujaico Rivera
 *  \date 21-05-2011
 *  \brief Ejemplo de uso del Filtro Iir con cero en -1 y polo en 0.9, para una 
 *  señal sinusoidal \f$x[n]\f$ con frecuencia (angular) en pi/32, 3*pi/4, y una 
 *  componente constante igual 1. 
 *
 *  <br>
 *  Test de un filtro IIR.
 *  \f[x[n]=1+sin(\frac{\pi}{32}n)+sin(\frac{3\pi}{4}n+\frac{\pi}{16})\f]
 *  El filtro usado será:
 *  \f[H(z)=\frac{1}{20}\frac{(1+z^{-1})}{(1-0.9z^{-1})}=\frac{\frac{1}{20}+\frac{1}{20}z^{-1}}{1-0.9z^{-1}} \f]
 *  <br>
 *  \image html test1grafico1.png "Señal de entrada x[n]."
 *  \image html test1grafico2.png "Señal de salida y[n]."
 */

#include <config.h>
#include <stdlib.h>
#include <pds/pdsdf.h>
#include <pds/pdsvector.h>
#include "extras.h"
#include <math.h>

int main(int argc, char** argv)
{
	///////////////////////////////////////////////////////////////
	PdsIir *IIR1=NULL;
	PdsVector *hden=NULL;
	PdsVector *hnum=NULL;

	PdsDfReal x,y;
	PdsDfNatural n,N;

	PdsVector *X=NULL;
	PdsVector *Y=NULL;

	//-------------------------- Creando filtro IIR --------------------------//

	// hnum=[ 1/20 1/20] 
	hnum=pds_vector_new_args(2, 1/20.0,1/20.0);
	// hden=[1.0 -0.9]
	hden=pds_vector_new_args(2,   1.0,   -0.9);

	IIR1=pds_iir_new(hnum,hden);

	//------------------------------- Filtrando ------------------------------//

	N=128;
	X=pds_vector_new(N);
	Y=pds_vector_new(N);

	for(n=0;n<N;n++)
	{
		x=1.0+sin(M_PI*n/32.0)+sin(3.0*M_PI*n/4.0+M_PI/16.0);
		pds_iir_evaluate_value(IIR1,x,&y);

		X->V[n]=x;
		Y->V[n]=y;
	}


	//-------------------------- Plotando resultados -------------------------//

    pds_vector_octave_png_plot(NULL,X,"n","X",NULL,"example1ax.png");
    pds_vector_octave_png_plot(NULL,Y,"n","Y",NULL,"example1ay.png");

	//---------------------------- Limpiando todo ----------------------------//
	
	pds_vector_free(hnum);	
	pds_vector_free(hden);
	pds_iir_free(IIR1);

	return EXIT_SUCCESS;
}
