/*
 * pdsfirlms.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */


#include <pds/pdsfir.h>
#include <pds/pdsfirlms.h>
#include <pds/pdsvector.h>


////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsFirLms                                              ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsFirLms *pds_fir_lms_new(PdsDfReal Mhu, PdsRaNatural M)
 *  \brief Crea un filtro FIR LMS con parametros h[i] del filtro FIR, con un 
 *  valor inicial de h[i]=1/(1+M). Por defecto el filtro FIR LMS estará 
 *  auto configurandose continuamente, a no ser que se deshabilite con
 *  pds_fir_lms_disable .
 * 
 *  \param[in] Mhu Es el paso de la constante de adaptación.
 *  \param[in] M Es el grado del filtro FIR, h[i], 0<= i <=M.
 *  \return Un puntero a una estructura de tipo PdsFirLms. O NULL en caso de error.
 *  \ingroup PdsFirLmsGroup
 */
PdsFirLms *pds_fir_lms_new(PdsDfReal Mhu, PdsRaNatural M)
{
	PdsFirLms	*FirLms=NULL;
	PdsVector	*h=NULL;
	PdsDfNatural	i;


	h=pds_vector_new(M+1);
	if(h==NULL)	return NULL;
	for(i=0;i<=M;i++)	h->V[i]=1.0/(M+1.0);


	FirLms=(PdsFirLms *)calloc(1,sizeof(PdsFirLms));
	if(FirLms==NULL)
	{	
		pds_vector_free(h);
		return NULL;
	}

	FirLms->Fir=pds_fir_new(h);
	if(FirLms->Fir==NULL)
	{	
		pds_vector_free(h);
		pds_fir_lms_free(FirLms);
		return NULL;
	}

	FirLms->Work=1;
	FirLms->Mhu=Mhu;

	pds_vector_free(h);

	return FirLms;
}


/** \fn int pds_fir_lms_disable(PdsFirLms *FirLms)
 *  \brief Deshabilita la reconfiguración de los pesos h[i] del filtro FIR LMS
 *  y se mantienen los ultimos pesos modificados
 * 
 *  \param[in,out] FirLms El filtro FIR LMS a usar.
 *  \return TRUE si todo fue bien o FALSE si no;
 *  \ingroup PdsFirLmsGroup
 */
int pds_fir_lms_disable(PdsFirLms *FirLms)
{

	if(FirLms==NULL)	return FALSE;

	FirLms->Work=0;
	return TRUE;
}


/** \fn int pds_fir_lms_enable(PdsFirLms *FirLms)
 *  \brief Habilita la reconfiguración de los pesos h[i] del filtro FIR LMS.
 * 
 *  \param[in,out] FirLms El filtro FIR LMS a usar.
 *  \return TRUE si todo fue bien o FALSE si no;
 *  \ingroup PdsFirLmsGroup
 */
int pds_fir_lms_enable(PdsFirLms *FirLms)
{

	if(FirLms==NULL)	return FALSE;

	FirLms->Work=1;
	return TRUE;
}

/** \fn int pds_fir_lms_set_mhu(PdsFirLms *FirLms,PdsDfReal Mhu)
 *  \brief Coloca el valor Mhu del filtro FIR LMS.
 * 
 *  \param[in,out] FirLms El filtro FIR LMS a usar.
 *  \param[in] Mhu Factor de aprendizaje Mhu.
 *  \return TRUE si todo fue bien o FALSE si no;
 *  \ingroup PdsFirLmsGroup
 */
int pds_fir_lms_set_mhu(PdsFirLms *FirLms,PdsDfReal Mhu)
{

	if(FirLms==NULL)	return FALSE;
	FirLms->Mhu=Mhu;
	return TRUE;
}


/** \fn int pds_fir_lms_evaluate_value(PdsFirLms *FirLms,PdsDfReal d,PdsDfReal x,PdsDfReal *e,PdsDfReal *y)
 *  \brief Evalúa el filtro FIR LMS con el valor de entrada "d" e "x", el 
 *  resultado es cargado en "e" e "y".
 * 
 *  \param[in,out] FirLms El filtro FIR LMS a usar.
 *  \param[in] d Es una entrada adicional al filtro FIR LMS.
 *  \param[in] x Es la señal de entrada del filtro FIR.
 *  \param[out] e Es la señal de error, es la diferencia entre "d" e "y".
 *  \param[out] y Es el valor de salida del filtro FIR.
 *  \return TRUE si todo fue bien o FALSE si no;
 *  \ingroup PdsFirLmsGroup
 */
int pds_fir_lms_evaluate_value(PdsFirLms *FirLms,PdsDfReal d,PdsDfReal x,PdsDfReal *e,PdsDfReal *y)
{
	PdsRaNatural i,N;

	if(FirLms==NULL)	return FALSE;

	pds_fir_evaluate_value(FirLms->Fir,x,y);
	*e=d-(*y);

	N=FirLms->Fir->N;
	if(FirLms->Work==1)
	{
		for(i=0;i<N;i++)
		FirLms->Fir->h->V[i] = 	FirLms->Fir->h->V[i]+
					FirLms->Mhu*(*e)*FirLms->Fir->X->V[i];
	}

	return TRUE;
}


/** \fn int pds_fir_lms_evaluate_vector(PdsFirLms *FirLms,const PdsVector *d,const PdsVector *x,PdsVector *e,PdsVector *y)
 *  \brief Evalúa el filtro FIR LMS con el vector de entrada x, el resultado
 *  es cargado en el vector y.
 * 
 *  Se recomienda usar esta función solo cuando x es mucho mayo que FIR->h.
 *  Solo se realizan corrimientos de FIR->X al inicio y al final del vector x
 *  en los casos intermediarios se aprovecha tener el vector y no se efectuan
 *  corrimientos, por lo que es un poco mas rápido que pds_fir_lms_evaluate_value
 *  cuando x es mucho mayo que FIR->h.
 *  \param[in,out] FirLms El filtro FIR a usar.
 *  \param[in] d Es el vector de una entrada adicional al filtro FIR LMS.
 *  \param[in] x Es el vector de la señal de entrada del filtro FIR.
 *  \param[out] e Es el vector de la señal de error, es la diferencia entre "d" e "y".
 *  \param[out] y Es el vector del valor de salida del filtro FIR.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsFirLmsGroup
 */
int pds_fir_lms_evaluate_vector(PdsFirLms *FirLms,const PdsVector *d,const PdsVector *x,PdsVector *e,PdsVector *y)
{
	PdsRaNatural i,j,M,N,Nel;
	PdsDfReal S;

	if(FirLms==NULL)	return FALSE;
	if((d==NULL)||(x==NULL)||(e==NULL)||(y==NULL))	return FALSE;
	if((x->Nel!=y->Nel)||(e->Nel!=d->Nel)||(x->Nel!=d->Nel))	return FALSE;

	N=FirLms->Fir->N;
	Nel=x->Nel;

	// Evaluó salida y tramo1.
	for(j=0;(j<N)&&(j<Nel);j++)
	{
		S=0;
		for(i=0;i<=j;i++)
		{
			S=S+FirLms->Fir->h->V[i]*x->V[j-i];
		}
		for(i=j+1;i<N;i++)
		{
			S=S+FirLms->Fir->h->V[i]*FirLms->Fir->X->V[i-(j+1)];
		}	

		y->V[j]=S;
		e->V[j]=d->V[j]-(y->V[j]);
		if(FirLms->Work==1)
		{
			for(i=0;i<=j;i++)
			FirLms->Fir->h->V[i] = 	FirLms->Fir->h->V[i]+
						FirLms->Mhu*e->V[j]*x->V[j-i];
			for(i=j+1;i<N;i++)
			FirLms->Fir->h->V[i] = 	FirLms->Fir->h->V[i]+
						FirLms->Mhu*e->V[j]*FirLms->Fir->X->V[i-(j+1)];
		}
	}
	// Evaluó salida y tramo2.
	for(j=N;j<Nel;j++)
	{
		S=0;
		for(i=0;i<=(N-1);i++)
		{
			S=S+FirLms->Fir->h->V[i]*x->V[j-i];
		}
		y->V[j]=S;
		e->V[j]=d->V[j]-(y->V[j]);
		if(FirLms->Work==1)
		{
			for(i=0;i<N;i++)
			FirLms->Fir->h->V[i] = 	FirLms->Fir->h->V[i]+
						FirLms->Mhu*e->V[j]*x->V[j-i];
		}
	}
	// Ordenamiento final.
	if(N>Nel)	M=N-Nel;
	else		M=0;
	for(j=0;j<M;j++)
	{
		FirLms->Fir->X->V[N-1-j]=FirLms->Fir->X->V[M-1-j];
	}
	for(j=0;(j<Nel)&&(j<N);j++)
	{
		FirLms->Fir->X->V[j]=x->V[(Nel-1)-j];
	}


	return TRUE;
}


/** \fn void pds_fir_lms_free(PdsFirLms *FirLms)
 *  \brief Libera el filtro de tipo PdsFirLms.
 *  \param[in] FirLms El filtro a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsFirLmsGroup
 */
void pds_fir_lms_free(PdsFirLms *FirLms)
{
	if(FirLms!=NULL)
	{
		pds_fir_free(FirLms->Fir);
		free(FirLms);
	}
}


/** \fn void pds_fir_lms_destroy(PdsFirLms **FirLms)
 *  \brief Libera el filtro de tipo PdsFirLms. y carga la variable con NULL.
 *  \param[in] FirLms El filtro a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsFirLmsGroup
 */
void pds_fir_lms_destroy(PdsFirLms **FirLms)
{
	if((*FirLms)!=NULL)
	{
		pds_fir_free((*FirLms)->Fir);
		free((*FirLms));
		(*FirLms)=NULL;
	}
}


