/*
 * pdsfir.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <math.h>
#include <pds/pdsfir.h>
#include <pds/pdsvector.h>
#include <pds/pdsdfutils.h>

////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsFir                                                 ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsFir *pds_fir_new(const PdsVector *h)
 *  \brief Crea un filtro FIR con parametros h.
 * 
 *  El vector h entregado como parámetro a esta función nunca es modificado.
 *  \param[in] h Vector para los coeficientes de h_i*x[n-i].
 *  \return Un puntero a una estructura de tipo PdsFir. O NULL en caso de error.
 *  \ingroup PdsFirGroup
 */
PdsFir *pds_fir_new(const PdsVector *h)
{
	PdsFir *FIR=NULL;

	if(h==NULL)	return NULL;

	FIR=(PdsFir *)calloc(1,sizeof(PdsFir));
	if(FIR==NULL)	return NULL;

	FIR->N=h->Nel;
	FIR->h=pds_vector_new_vector(h);
	if(FIR->h==NULL)
	{
		free(FIR);
		return NULL;
	}

	FIR->X=pds_vector_new(FIR->N);
	if(FIR->X==NULL)
	{
		pds_vector_free(FIR->h);
		free(FIR);
		return NULL;
	}

	return FIR;
}


/** \fn PdsFir *pds_fir_get_new_lowpass(PdsDfNatural Order,PdsDfReal Wn,const PdsVector *Window)
 *  \brief Crea un filtro FIR pasa bajo, con un orden y frecuencia de corte 
 *  especificado. 
 * 
 *  Puede ser usado un vector de ponderación (Window) para mejorar el 
 *  comportamiento del filtro.
 *  Esta función crea un filtro de tipo I : 
 *  http://www.mathworks.com/help/signal/ug/fir-filter-design.html 
 *  Para elegir el orden minimo del filtro puede elegirse el criterio : 
 *  \f[ Order \geq \frac{4}{BW} \f] 
 *  where
 *  \f[ BW=|Wn-round(Wn)| \f]
 *  Este criterio es similar a http://www.dspguide.com/ch16/2.htm
 *  \param[in] Order El orden del filtro pasa bajo.
 *  \param[in] Wn La frecuencia de corte normalizada del filtro pasa bajo. Wn
 *  significa que la frecuencia de corte es Wn*Fs/2, donde Fs es la frecuencia
 *  de muestreo.
 *  \param[in] Window Es un vector de ponderación para mejorar el comportamiento
 *  del filtro. Si este valor es igual a NULL entonces es usado una ventana de
 *  hamming como vector de ponderación.
 *  \return Un puntero a una estructura de tipo PdsFir. O NULL en caso de error.
 *  \ingroup PdsFirGroup
 */
PdsFir *pds_fir_get_new_lowpass(PdsDfNatural Order,PdsDfReal Wn,const PdsVector *Window)
{
    PdsVector *h=NULL;
    PdsVector *W=NULL;
    PdsFir *fir=NULL;
    PdsDfNatural i;
    PdsDfReal n,Sum;

    if((Wn>1.0)&&(Wn<0))  return NULL;
    
    if(Window==NULL)
    {
        W=pds_vector_new_hamming(Order+1);
        if (W==NULL)    return NULL;
    }
    else
    {
        if(Window->Nel!=(Order+1))
        {  
            fprintf(stderr,"The order+1 of filter is different of length of window vector.");
            return NULL;
        }

        W=pds_vector_new_vector(Window);
        if (W==NULL)    return NULL;
    }
    
    h=pds_vector_new(Order+1);
    if(h==NULL)
    {
        pds_vector_free(W);
        return NULL;
    }

    for(i=0;i<h->Nel;i++)
    {
        n=-(Order/2.0)+i;

        if(n!=0)        h->V[i]=sin(M_PI*Wn*n)/(M_PI*n);

    }
    if(Order%2==0)  h->V[Order/2]=Wn;

    // Verificando que la funcion sea par.
    for(i=0;i<lround(Order/2.0);i++)
    {
        h->V[i]=0.5*(h->V[i]+h->V[h->Nel-1-i]);
        h->V[h->Nel-1-i]=h->V[i];
    }

    pds_vector_mul_vector_elements(h,W); 

    // Encontrando la suma de todos los elementos.
    for(i=0,Sum=0;i<h->Nel;i++) Sum=Sum+h->V[i];
    // Normalizando el vector para tener suma 1.
    for(i=0;i<h->Nel;i++) h->V[i]=h->V[i]/Sum;

    fir=pds_fir_new(h);

    pds_vector_free(h);
    pds_vector_free(W);


    return fir;
}



/** \fn PdsFir *pds_fir_get_new_highpass(PdsDfNatural Order,PdsDfReal Wn,const PdsVector *Window)
 *  \brief Crea un filtro FIR pasa alto, con un orden y frecuencia de corte 
 *  especificado. 
 * 
 *  Puede ser usado un vector de ponderación (Window) para mejorar el 
 *  comportamiento del filtro.
 *  Esta función crea un filtro de tipo I si Order es par e tipo IV si Order es 
 *  impar : 
 *  http://www.mathworks.com/help/signal/ug/fir-filter-design.html 
 *  Para elegir el orden minimo del filtro puede elegirse el criterio : 
 *  \f[ Order \geq \frac{4}{BW} \f] 
 *  where
 *  \f[ BW=|Wn-round(Wn)| \f]
 *  Este criterio es similar a http://www.dspguide.com/ch16/2.htm
 *  \param[in] Order El orden del filtro pasa alto.
 *  \param[in] Wn La frecuencia de corte normalizada del filtro pasa alto. Wn
 *  significa que la frecuencia de corte es Wn*Fs/2, donde Fs es la frecuencia
 *  de muestreo.
 *  \param[in] Window Es un vector de ponderación para mejorar el comportamiento
 *  del filtro. Si este valor es igual a NULL entonces es usado una ventana de
 *  hamming como vector de ponderación.
 *  \return Un puntero a una estructura de tipo PdsFir. O NULL en caso de error.
 *  \ingroup PdsFirGroup
 */
PdsFir *pds_fir_get_new_highpass(PdsDfNatural Order,PdsDfReal Wn,const PdsVector *Window)
{
    PdsFir *fir=NULL;
    int i;

    if((Wn>1.0)&&(Wn<0))  return NULL;

    fir=pds_fir_get_new_lowpass(Order,1-Wn,Window);
    if(fir==NULL)   return NULL;
    for(i=0;i<=Order;i++)
    {
        if ((i%2)!=0)   fir->h->V[i]=-fir->h->V[i]; 
    }

    if((Order%2)==0)
    if (fir->h->V[Order/2]<0)
    for(i=0;i<=Order;i++)
    {
        fir->h->V[i]=-fir->h->V[i]; 
    }
    return fir;
}



/** \fn PdsFir *pds_fir_get_new_bandpass(PdsDfNatural Order,PdsDfReal Wn1,PdsDfReal Wn2,const PdsVector *Window)
 *  \brief Crea un filtro FIR pasa banda, con un orden y frecuencias de corte 
 *  especificados. 
 * 
 *  Puede ser usado un vector de ponderación (Window) para mejorar el 
 *  comportamiento del filtro.
 *  Esta función crea un filtro de tipo I si Order es par e tipo II si Order es 
 *  impar : 
 *  http://www.mathworks.com/help/signal/ug/fir-filter-design.html 
 *  Para elegir el orden minimo del filtro puede elegirse el criterio : 
 *  \f[ Order \geq \frac{8}{BW} \f] 
 *  where
 *  \f[ BW=|Wn1-Wn2-round(Wn1-Wn2)| \f]
 *  Este criterio es similar a http://www.dspguide.com/ch16/2.htm
 *  \param[in] Order El orden del filtro pasa alto.
 *  \param[in] Wn1 La frecuencia de corte normalizada del filtro pasa alto. Wn1
 *  significa que la frecuencia de corte es Wn1*Fs/2, donde Fs es la frecuencia
 *  de muestreo.
 *  \param[in] Wn2 La frecuencia de corte normalizada del filtro pasa bajo. Wn2
 *  significa que la frecuencia de corte es Wn2*Fs/2, donde Fs es la frecuencia
 *  de muestreo.
 *  \param[in] Window Es un vector de ponderación para mejorar el comportamiento
 *  del filtro. Si este valor es igual a NULL entonces es usado una ventana de
 *  hamming como vector de ponderación.
 *  \return Un puntero a una estructura de tipo PdsFir. O NULL en caso de error.
 *  \ingroup PdsFirGroup
 */
PdsFir *pds_fir_get_new_bandpass(PdsDfNatural Order,PdsDfReal Wn1,PdsDfReal Wn2,const PdsVector *Window)
{
    PdsFir *fir=NULL;

    PdsVector *Wr=NULL;
    PdsVector *Wr1=NULL;
    PdsVector *W=NULL;
    PdsFir *lpf=NULL;
    PdsFir *hpf=NULL;
    PdsVector *hc=NULL;
    PdsVector *h=NULL;
    PdsRaNatural i;
    int dat;
    PdsRaReal max;

    if((Wn1>1.0)&&(Wn1<0))  return NULL;
    if((Wn2>1.0)&&(Wn2<0))  return NULL;

    if(Window==NULL)
    {
        W=pds_vector_new_hamming(Order+1);
        if (W==NULL)    return NULL;
    }
    else
    {
        if(Window->Nel!=(Order+1))
        {  
            fprintf(stderr,"The order+1 of filter is different of length of window vector.");
            return NULL;
        }

        W=pds_vector_new_vector(Window);
        if (W==NULL)    return NULL;
    }
    
    Wr =pds_vector_new_rectangular(Order+1);
    Wr1=pds_vector_new_rectangular(Order);
    if( (Wr==NULL)||(Wr1==NULL) )
    {
        pds_vector_free(W);
        pds_vector_free(Wr);
        pds_vector_free(Wr1);
        return NULL;
    }


    if((Order%2)==0)  // Order Par
    {
        hpf=pds_fir_get_new_highpass(Order, Wn1, Wr);
        lpf=pds_fir_get_new_lowpass (Order, Wn2, Wr);
        if((lpf==NULL)||(hpf==NULL))
        {
            pds_vector_free(W);     pds_vector_free(Wr);    pds_vector_free(Wr1);
            pds_fir_free(lpf);      pds_fir_free(hpf);
            return NULL;
        }

        hc=pds_vector_new_conv(lpf->h,hpf->h); // order:Order+Order;
        h =pds_vector_new(Order+1);
        if((h==NULL)||(hc==NULL))
        {
            pds_vector_free(W);     pds_vector_free(Wr);    pds_vector_free(Wr1);
            pds_fir_free(lpf);      pds_fir_free(hpf);
            pds_vector_free(h);     pds_vector_free(hc);
            return NULL;
        }

        for(i=Order/2;i<=3*Order/2;i++)
        {   
            h->V[i-Order/2]=hc->V[i];
        }
    }
    else 
    {
        lpf=pds_fir_get_new_lowpass (Order  , Wn2, Wr);
        hpf=pds_fir_get_new_highpass(Order-1, Wn1, Wr1);
 
        if((lpf==NULL)||(hpf==NULL))
        {
            pds_vector_free(W);     pds_vector_free(Wr);    pds_vector_free(Wr1);
            pds_fir_free(lpf);      pds_fir_free(hpf);
            return NULL;
        }

        hc=pds_vector_new_conv(lpf->h,hpf->h); // order:Order+Order;
        h =pds_vector_new(Order+1);
        if((h==NULL)||(hc==NULL))
        {
            pds_vector_free(W);     pds_vector_free(Wr);    pds_vector_free(Wr1);
            pds_fir_free(lpf);      pds_fir_free(hpf);
            pds_vector_free(h);     pds_vector_free(hc);
            return NULL;
        }

        for(i=(Order-1)/2;i<=(3*Order-1)/2;i++)
        {   
            h->V[i-(Order-1)/2]=hc->V[i];
        }
    }

    // Windowing
    for(i=0;i<=Order;i++)
    {   
        h->V[i]=h->V[i]*W->V[i];
    }
    pds_vector_free(W);     pds_vector_free(Wr);    pds_vector_free(Wr1);
    pds_fir_free(lpf);      pds_fir_free(hpf);
    pds_vector_free(hc);

    PdsVector *Hw=pds_vector_new(round(10*Order));
    if(Hw==NULL)
    {
        pds_vector_free(h);
        return NULL;
    }

    dat=pds_vector_fir_frequency_response(h,Hw);
    if(dat==FALSE)
    {
        pds_vector_free(Hw);
        pds_vector_free(h);
        return NULL;
    }
    pds_vector_get_max_value(Hw,&max);
    pds_vector_mul_value(h,1/max);

    pds_vector_free(Hw);

    fir=pds_fir_new(h);

    pds_vector_free(h);     

    return fir;
}




/** \fn PdsFir *pds_fir_get_new_bandstop(PdsDfNatural Order,PdsDfReal Wn1,PdsDfReal Wn2,const PdsVector *Window)
 *  \brief Crea un filtro FIR rechaza banda, con un orden y frecuencias de corte 
 *  especificados. 
 * 
 *  Puede ser usado un vector de ponderación (Window) para mejorar el 
 *  comportamiento del filtro.
 *  Esta función crea un filtro de tipo I si Order es par e tipo II si Order es 
 *  impar : 
 *  http://www.mathworks.com/help/signal/ug/fir-filter-design.html 
 *  Para elegir el orden minimo del filtro puede elegirse el criterio : 
 *  \f[ Order \geq \frac{8}{BW} \f] 
 *  where
 *  \f[ BW=|Wn1-Wn2-round(Wn1-Wn2)| \f]
 *  Este criterio es similar a http://www.dspguide.com/ch16/2.htm
 *  \param[in] Order El orden del filtro pasa alto.
 *  \param[in] Wn1 La frecuencia de corte normalizada del filtro bajo. Wn1
 *  significa que la frecuencia de corte es Wn1*Fs/2, donde Fs es la frecuencia
 *  de muestreo.
 *  \param[in] Wn2 La frecuencia de corte normalizada del filtro alto. Wn2
 *  significa que la frecuencia de corte es Wn2*Fs/2, donde Fs es la frecuencia
 *  de muestreo.
 *  \param[in] Window Es un vector de ponderación para mejorar el comportamiento
 *  del filtro. Si este valor es igual a NULL entonces es usado una ventana de
 *  hamming como vector de ponderación.
 *  \return Un puntero a una estructura de tipo PdsFir. O NULL en caso de error.
 *  \ingroup PdsFirGroup
 */
PdsFir *pds_fir_get_new_bandstop(PdsDfNatural Order,PdsDfReal Wn1,PdsDfReal Wn2,const PdsVector *Window)
{
    PdsFir *fir=NULL;

    PdsVector *Wr=NULL;
    PdsVector *Wr1=NULL;
    PdsVector *W=NULL;
    PdsFir *lpf=NULL;
    PdsFir *hpf=NULL;
    PdsVector *hc=NULL;
    PdsVector *h=NULL;
    PdsRaNatural i;
    int dat;
    PdsRaReal max;

    PdsVector *Hw=NULL;

    if((Wn1>1.0)&&(Wn1<0))  return NULL;
    if((Wn2>1.0)&&(Wn2<0))  return NULL;

    if(Window==NULL)
    {
        W=pds_vector_new_hamming(Order+1);
        if (W==NULL)    return NULL;
    }
    else
    {
        if(Window->Nel!=(Order+1))
        {  
            fprintf(stderr,"The order+1 of filter is different of length of window vector.");
            return NULL;
        }

        W=pds_vector_new_vector(Window);
        if (W==NULL)    return NULL;
    }
    
    Wr =pds_vector_new_rectangular(Order+1);
    Wr1=pds_vector_new_rectangular(Order);
    if( (Wr==NULL)||(Wr1==NULL) )
    {
        pds_vector_free(W);
        pds_vector_free(Wr);
        pds_vector_free(Wr1);
        return NULL;
    }


    if((Order%2)==0)  // Order Par
    {
        hpf=pds_fir_get_new_highpass(Order, Wn1, Wr);
        lpf=pds_fir_get_new_lowpass (Order, Wn2, Wr);
        if((lpf==NULL)||(hpf==NULL))
        {
            pds_vector_free(W);     pds_vector_free(Wr);    pds_vector_free(Wr1);
            pds_fir_free(lpf);      pds_fir_free(hpf);
            return NULL;
        }

        hc=pds_vector_new_conv(lpf->h,hpf->h); // order:Order+Order;
        h =pds_vector_new(Order+1);
        if((h==NULL)||(hc==NULL))
        {
            pds_vector_free(W);     pds_vector_free(Wr);    pds_vector_free(Wr1);
            pds_fir_free(lpf);      pds_fir_free(hpf);
            pds_vector_free(h);     pds_vector_free(hc);
            return NULL;
        }

        for(i=Order/2;i<=3*Order/2;i++)
        {   
            h->V[i-Order/2]=hc->V[i];
        }

        // Windowing
        for(i=0;i<=Order;i++)
        {   
            h->V[i]=h->V[i]*W->V[i];
        }

        Hw=pds_vector_new(round(10*Order));
        if(Hw==NULL)
        {
            pds_vector_free(W);     pds_vector_free(Wr);    pds_vector_free(Wr1);
            pds_fir_free(lpf);      pds_fir_free(hpf);
            pds_vector_free(h);     pds_vector_free(hc);
            return NULL;
        }        
        dat=pds_vector_fir_frequency_response(h,Hw);
        pds_vector_get_max_value(Hw,&max);
        pds_vector_mul_value(h,1/max);
        pds_vector_free(Hw);

        for(i=0;i<=Order;i++)
        {   
            h->V[i]=-h->V[i];
        }
        h->V[Order/2]=1+h->V[Order/2];
    }
    else 
    {
        fprintf(stderr,"\nThis function can't creates a band stop fir filter with order %d of types I, II, III or IV.\n",Order);
        fprintf(stderr,"Please see types of fir filter in:\n");
        fprintf(stderr,"http://www.mathworks.com/help/signal/ug/fir-filter-design.html\n");
        pds_vector_free(W);     pds_vector_free(Wr);
        return NULL;
    }


    pds_vector_free(W);     pds_vector_free(Wr);    pds_vector_free(Wr1);
    pds_fir_free(lpf);      pds_fir_free(hpf);
    pds_vector_free(hc);

    fir=pds_fir_new(h);

    pds_vector_free(h);     

    return fir;
}


/** \fn int pds_fir_evaluate_value(PdsFir *FIR,PdsDfReal x,PdsDfReal *y)
 *  \brief Evalúa el filtro FIR con el valor de entrada x, el resultado
 *  es cargado en y.
 * 
 *  En cada iteración se realizan corrimientos para el vector FIR->X.
 *  \param[in,out] FIR El filtro FIR a usar.
 *  \param[in] x El valor de entrada del filtro.
 *  \param[out] y El valor de salida del filtro.
 *  \return TRUE si todo fue bien o FALSE si no;
 *  \ingroup PdsFirGroup
 */
int pds_fir_evaluate_value(PdsFir *FIR,PdsDfReal x,PdsDfReal *y)
{
	PdsRaNatural i;
	PdsDfReal S;

	if(FIR==NULL)	return FALSE;

	// Corrimiento de x.
	for(i=(FIR->N-1);i>0;i--) 
	{
		FIR->X->V[i]=FIR->X->V[i-1];
	}
	FIR->X->V[0]=x;

	// Evaluo salida y.
	for(i=0,S=0;i<FIR->N;i++)
	{
		S=S+FIR->h->V[i]*FIR->X->V[i];
	}

	*y=S;

	return TRUE;
}


/** \fn int pds_fir_evaluate_vector(PdsFir *FIR,const PdsVector *x,PdsVector *y)
 *  \brief Evalúa el filtro FIR con el vector de entrada x, el resultado
 *  es cargado en el vector y.
 * 
 *  Se recomienda usar esta función solo cuando x es mucho mayo que FIR->h.
 *  Solo se realizan corrimientos de FIR->X al inicio y al final del vector x
 *  en los casos intermediarios se aprovecha tener el vector y no se efectuan
 *  corrimientos, por lo que es un poco mas rápido que pds_fir_evaluate_value
 *  cuando x es mucho mayo que FIR->h.
 *  \param[in,out] FIR El filtro FIR a usar.
 *  \param[in] x El vector de entrada del filtro.
 *  \param[out] y El vector de salida del filtro.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsFirGroup
 */
int pds_fir_evaluate_vector(PdsFir *FIR,const PdsVector *x,PdsVector *y)
{
	PdsRaNatural i,j,M,N,Nel;
	PdsDfReal S;

	if(FIR==NULL)		return FALSE;
	if(x==NULL)		return FALSE;
	if(y==NULL)		return FALSE;
	if(x->Nel!=y->Nel)	return FALSE;
	

	N=FIR->N;
	Nel=x->Nel;
	// Evaluó salida y tramo1.
	for(j=0;(j<N)&&(j<Nel);j++)
	{
		S=0;
		for(i=0;i<=j;i++)
		{
			S=S+FIR->h->V[i]*x->V[j-i];
		}
		for(i=j+1;i<N;i++)
		{
			S=S+FIR->h->V[i]*FIR->X->V[i-(j+1)];
		}	
		y->V[j]=S;
	}
	// Evaluó salida y tramo2.
	for(j=N;j<Nel;j++)
	{
		S=0;
		for(i=0;i<=(N-1);i++)
		{
			S=S+FIR->h->V[i]*x->V[j-i];
		}
		y->V[j]=S;
	}
	// Ordenamiento final.
	if(N>Nel)	M=N-Nel;
	else			M=0;
	for(j=0;j<M;j++)
	{
		FIR->X->V[N-1-j]=FIR->X->V[M-1-j];
	}
	for(j=0;(j<Nel)&&(j<N);j++)
	{
		FIR->X->V[j]=x->V[(Nel-1)-j];
	}
	return TRUE;
}


/** \fn int pds_fir_get_x_value(PdsFir *FIR,PdsDfNatural i,PdsDfReal *x)
 *  \brief Esta funcion retorna el valor de X[n-i].
 * 
 *  Siendo n la muestra actual e i indica la i-esima muestra anterior.
 *
 *  \param[in] FIR El filtro FIR a usar.
 *  \param[in] i El valor i de la posicion de X[n-i] en la entrada del filtro.
 *  \param[out] x El valor de X[n-i].
 *  \return TRUE si todo fue bien o FALSE si no;
 *  \ingroup PdsFirGroup
 */
int pds_fir_get_x_value(PdsFir *FIR,PdsDfNatural i,PdsDfReal *x)
{
	PdsDfReal S;

	if(FIR==NULL)	return FALSE;

	if((i>=0)&&(i<FIR->N))	*x=FIR->X->V[i];

	return TRUE;
}

/** \fn int pds_fir_get_h_value(PdsFir *FIR,PdsDfNatural i,PdsDfReal *h)
 *  \brief Esta funcion retorna el valor de h[i].
 * 
 *  Siendo h[i] el factor de ponderacion de X[n-i].
 *
 *  \param[in] FIR El filtro FIR a usar.
 *  \param[in] i El valor i de la posicion de h[i].
 *  \param[out] h El valor de h[i].
 *  \return TRUE si todo fue bien o FALSE si no;
 *  \ingroup PdsFirGroup
 */
int pds_fir_get_h_value(PdsFir *FIR,PdsDfNatural i,PdsDfReal *h)
{
	PdsDfReal S;

	if(FIR==NULL)	return FALSE;

	if((i>=0)&&(i<FIR->N))	*h=FIR->h->V[i];

	return TRUE;
}

/** \fn const PdsVector* pds_fir_get_h_vector(PdsFir *FIR)
 *  \brief Esta funcion retorna la dirección del vector h con los valores del 
 *  filtro fir.
 * 
 *  \param[in] FIR El filtro FIR a usar.
 *  \return Retorna la dirección del vector h con los valores del filtro fir. o NULL en caso 
 *  de error. Este vector no debe ser modificado pues también modificaría al filtro
 *  \ingroup PdsFirGroup
 */
const PdsVector* pds_fir_get_h_vector(PdsFir *FIR)
{
	if(FIR==NULL)	return NULL;

	return FIR->h;
}

/** \fn int pds_fir_get_order(PdsFir *FIR)
 *  \brief Esta función retorna o grado del filtro FIR.
 *
 *  \param[in] FIR El filtro FIR a usar.
 *  \return El orden del filtro o -1 en caso de error;
 *  \ingroup PdsFirGroup
 */
int pds_fir_get_order(PdsFir *FIR)
{
	if(FIR==NULL)	return -1;

	return (int)FIR->N-1;
}

/** \fn int pds_fir_get_length(PdsFir *FIR)
 *  \brief Esta función retorna la longitud del vector que representa el filtro FIR.
 *
 *  \param[in] FIR El filtro FIR a usar.
 *  \return La longitud del filtro o -1 en caso de error;
 *  \ingroup PdsFirGroup
 */
int pds_fir_get_length(PdsFir *FIR)
{
	if(FIR==NULL)	return -1;

	return (int)FIR->N;
}


/** \fn int pds_fir_frequency_response(const PdsFir *FIR,PdsVector *H)
 *  \brief Encuentra el módulo de la respuesta en frecuencia, normalizada 
 *  de 0 a pi del filtro digital conformado por la respuesta al impulso h.
 *
 *  <b>Teoría</b><br>
 *  \f[ H(w)=\left|h(Z=e^{j w})\right|, ~\forall ~ 0 \leq w \leq \pi \f]
 *
 *  \param[in] FIR Filtro fir a anaizar.
 *  \param[out] H Donde se guardará el módulo cuadrado de la respuesta en 
 *  frecuencia del filtro digital.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. FIR==NULL o H==NULL)
 *  \ingroup PdsFirGroup
 */
int pds_fir_frequency_response(const PdsFir *FIR,PdsVector *H)
{
    int dat;

	if(FIR==NULL)	return FALSE;
	if(H==NULL)	    return FALSE;

    dat=pds_vector_fir_frequency_response(FIR->h,H);
    
	return dat;
}

/** \fn void pds_fir_free(PdsFir *FIR)
 *  \brief Libera el filtro de tipo PdsFir.
 *  \param[in] FIR El filtro a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsFirGroup
 */
void pds_fir_free(PdsFir *FIR)
{
	if(FIR!=NULL)
	{
		pds_vector_free(FIR->h);
		pds_vector_free(FIR->X);
		free(FIR);
	}
}


/** \fn void pds_fir_destroy(PdsFir **FIR)
 *  \brief Libera el filtro de tipo PdsFir. y carga la variable con NULL.
 *  \param[in] FIR El filtro a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsFirGroup
 */
void pds_fir_destroy(PdsFir **FIR)
{
	if((*FIR)!=NULL)
	{
		pds_vector_free((*FIR)->h);
		pds_vector_free((*FIR)->X);
		free((*FIR));
		(*FIR)=NULL;
	}
}


