/*
 * pdsdfutils.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <math.h>
#include <pds/pdsvector.h>
#include <pds/pdsdfutils.h>

/** \fn int pds_vector_lowpass_butterworth(PdsVector *hnum,PdsVector *hden,PdsDfReal Wn)
 *  \brief Encuentra el numerador y denominador de un filtro Butterworth pasa bajo. 
 *  Ambos vectores de entrada tienen la misma longitud Nel y Nel-1 es el orden
 *  N del filtro Butterworth
 *
 *  \f[ \left|H(j\Omega )\right|={1 \over {\sqrt {1+(\Omega /\Omega_{c})^{2N}}}}, ~~\forall~ \Omega=2 \pi f, ~ \Omega_{c}=2 \pi f_{c} \f]
 *  \f[ H(j\Omega )H(-j\Omega ) =\left|H(j\Omega )\right|^2\f]
 *
 *  <b>Teoría</b><br>
 *  Usa una transformación bi-linear 
 *  \f[ \Omega \leftarrow -j~ \frac{2}{T_d}~{\frac{z-1}{z+1}} \equiv  \frac{2}{T_d}~tan(\frac{w}{2}) \f]
 *
 *  Con una frecuencia de corte igual a \f$ \Omega_c \leftarrow \frac{2}{T_d}~tan(\frac{W_n \pi}{2}) \f$ 
 *  Por comodidad es escogido \f$  T_d \equiv  2~tan(\frac{W_n \pi}{2})\f$ 
 *
 *  \f[ \left( \frac{\Omega}{\Omega_{c}}\right)^{2n} \leftarrow (-1)^n \left(\frac{2}{T_d}\right)^{2n} \left(\frac{z-1}{z+1}\right)^{2n} \f]
 *
 *  \param[out] hnum Coeficientes del numerador del filtro digital.
 *  \param[out] hden Coeficientes del denominador del filtro digital.
 *  \param[in] Wn Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. hnum==NULL, hden==NULL, Wn<0 o Wn>1)
 *  \ingroup PdsDfUtilsGroup
 */
int pds_vector_lowpass_butterworth(PdsVector *hnum,PdsVector *hden,PdsDfReal Wn)
{
	PdsDfNatural i,j,Order,Nel;
	PdsRaReal a,b;
	PdsRaReal teta;
	PdsRaReal T;

	if(hden->Nel!=hnum->Nel)	return FALSE;
	if(hden->Nel<2)			return FALSE;
	if(Wn>1.0)			return FALSE;
	if(Wn<0.0)			return FALSE;

	Nel=hden->Nel;
	Order  =Nel-1;

	T=2.0*tan(Wn*M_PI/2.0);
	b=2.0/T;

	pds_vector_init_value(hnum,0.0);
	pds_vector_init_value(hden,0.0);


	if( (Order%2)==1 )	//Impar
	{
		hnum->V[0]=1.0;		hnum->V[1]=1.0;
		hden->V[0]=1.0+b;	hden->V[1]=1.0-b;
	}
    else
    {
	    hnum->V[0]=1.0;	
    	hden->V[0]=1.0;	
    }

	for(i=0;(2*i)<(Order-1.0);i++)
	{	
		teta=M_PI*(Order+1.0+2.0*i)/(2.0*Order);

		a=-2.0*cos(teta);

		for(j=0;j<Nel;j++)
		{
			if     (Order>=(j+2))	
			{
				hnum->V[Order-j]=	1.0*hnum->V[Order-j] + 2.0*hnum->V[Order-j-1] +	1.0*hnum->V[Order-j-2];

				hden->V[Order-j]=	(1.0+a*b+b*b)*hden->V[Order-j] +
						2.0*(1.0-b*b)*hden->V[Order-j-1] +
						(1.0-a*b+b*b)*hden->V[Order-j-2];
			}
			else if(Order>=(j+1))
			{
				hnum->V[Order-j]=	1.0*hnum->V[Order-j] +
							2.0*hnum->V[Order-j-1];
				hden->V[Order-j]=	(1.0+a*b+b*b)*hden->V[Order-j] +
							2.0*(1.0-b*b)*hden->V[Order-j-1];
			}
			else if(Order>=j)
			{
				hnum->V[Order-j]=	1.0*hnum->V[Order-j];
				hden->V[Order-j]=	(1.0+a*b+b*b)*hden->V[Order-j];
			}

		}
	}

	pds_vector_mul_value(hnum,1/hden->V[0]);
	pds_vector_mul_value(hden,1/hden->V[0]);
	return TRUE;
}


/** \fn PdsVector *pds_vector_new_butterworth_hnum(PdsDfNatural Order,PdsDfReal Wn)
 *  \brief Encuentra el numerador de un filtro Butterworth pasa bajo de orden Order. 
 *
 *  \f[ \left|H(j\Omega )\right|={1 \over {\sqrt {1+(\Omega /\Omega_{c})^{2N}}}}, ~~\forall~ \Omega=2 \pi f, ~ \Omega_{c}=2 \pi f_{c} \f]
 *  \f[ H(j\Omega )H(-j\Omega ) =\left|H(j\Omega )\right|^2\f]
 *
 *  <b>Teoría</b><br>
 *  Usa una transformación bi-linear 
 *  \f[ \Omega \leftarrow -j~ \frac{2}{T_d}~{\frac{z-1}{z+1}} \equiv  \frac{2}{T_d}~tan(\frac{w}{2}) \f]
 *
 *  Con una frecuencia de corte igual a \f$ \Omega_c \leftarrow \frac{2}{T_d}~tan(\frac{W_n \pi}{2}) \f$ 
 *  Por comodidad es escogido \f$  T_d \equiv  2~tan(\frac{W_n \pi}{2})\f$ 
 *
 *  \f[ \left( \frac{\Omega}{\Omega_{c}}\right)^{2n} \leftarrow (-1)^n \left(\frac{2}{T_d}\right)^{2n} \left(\frac{z-1}{z+1}\right)^{2n} \f]
 *
 *  \param[in] Order Orden del filtro Butterworth.
 *  \param[in] Wn Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital.
 *  \return Un puntero a un vector numerador o NULL en caso de error. 
 *  (ej. Order==0 Wn>1 o Wn<0)
 *  \ingroup PdsDfUtilsGroup
 */
PdsVector *pds_vector_new_butterworth_hnum(PdsDfNatural Order,PdsDfReal Wn)
{
	PdsDfNatural i,j,Nel;
	PdsRaReal a,b;
	PdsRaReal teta;
	PdsRaReal T;
	PdsVector *hnum=NULL;

	if(Order==0)	return NULL;
	if(Wn>1.0)	return NULL;
	if(Wn<0.0)	return NULL;

	Nel=Order+1;

	hnum=pds_vector_new(Nel);
	if(hnum==NULL)	return NULL;

	T=2.0*tan(Wn*M_PI/2.0);
	b=2.0/T;

	if( (Order%2)==1 )	//impar
	{
		hnum->V[0]=1.0;		hnum->V[1]=1.0;
	}
    else
    {
    	hnum->V[0]=1.0;	
    }

	for(i=0;(2*i)<(Order-1.0);i++)
	{	
		teta=M_PI*(Order+1.0+2.0*i)/(2.0*Order);

		a=-2.0*cos(teta);

		for(j=0;j<Nel;j++)
		{
			if(Order>=(2+j))	
			{
				hnum->V[Order-j]=	1.0*hnum->V[Order-j] +
						2.0*hnum->V[Order-j-1] +
						1.0*hnum->V[Order-j-2];
			}
			else if(Order>=(j+1))
			{
				hnum->V[Order-j]=	1.0*hnum->V[Order-j] +
							2.0*hnum->V[Order-j-1];
			}
			else if(Order>=j)
			{
				hnum->V[Order-j]=	1.0*hnum->V[Order-j];
			}
		}
	}

	return hnum;
}

/** \fn PdsVector *pds_vector_new_butterworth_hden(PdsDfNatural Order,PdsDfReal Wn)
 *  \brief Encuentra el denominador de un filtro Butterworth pasa bajo de orden Order. 
 *
 *  \f[ \left|H(j\Omega )\right|={1 \over {\sqrt {1+(\Omega /\Omega_{c})^{2N}}}}, ~~\forall~ \Omega=2 \pi f, ~ \Omega_{c}=2 \pi f_{c} \f]
 *  \f[ H(j\Omega )H(-j\Omega ) =\left|H(j\Omega )\right|^2\f]
 *
 *  <b>Teoría</b><br>
 *  Usa una transformación bi-linear 
 *  \f[ \Omega \leftarrow -j~ \frac{2}{T_d}~{\frac{z-1}{z+1}} \equiv  \frac{2}{T_d}~tan(\frac{w}{2}) \f]
 *
 *  Con una frecuencia de corte igual a \f$ \Omega_c \leftarrow \frac{2}{T_d}~tan(\frac{W_n \pi}{2}) \f$ 
 *  Por comodidad es escogido \f$  T_d \equiv  2~tan(\frac{W_n \pi}{2})\f$ 
 *
 *  \f[ \left( \frac{\Omega}{\Omega_{c}}\right)^{2n} \leftarrow (-1)^n \left(\frac{2}{T_d}\right)^{2n} \left(\frac{z-1}{z+1}\right)^{2n} \f]
 *
 *  \param[in] Order  Orden del filtro Butterworth.
 *  \param[in] Wn Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital.
 *  \return Un puntero a un vector numerador o NULL en caso de error. 
 *  (ej. N==0 Wn>1 o Wn<0)
 *  \ingroup PdsDfUtilsGroup
 */
PdsVector *pds_vector_new_butterworth_hden(PdsDfNatural Order,PdsDfReal Wn)
{
	PdsDfNatural i,j,Nel;
	PdsRaReal a,b;
	PdsRaReal teta;
	PdsRaReal T;
	PdsVector *hden=NULL;

	if(Order==0)	return NULL;
	if(Wn>1.0)	return NULL;
	if(Wn<0.0)	return NULL;

	Nel=Order+1;

	hden=pds_vector_new(Nel);
	if(hden==NULL)	return NULL;

	T=2.0*tan(Wn*M_PI/2.0);
	b=2.0/T;

	hden->V[0]=1.0;	

	if( (Order%2)==1 )	
	{
		hden->V[0]=1.0+b;	hden->V[1]=1.0-b;
	}

	for(i=0;(2*i)<(Order-1.0);i++)
	{	
		teta=M_PI*(Order+1.0+2.0*i)/(2.0*Order);

		a=-2.0*cos(teta);

		for(j=0;j<Nel;j++)
		{
			if(Order>=(2+j))	
			{
				hden->V[Order-j]=	(1.0+a*b+b*b)*hden->V[Order-j] +
						2.0*(1.0-b*b)*hden->V[Order-j-1] +
						(1.0-a*b+b*b)*hden->V[Order-j-2];
			}
			else if(Order>=(j+1))
			{
				hden->V[Order-j]=	(1.0+a*b+b*b)*hden->V[Order-j] +
						2.0*(1.0-b*b)*hden->V[Order-j-1];
			}
			else if(Order>=j)
			{
				hden->V[Order-j]=	(1.0+a*b+b*b)*hden->V[Order-j];
			}
		}
	}

	return hden;
}

/** \fn int pds_vector_iir_frequency_response(const PdsVector *hnum,const PdsVector *hden,PdsVector *H)
 *  \brief Encuentra el módulo de la respuesta en frecuencia, normalizada 
 *  de 0 a pi del filtro digital conformado por el numerador hnum y denominador hden.
 *
 *  El numero de puntos analizados en la respuesta en frecuencia, es el número 
 *  de elementos del vector H. Los datos de H serán sobrescritos.
 *
 *  <b>Teoría</b><br>
 *  \f[ H(w)=\left| \frac{hnum(Z=e^{j w})}{hden(Z=e^{j w})}\right| , ~\forall ~ 0 \leq w \leq \pi \f]
 *
 *  \f[ \Omega \leftarrow -j~ \frac{2}{T_d}~{\frac{z-1}{z+1}} \equiv  \frac{2}{T_d}~tan(\frac{w}{2}) \f]
 *  \param[in] hnum Coeficientes del numerador del filtro digital.
 *  \param[in] hden Coeficientes del denominador del filtro digital.
 *  \param[out] H Donde se guardará el módulo cuadrado de la respuesta en 
 *  frecuencia del filtro digital.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. hnum==NULL, hden==NULL o H==NULL)
 *  \ingroup PdsDfUtilsGroup
 */
int pds_vector_iir_frequency_response(const PdsVector *hnum,const PdsVector *hden,PdsVector *H)
{
	PdsDfNatural i,k,N;
	PdsDfReal NumReal;
	PdsDfReal NumImag;
	PdsDfReal DenReal;
	PdsDfReal DenImag;
	PdsDfReal Modulo2;
	PdsDfReal teta;

	if(hnum==NULL)	return FALSE;
	if(hden==NULL)	return FALSE;
	if(H==NULL)	return FALSE;

	N=H->Nel;

	for(i=0;i<N;i++)
	{
		teta=i*M_PI/N;

		for(NumReal=0.0,k=0;k<hnum->Nel;k++)
		{
			NumReal=NumReal+hnum->V[k]*cos(teta*k);
		}

		for(NumImag=0.0,k=0;k<hnum->Nel;k++)
		{
			NumImag=NumImag-hnum->V[k]*sin(teta*k);
		}

		for(DenReal=0.0,k=0;k<hden->Nel;k++)
		{
			DenReal=DenReal+hden->V[k]*cos(teta*k);
		}

		for(DenImag=0.0,k=0;k<hden->Nel;k++)
		{
			DenImag=DenImag-hden->V[k]*sin(teta*k);
		}

		Modulo2=(NumReal*NumReal+NumImag*NumImag)/(DenReal*DenReal+DenImag*DenImag);

		H->V[i]=sqrt(Modulo2);
	}

	return TRUE;
}


/** \fn int pds_vector_lowpass_rectangular(PdsVector *h,PdsDfReal Wn)
 *  \brief Encuentra los coeficientes de un filtro FIR pasa bajo. 
 *  Usando el método de la ventana, con ventana rectangular.
 *
 *  <b>Teoría:</b><br>
 *  \f[ h(n)=\frac{sin(W_n \pi (n-\frac{Order}{2}))}{\pi (n-\frac{Order}{2})}, ~\forall~n \in Z^+, 0 \leq n \leq Order \f]
 *
 *  \param[out] h Donde se guardarán los coeficientes del filtro digital.
 *  \param[in] Wn Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital.
 *  Wn indica una frecuencia de corte a Wn*Fs/2, donde Fs es la frecuencia de
 *  muestreo.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. h==NULL, Wn<0 o Wn>1)
 *  \ingroup PdsDfUtilsGroup
 */
int pds_vector_lowpass_rectangular(PdsVector *h,PdsDfReal Wn)
{
	PdsDfNatural i,Nel;
	PdsDfReal Order_over_2;

	if(h==NULL)	return FALSE;
	if(Wn>1.0)			return FALSE;
	if(Wn<0.0)			return FALSE;

	Nel=h->Nel;
	Order_over_2=(Nel-1.0)/2.0;

	for(i=0;i<Nel;i++)
	{
		if(i*1.0!=Order_over_2)
		h->V[i]=Wn*sin(Wn*M_PI*(i-Order_over_2))/(Wn*M_PI*(i-Order_over_2));
		else
		h->V[i]=Wn;
	}

	return TRUE;
}


/** \fn PdsVector *pds_vector_new_lowpass_rectangular(PdsDfNatural N,PdsDfReal Wn)
 *  \brief Devuelve un vector con los coeficientes de un filtro FIR pasa bajo. 
 *  Usando el método de la ventana, con ventana rectangular.
 *
 *  <b>Teoría:</b><br>
 *  \f[ h(n)=\frac{sin(W_n \pi (n-\frac{Order}{2}))}{\pi (n-\frac{Order}{2})}, ~\forall~n \in Z^+, 0 \leq n \leq Order \f]
 *
 *  \param[in] N Orden del filtro FIR.
 *  \param[in] Wn Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital. 
 *  Wn indica una frecuencia de corte a Wn*Fs/2, donde Fs es la frecuencia de
 *  muestreo.
 *  \return Un puntero a un vector de coeficientes de filtro FIR o NULL en caso de error. 
 *  (ej. N==0 Wn>1 o Wn<0)
 *  \ingroup PdsDfUtilsGroup
 */
PdsVector *pds_vector_new_lowpass_rectangular(PdsDfNatural N,PdsDfReal Wn)
{
	PdsDfNatural i,Nel;
	PdsDfReal Order_over_2;
	PdsVector *H=NULL;

	if(N==0)	return NULL;
	if(Wn>1.0)	return NULL;
	if(Wn<0.0)	return NULL;

	Nel=N+1;
	H=pds_vector_new(Nel);
	if(H==NULL)	return NULL;

	Order_over_2=(Nel-1.0)/2.0;

	for(i=0;i<Nel;i++)
	{
		if(i*1.0!=Order_over_2)
		H->V[i]=Wn*sin(Wn*M_PI*(i-Order_over_2))/(Wn*M_PI*(i-Order_over_2));
		else
		H->V[i]=Wn;
	}
	return H;
}


/** \fn int pds_vector_lowpass_hamming(PdsVector *h,PdsDfReal Wn)
 *  \brief Encuentra los coeficientes de un filtro FIR pasa bajo. 
 *  Usando el método de la ventana, con ventana de Hamming.
 *
 *  <b>Teoría:</b><br>
 *  \f[ h(n)=\frac{sin(W_n \pi (n-\frac{Order}{2}))}{\pi (n-\frac{Order}{2})}hamming(n), ~\forall~n \in Z^+, 0 \leq n \leq Order \f]
 *
 *  \param[out] h Donde se guardarán los coeficientes del filtro digital.
 *  \param[in] Wn Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital.
 *  Wn indica una frecuencia de corte a Wn*Fs/2, donde Fs es la frecuencia de
 *  muestreo.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. h==NULL, Wn<0 o Wn>1)
 *  \ingroup PdsDfUtilsGroup
 */
int pds_vector_lowpass_hamming(PdsVector *h,PdsDfReal Wn)
{
	PdsDfNatural i,Nel;
	PdsDfReal Order_over_2;

	if(h==NULL)	return FALSE;
	if(Wn>1.0)			return FALSE;
	if(Wn<0.0)			return FALSE;

	Nel=h->Nel;
	Order_over_2=(Nel-1.0)/2.0;

	for(i=0;i<Nel;i++)
	{
		if(i*1.0!=Order_over_2)
		h->V[i]=Wn*sin(Wn*M_PI*(i-Order_over_2))/(Wn*M_PI*(i-Order_over_2));
		else
		h->V[i]=Wn;

		h->V[i]=h->V[i]*(0.54-0.46*cos(2*M_PI*i/(Nel-1.0)));
	}

	return TRUE;
}


/** \fn PdsVector *pds_vector_new_lowpass_hamming(PdsDfNatural N,PdsDfReal Wn)
 *  \brief Devuelve un vector con los coeficientes de un filtro FIR pasa bajo. 
 *  Usando el método de la ventana, con ventana de Hamming.
 *
 *  <b>Teoría:</b><br>
 *  \f[ h(n)=\frac{sin(W_n \pi (n-\frac{Order}{2}))}{\pi (n-\frac{Order}{2})}hamming(n), ~\forall~n \in Z^+, 0 \leq n \leq Order \f]
 *
 *  \param[in] N Orden del filtro FIR.
 *  \param[in] Wn Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital. 
 *  Wn indica una frecuencia de corte a Wn*Fs/2, donde Fs es la frecuencia de
 *  muestreo.
 *  \return Un puntero a un vector de coeficientes de filtro FIR o NULL en caso de error. 
 *  (ej. N==0 Wn>1 o Wn<0)
 *  \ingroup PdsDfUtilsGroup
 */
PdsVector *pds_vector_new_lowpass_hamming(PdsDfNatural N,PdsDfReal Wn)
{
	PdsDfNatural i,Nel;
	PdsDfReal Order_over_2;
	PdsVector *H=NULL;

	if(N==0)	return NULL;
	if(Wn>1.0)	return NULL;
	if(Wn<0.0)	return NULL;

	Nel=N+1;
	H=pds_vector_new(Nel);
	if(H==NULL)	return NULL;

	Order_over_2=(Nel-1.0)/2.0;

	for(i=0;i<Nel;i++)
	{
		if(i*1.0!=Order_over_2)
		H->V[i]=Wn*sin(Wn*M_PI*(i-Order_over_2))/(Wn*M_PI*(i-Order_over_2));
		else
		H->V[i]=Wn;

		H->V[i]=H->V[i]*(0.54-0.46*cos(2*M_PI*i/(Nel-1.0)));
	}
	return H;
}


/** \fn int pds_vector_lowpass_hanning(PdsVector *h,PdsDfReal Wn)
 *  \brief Encuentra los coeficientes de un filtro FIR pasa bajo. 
 *  Usando el método de la ventana, con ventana de Hanning.
 *
 *  <b>Teoría:</b><br>
 *  \f[ h(n)=\frac{sin(W_n \pi (n-\frac{Order}{2}))}{\pi (n-\frac{Order}{2})}hanning(n), ~\forall~n \in Z^+, 0 \leq n \leq Order \f]
 *
 *  \param[out] h Donde se guardarán los coeficientes del filtro digital.
 *  \param[in] Wn Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital.
 *  Wn indica una frecuencia de corte a Wn*Fs/2, donde Fs es la frecuencia de
 *  muestreo.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. h==NULL, Wn<0 o Wn>1)
 *  \ingroup PdsDfUtilsGroup
 */
int pds_vector_lowpass_hanning(PdsVector *h,PdsDfReal Wn)
{
	PdsDfNatural i,Nel;
	PdsDfReal Order_over_2;

	if(h==NULL)	return FALSE;
	if(Wn>1.0)			return FALSE;
	if(Wn<0.0)			return FALSE;

	Nel=h->Nel;
	Order_over_2=(Nel-1.0)/2.0;

	for(i=0;i<Nel;i++)
	{
		if(i*1.0!=Order_over_2)
		h->V[i]=Wn*sin(Wn*M_PI*(i-Order_over_2))/(Wn*M_PI*(i-Order_over_2));
		else
		h->V[i]=Wn;

		h->V[i]=h->V[i]*(0.5-0.5*cos(2*M_PI*i/(Nel-1.0)));
	}

	return TRUE;
}

/** \fn PdsVector *pds_vector_new_lowpass_hanning(PdsDfNatural N,PdsDfReal Wn)
 *  \brief Devuelve un vector con los coeficientes de un filtro FIR pasa bajo. 
 *  Usando el método de la ventana, con ventana de Hanning.
 *
 *  <b>Teoría:</b><br>
 *  \f[ h(n)=\frac{sin(W_n \pi (n-\frac{Order}{2}))}{\pi (n-\frac{Order}{2})}hanning(n), ~\forall~n \in Z^+, 0 \leq n \leq Order \f]
 *
 *  \param[in] N Orden del filtro FIR.
 *  \param[in] Wn Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital. 
 *  Wn indica una frecuencia de corte a Wn*Fs/2, donde Fs es la frecuencia de
 *  muestreo.
 *  \return Un puntero a un vector de coeficientes de filtro FIR o NULL en caso de error. 
 *  (ej. N==0 Wn>1 o Wn<0)
 *  \ingroup PdsDfUtilsGroup
 */
PdsVector *pds_vector_new_lowpass_hanning(PdsDfNatural N,PdsDfReal Wn)
{
	PdsDfNatural i,Nel;
	PdsDfReal Order_over_2;
	PdsVector *H=NULL;

	if(N==0)	return NULL;
	if(Wn>1.0)	return NULL;
	if(Wn<0.0)	return NULL;

	Nel=N+1;
	H=pds_vector_new(Nel);
	if(H==NULL)	return NULL;

	Order_over_2=(Nel-1.0)/2.0;

	for(i=0;i<Nel;i++)
	{
		if(i*1.0!=Order_over_2)
		H->V[i]=Wn*sin(Wn*M_PI*(i-Order_over_2))/(Wn*M_PI*(i-Order_over_2));
		else
		H->V[i]=Wn;

		H->V[i]=H->V[i]*(0.5-0.5*cos(2*M_PI*i/(Nel-1.0)));
	}
	return H;
}

/** \fn int pds_vector_fir_frequency_response(const PdsVector *h,PdsVector *H)
 *  \brief Encuentra el módulo de la respuesta en frecuencia, normalizada 
 *  de 0 a pi del filtro digital conformado por el numerador h.
 *
 *  <b>Teoría</b><br>
 *  \f[ H(w)=\left|h(Z=e^{j w})\right|, ~\forall ~ 0 \leq w \leq \pi \f]
 *
 *  \param[in] h Coeficientes del filtro digital.
 *  \param[out] H Donde se guardará el módulo cuadrado de la respuesta en 
 *  frecuencia del filtro digital.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. h==NULL o H==NULL)
 *  \ingroup PdsDfUtilsGroup
 */
int pds_vector_fir_frequency_response(const PdsVector *h,PdsVector *H)
{
	PdsDfNatural i,k,N;
	PdsDfReal NumReal;
	PdsDfReal NumImag;
	PdsDfReal Modulo2;
	PdsDfReal teta;

	if(h==NULL)	return FALSE;
	if(H==NULL)	return FALSE;

	N=H->Nel;

	for(i=0;i<N;i++)
	{
		teta=i*M_PI/(N-1);

		for(NumReal=0.0,k=0;k<h->Nel;k++)
		{
			NumReal=NumReal+h->V[k]*cos(teta*k);
		}

		for(NumImag=0.0,k=0;k<h->Nel;k++)
		{
			NumImag=NumImag-h->V[k]*sin(teta*k);
		}

		Modulo2=(NumReal*NumReal+NumImag*NumImag);

		H->V[i]=sqrt(Modulo2);
	}

	return TRUE;
}



/** \fn int pds_vector_lowpass_to_highpass(PdsVector *H)
 *  \brief Carga un vector con \f$H(-Z)\f$. Tranforma un filtro pasa bajo con un 
 *  corte en Wc a un filtro pasa alto con un corte en PI-Wc. 
 *  
 *  En realidad lo que hace es un corrimiento de PI de la
 *  respuesta en frecuencia. Que es equivalente a cambiar el valor de Z por -Z.
 *  \param[in,out] H Donde se encuentra el filtro pasa bajo y donde se guardará 
 *  el filtro pasa alto.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. H==NULL)
 *  \ingroup PdsDfUtilsGroup
 */
int pds_vector_lowpass_to_highpass(PdsVector *H)
{
	PdsDfNatural i;

	if(H==NULL)	return FALSE;

	for(i=1;i<H->Nel;i=i+2)
	{
		H->V[i]=-H->V[i];
	}

	return TRUE;
}


/** \fn PdsVector *pds_vector_new_lowpass_to_highpass(const PdsVector *H)
 *  \brief Retorna un vector con \f$H(-Z)\f$. Tranforma un filtro pasa bajo con 
 *  una frecuencia de corte en Wc a un filtro pasa alto con un corte en PI-Wc. 
 *  
 *  En realidad lo que hace es un corrimiento de PI de la
 *  respuesta en frecuencia. Que es equivalente a cambiar el valor de Z por -Z.
 *  \param[in] H Donde se encuentra el filtro pasa bajo y donde se guardará 
 *  el filtro pasa alto.
 *  \return Si todo fue bien retorna un puntero a un vector con el filtro pasa 
 *  alto, o NULL si no. (ej. H==NULL)
 *  \ingroup PdsDfUtilsGroup
 */
PdsVector *pds_vector_new_lowpass_to_highpass(const PdsVector *H)
{
	PdsVector *Hout=NULL;
    int id;

	if(H==NULL)	return FALSE;

    Hout=pds_vector_new_vector(H);
    if(Hout==NULL)  return NULL;

    id=pds_vector_lowpass_to_highpass(Hout);
    if(id==FALSE)
    {
        pds_vector_free(Hout);
        return NULL;
    }

	return Hout;
}


/** \fn PdsVector *pds_vector_new_lowpass_to_bandpass(const PdsVector *H)
 *  \brief Retorna un vector con \f$H(-Z^2)\f$. Crea a partir de un filtro pasa 
 *  bajoo con un corte en Wc un filtro pasa banda centrado en (PI/2) con ancho de 
 *  banda Wc. 
 *
 *  En realidad lo que hace es comprimir
 *  toda respuesta en frecuencia de [-PI,PI] a [-PI/2,PI/2] y dislocarla a PI/2 y
 *  -PI/2. Que es equivalente a cambiar el valor de Z por -Z^2. Conociendo que Nel
 *  es el numero de elementos de H, entonces el vector entregado tendrá 2N-1 elementos.
 *  \param[in] H Donde se encuentra el filtro pasa bajo.
 *  \return Si todo fue bien retorna un puntero a un vector con el filtro pasa 
 *  banda, o NULL si no. (ej. H==NULL)
 *  \ingroup PdsDfUtilsGroup
 */
PdsVector *pds_vector_new_lowpass_to_bandpass(const PdsVector *H)
{
	PdsDfNatural i;

	PdsVector *Hband=NULL;

	if(H==NULL)	return NULL;
	
	Hband=pds_vector_new(2*H->Nel-1);

	for(i=0;i<H->Nel;i++)
	{
		if(i%2==0)	Hband->V[2*i]=H->V[i];
		else		Hband->V[2*i]=-H->V[i];
	}

	return Hband;
}


/** \fn PdsVector *pds_vector_new_lowpass_to_bandreject(const PdsVector *H)
 *  \brief Retorna un vector con \f$H(Z^2)\f$. Crea a partir de un filtro pasa bajo 
 *  con un corte en Wc un filtro rechaza banda centrado en (PI/2) con ancho de 
 *  rechaza banda PI-Wc. 
 *
 *  En realidad lo que hace es comprimir toda respuesta en frecuencia de [-PI,PI] 
 *  a [-PI/2,PI/2] y dislocarla a 0, PI y -PI. Que es equivalente a cambiar el 
 *  valor de Z por Z^2. Conociendo que Nel es el número de elementos de H, entonces 
 *  el vector entregado tendrá 2N-1 elementos.
 *  \param[in] H Donde se encuentra el filtro pasa bajo.
 *  \return Si todo fue bien retorna un puntero a un vector con el filtro rechaza 
 *  banda, o NULL si no. (ej. H==NULL)
 *  \ingroup PdsDfUtilsGroup
 */
PdsVector *pds_vector_new_lowpass_to_bandreject(const PdsVector *H)
{
	PdsDfNatural i;

	PdsVector *Hreject=NULL;

	if(H==NULL)	return NULL;
	
	Hreject=pds_vector_new(2*H->Nel-1);

	for(i=0;i<H->Nel;i++)
	{
		Hreject->V[2*i]=H->V[i];
	}

	return Hreject;
}

