/*
 * pdslpfir.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdslpfir.h
 *  \author Fernando Pujaico Rivera
 *  \date 21-05-2011
 *  \brief Funciones que trabajan con estructuras Filtro Linear-Phase FIR.
 *  
 *  <br>
 *  \f[ h[n]=h[N-1-n]\f].
 *  \image html lpfir1.png "Filtro Linear-Phase FIR, h con M=3 elementos, y  N=6 par"
 *  \f[ M=N/2\f].
 *  \f[ y[n]=\sum_{i=0}^{\frac{N}{2}-1}h_i(x[n-i]+x[n-(N-1-i)]) \f].
 *  \f[ \frac{Y[z]}{X[z]} = H[z] = \sum_{i=0}^{M-1}h_i(z^{-i}+z^{-(N-1-i)}) \f].
 *  \image html lpfir2.png "Filtro Linear-Phase FIR, h con M=3 elementos, y  N=5 impar"
 *  \f[ M=(N+1)/2\f].
 *  \f[ y[n]=\sum_{i=0}^{M-2}h_i(x[n-i]+x[n-(N-1-i)]) + h_{M-1} x[n-(M-1)] \f].
 *  \f[ \frac{Y[z]}{X[z]} = H[z] = \sum_{i=0}^{M-2}h_i(z^{-i}+z^{-(N-1-i)}) + h_{M-1}z^{-(M-1)}\f].
 */

#ifndef __PDSLPLPFIR_H__
#define __PDSLPLPFIR_H__


#ifdef __cplusplus
extern "C" {
#endif 

#include <pds/pdsdfglobal.h>
#include <pds/pdsvector.h>

#include <stdio.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

/** \defgroup PdsLpFirGroup Funciones del módulo PdsLpFir.
 *
 *  <br>
 *  \f[ h[n]=h[N-1-n]\f].
 *  \image html lpfir1.png "Filtro Linear-Phase FIR, h con M=3 elementos, y  N=6 par"
 *  \f[ M=N/2\f].
 *  \f[ y[n]=\sum_{i=0}^{\frac{N}{2}-1}h_i(x[n-i]+x[n-(N-1-i)]) \f].
 *  \f[ \frac{Y[z]}{X[z]} = H[z] = \sum_{i=0}^{M-1}h_i(z^{-i}+z^{-(N-1-i)}) \f].
 *  \image html lpfir2.png "Filtro Linear-Phase FIR, h con M=3 elementos, y  N=5 impar"
 *  \f[ M=(N+1)/2\f].
 *  \f[ y[n]=\sum_{i=0}^{M-2}h_i(x[n-i]+x[n-(N-1-i)]) + h_{M-1} x[n-(M-1)] \f].
 *  \f[ \frac{Y[z]}{X[z]} = H[z] = \sum_{i=0}^{M-2}h_i(z^{-i}+z^{-(N-1-i)}) + h_{M-1}z^{-(M-1)}\f].
 * @{
 */


/*! \struct PdsLpFir
 *  \brief Una estructura tipo  PdsLpFir .
 *
 *  Esta estructura genera un filtro Linear-Phase FIR.
 *  Para usar incluir pds/pdsdf.h.
 *  <br>
 *  \f[ h[n]=h[N-1-n]\f].
 *  \image html lpfir1.png "Filtro Linear-Phase FIR, h con M=3 elementos, y  N=6 par"
 *  \f[ M=N/2\f].
 *  \f[ y[n]=\sum_{i=0}^{\frac{N}{2}-1}h_i(x[n-i]+x[n-(N-1-i)]) \f].
 *  \f[ \frac{Y[z]}{X[z]} = H[z] = \sum_{i=0}^{M-1}h_i(z^{-i}+z^{-(N-1-i)}) \f].
 *  \image html lpfir2.png "Filtro Linear-Phase FIR, h con M=3 elementos, y  N=5 impar"
 *  \f[ M=(N+1)/2\f].
 *  \f[ y[n]=\sum_{i=0}^{M-2}h_i(x[n-i]+x[n-(N-1-i)]) + h_{M-1} x[n-(M-1)] \f].
 *  \f[ \frac{Y[z]}{X[z]} = H[z] = \sum_{i=0}^{M-2}h_i(z^{-i}+z^{-(N-1-i)}) + h_{M-1}z^{-(M-1)}\f].
 *  \ingroup PdsLpFirGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! Es el vector de elementos h_i. */
	PdsVector *h;
	/*! Si el vector H tiene una cantidad de elementos par o impar. */
	PdsDfNatural Par;
	/*! El número de elementos x_i. */
	PdsDfNatural N;
	/*! Vector de elemnotos x[n-i]. */
	PdsVector *X;
}PdsLpFir;

/** \fn PdsLpFir *pds_lpfir_new(const PdsVector *h,PdsDfNatural par)
 *  \brief Crea un filtro Linear-Phase FIR con parametros h.
 * 
 *  El vector h entregado como parámetro a esta función nunca es modificado.
 *  \param[in] h Vector para los coeficientes de h_i*x[n-i].
 *  \param[in] par Indica si el vector H total tiene una cantidad de elementos par.
 *  \return Un puntero a una estructura de tipo PdsLpFir. O NULL en caso de error.
 *  \ingroup PdsLpFirGroup
 */
PdsLpFir *pds_lpfir_new(const PdsVector *h,PdsDfNatural par);


/** \fn int pds_lpfir_evaluate_value(PdsLpFir *LPFIR,PdsDfReal x,PdsDfReal *y)
 *  \brief Evalúa el filtro Linear-Phase FIR con el valor de entrada x, el resultado
 *  es cargado en y.
 * 
 *  En cada iteración se realizan corrimientos para el vector LPFIR->X.
 *  \param[in,out] LPFIR El filtro Linear-Phase FIR a usar.
 *  \param[in] x El valor de entrada del filtro.
 *  \param[out] y El valor de salida del filtro.
 *  \return TRUE si todo fue bien o FALSE si no;
 *  \ingroup PdsLpFirGroup
 */
int pds_lpfir_evaluate_value(PdsLpFir *LPFIR,PdsDfReal x,PdsDfReal *y);


/** \fn int pds_lpfir_evaluate_vector(PdsLpFir *LPFIR,const PdsVector *x,PdsVector *y)
 *  \brief Evalúa el filtro Linear-Phase FIR con el vector de entrada x, el resultado
 *  es cargado en el vector y.
 * 
 *  Se recomienda usar esta función solo cuando x es mucho mayor que LPFIR->h.
 *  Solo se realizan corrimientos de LPFIR->X al inicio y al final del vector x
 *  en los casos intermediarios se aprovecha tener el vector, y no se efectuan
 *  corrimientos, por lo que es un poco mas rápido que pds_lpfir_evaluate_value
 *  cuando x es mucho mayo que LPFIR->h.
 *  \param[in,out] LPFIR El filtro Linear-Phase FIR a usar.
 *  \param[in] x El vector de entrada del filtro.
 *  \param[out] y El vector de salida del filtro.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsLpFirGroup
 */
int pds_lpfir_evaluate_vector(PdsLpFir *LPFIR,const PdsVector *x,PdsVector *y);


/** \fn void pds_lpfir_free(PdsLpFir *LPFIR)
 *  \brief Libera el filtro de tipo PdsLpFir.
 *  \param[in] LPFIR El filtro a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsLpFirGroup
 */
void pds_lpfir_free(PdsLpFir *LPFIR);


/** \fn void pds_lpfir_destroy(PdsLpFir **LPFIR)
 *  \brief Libera el filtro de tipo PdsLpFir. y carga la variable con NULL.
 *  \param[in] LPFIR El filtro a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsLpFirGroup
 */
void pds_lpfir_destroy(PdsLpFir **LPFIR);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif	/* __PDSLPLPFIR_H__ */ 

