/*
 * pdsdfutils.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsdfutils.h
 *  \author Fernando Pujaico Rivera
 *  \date 18-01-2012
 *  \brief Funciones que necesita la biblioteca.
 *  
 */

#ifndef __PDSDFUTILS_H__
#define __PDSDFUTILS_H__

#ifdef __cplusplus
extern "C" {
#endif 


#include <pds/pdsdfglobal.h>
#include <pds/pdsvector.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

/** \defgroup PdsDfUtilsGroup Funciones útiles de la biblioteca.
 *  \brief Funciones útiles de la biblioteca.
 *  
 * @{
 */

/** \fn int pds_df_butterworth(PdsVector *hnum,PdsVector *hden,PdsDfReal factor)
 *  \brief Encuentra el numerador y denominador de un filtro Butterworth pasa bajo. 
 *  Usando una transformación bi-linear y frecuencia de corte igual a Wc=factor*pi. 
 *  (\f$ W_c=factor \pi \f$)
 *  \param[out] hnum Coeficientes del numerador del filtro digital.
 *  \param[out] hden Coeficientes del denominador del filtro digital.
 *  \param[in] factor Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. hnum==NULL, hden==NULL, factor<0 o factor>1)
 *  \ingroup PdsDfUtilsGroup
 */
int pds_df_butterworth(PdsVector *hnum,PdsVector *hden,PdsDfReal factor);


/** \fn PdsVector *pds_df_butterworth_hnum_new(PdsDfNatural N,PdsDfReal factor)
 *  \brief Encuentra el numerador de un filtro Butterworth pasa bajo de orden N. 
 *  Usando una transformación bi-linear y frecuencia de corte igual a Wc=factor*pi. 
 *  (\f$ W_c=factor \pi \f$)
 *  \param[in] N Orden del filtro Butterworth.
 *  \param[in] factor Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital.
 *  \return Un puntero a un vector numerador o NULL en caso de error. 
 *  (ej. N==0 factor>1 o factor<0)
 *  \ingroup PdsDfUtilsGroup
 */
PdsVector *pds_df_butterworth_hnum_new(PdsDfNatural N,PdsDfReal factor);


/** \fn PdsVector *pds_df_butterworth_hden_new(PdsDfNatural N,PdsDfReal factor)
 *  \brief Encuentra el denominador de un filtro Butterworth pasa bajo de orden N. 
 *  Usando una transformación bi-linear y frecuencia de corte igual a Wc=factor*pi. 
 *  (\f$ W_c=factor \pi \f$)
 *  \param[in] N Orden del filtro Butterworth.
 *  \param[in] factor Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital.
 *  \return Un puntero a un vector numerador o NULL en caso de error. 
 *  (ej. N==0 factor>1 o factor<0)
 *  \ingroup PdsDfUtilsGroup
 */
PdsVector *pds_df_butterworth_hden_new(PdsDfNatural N,PdsDfReal factor);


/** \fn int pds_df_frequency_response_iir(const PdsVector *hnum,const PdsVector *hden,PdsVector *H)
 *  \brief Encuentra el módulo de la respuesta en frecuencia, normalizada 
 *  de 0 a 2*pi del filtro digital conformado por el numerador hnum y denominador hden.
 *  \param[in] hnum Coeficientes del numerador del filtro digital.
 *  \param[in] hden Coeficientes del denominador del filtro digital.
 *  \param[out] H Donde se guardará el módulo cuadrado de la respuesta en 
 *  frecuencia del filtro digital.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. hnum==NULL, hden==NULL o H==NULL)
 *  \ingroup PdsDfUtilsGroup
 */
int pds_df_frequency_response_iir(const PdsVector *hnum,const PdsVector *hden,PdsVector *H);


/** \fn int pds_df_rectangular_window(PdsVector *h,PdsDfReal factor)
 *  \brief Encuentra los coeficientes de un filtro FIR pasa bajo. 
 *  Usando el método de la ventana, con ventana rectangular.
 *  \param[out] h Donde se guardarán los coeficientes del filtro digital.
 *  \param[in] factor Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. h==NULL, factor<0 o factor>1)
 *  \ingroup PdsDfUtilsGroup
 */
int pds_df_rectangular_window(PdsVector *h,PdsDfReal factor);


/** \fn PdsVector *pds_df_butterworth_hnum_new(PdsDfNatural N,PdsDfReal factor)
 *  \brief Devuelve los coeficientes de un filtro FIR pasa bajo. 
 *  Usando el método de la ventana, con ventana rectangular
 *  \param[in] N Orden del filtro FIR.
 *  \param[in] factor Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital.
 *  \return Un puntero a un vector de coeficientes de filtro FIR o NULL en caso de error. 
 *  (ej. N==0 factor>1 o factor<0)
 *  \ingroup PdsDfUtilsGroup
 */
PdsVector *pds_df_rectangular_window_new(PdsDfNatural N,PdsDfReal factor);


/** \fn int pds_df_hamming_window(PdsVector *h,PdsDfReal factor)
 *  \brief Encuentra los coeficientes de un filtro FIR pasa bajo. 
 *  Usando el método de la ventana, con ventana de Hamming.
 *  \param[out] h Donde se guardarán los coeficientes del filtro digital.
 *  \param[in] factor Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. h==NULL, factor<0 o factor>1)
 *  \ingroup PdsDfUtilsGroup
 */
int pds_df_hamming_window(PdsVector *h,PdsDfReal factor);


/** \fn PdsVector *pds_df_hamming_window_new(PdsDfNatural N,PdsDfReal factor)
 *  \brief Devuelve los coeficientes de un filtro FIR pasa bajo. 
 *  Usando el método de la ventana, con ventana de Hamming.
 *  \param[in] N Orden del filtro FIR.
 *  \param[in] factor Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital.
 *  \return Un puntero a un vector de coeficientes de filtro FIR o NULL en caso de error. 
 *  (ej. N==0 factor>1 o factor<0)
 *  \ingroup PdsDfUtilsGroup
 */
PdsVector *pds_df_hamming_window_new(PdsDfNatural N,PdsDfReal factor);


/** \fn int pds_df_hanning_window(PdsVector *h,PdsDfReal factor)
 *  \brief Encuentra los coeficientes de un filtro FIR pasa bajo. 
 *  Usando el método de la ventana, con ventana de Hanning.
 *  \param[out] h Donde se guardarán los coeficientes del filtro digital.
 *  \param[in] factor Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. h==NULL, factor<0 o factor>1)
 *  \ingroup PdsDfUtilsGroup
 */
int pds_df_hanning_window(PdsVector *h,PdsDfReal factor);


/** \fn PdsVector *pds_df_hanning_window_new(PdsDfNatural N,PdsDfReal factor)
 *  \brief Devuelve los coeficientes de un filtro FIR pasa bajo. 
 *  Usando el método de la ventana, con ventana de Hanning.
 *  \param[in] N Orden del filtro FIR.
 *  \param[in] factor Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital.
 *  \return Un puntero a un vector de coeficientes de filtro FIR o NULL en caso de error. 
 *  (ej. N==0 factor>1 o factor<0)
 *  \ingroup PdsDfUtilsGroup
 */
PdsVector *pds_df_hanning_window_new(PdsDfNatural N,PdsDfReal factor);


/** \fn int pds_df_frequency_response_fir(const PdsVector *h,PdsVector *H)
 *  \brief Encuentra el módulo de la respuesta en frecuencia, normalizada 
 *  de 0 a 2*pi del filtro digital conformado por el numerador h.
 *  \param[in] h Coeficientes del filtro digital.
 *  \param[out] H Donde se guardará el módulo cuadrado de la respuesta en 
 *  frecuencia del filtro digital.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. h==NULL o H==NULL)
 *  \ingroup PdsDfUtilsGroup
 */
int pds_df_frequency_response_fir(const PdsVector *h,PdsVector *H);


/** \fn int pds_df_lowpass_to_highpass(PdsVector *H)
 *  \brief Tranforma un filtro pasa bajo con un corte en Wc a un filtro pasa alto
 *  con un corte en PI-Wc. En realidad lo que hace es un corrimiento de PI de la
 *  respuesta en frecuencia. Que es equivalente a cambiar el valor de Z por -Z.
 *  \param[in,out] H Donde se encuentra el filtro pasa bajo y donde se guardará 
 *  el filtro pasa alto.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. H==NULL)
 *  \ingroup PdsDfUtilsGroup
 */
int pds_df_lowpass_to_highpass(PdsVector *H);



/** \fn PdsVector *pds_df_lowpass_to_bandpass(const PdsVector *H)
 *  \brief Crea a partir de un filtro pasa bajo con un corte en Wc un filtro pasa banda
 *  centrado en (PI/2) con ancho de banda Wc. 

 *  En realidad lo que hace es comprimir
 *  toda respuesta en frecuencia de [-PI,PI] a [-PI/2,PI/2] y dislocarla a PI/2 y
 *  -PI/2. Que es equivalente a cambiar el valor de Z por -Z^2. Conociendo que Nel
 *  es el numero de elementos de H, entonces el vector entregado tendrá 2N-1 elementos.
 *  \param[in] H Donde se encuentra el filtro pasa bajo.
 *  \return Si todo fue bien retorna un puntero a un vector con el filtro pasa 
 *  banda, o NULL si no. (ej. H==NULL)
 *  \ingroup PdsDfUtilsGroup
 */
PdsVector *pds_df_lowpass_to_bandpass(const PdsVector *H);


/** \fn PdsVector *pds_df_lowpass_to_bandreject(const PdsVector *H)
 *  \brief Crea a partir de un filtro pasa bajo con un corte en Wc un filtro rechaza banda
 *  centrado en (PI/2) con ancho de rechaza banda PI-Wc. 

 *  En realidad lo que hace es comprimir
 *  toda respuesta en frecuencia de [-PI,PI] a [-PI/2,PI/2] y dislocarla a 0, PI y
 *  -PI. Que es equivalente a cambiar el valor de Z por Z^2. Conociendo que Nel
 *  es el numero de elementos de H, entonces el vector entregado tendrá 2N-1 elementos.
 *  \param[in] H Donde se encuentra el filtro pasa bajo.
 *  \return Si todo fue bien retorna un puntero a un vector con el filtro rechaza 
 *  banda, o NULL si no. (ej. H==NULL)
 *  \ingroup PdsDfUtilsGroup
 */
PdsVector *pds_df_lowpass_to_bandreject(const PdsVector *H);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif


