/*
 * pdsdfutils.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <math.h>
#include <pds/pdsvector.h>
#include <pds/pdsdfutils.h>

/** \fn int pds_df_butterworth(PdsVector *hnum,PdsVector *hden,PdsDfReal factor)
 *  \brief Encuentra el numerador y denominador de un filtro Butterworth pasa bajo. 
 *  Usando una transformación bi-linear y frecuencia de corte igual a Wc=factor*pi. 
 *  (\f$ W_c=factor \pi \f$)
 *  \param[out] hnum Coeficientes del numerador del filtro digital.
 *  \param[out] hden Coeficientes del denominador del filtro digital.
 *  \param[in] factor Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. hnum==NULL, hden==NULL, factor<0 o factor>1)
 *  \ingroup PdsDfUtilsGroup
 */
int pds_df_butterworth(PdsVector *hnum,PdsVector *hden,PdsDfReal factor)
{
	PdsDfNatural i,j,N,Nel;
	PdsRaReal a,b;
	PdsRaReal teta;
	PdsRaReal T;

	if(hden->Nel!=hnum->Nel)	return FALSE;
	if(hden->Nel<2)			return FALSE;
	if(factor>1.0)			return FALSE;
	if(factor<0.0)			return FALSE;

	Nel=hden->Nel;
	N  =Nel-1;

	T=2.0*tan(factor*M_PI/2.0);
	b=2.0/T;

	pds_vector_init_value(hnum,0.0);
	pds_vector_init_value(hden,0.0);

	hnum->V[0]=1.0;	
	hden->V[0]=1.0;	

	if( (N%2)==1 )	
	{
		hnum->V[0]=1.0;		hnum->V[1]=1.0;
		hden->V[0]=1.0+b;	hden->V[1]=1.0-b;
	}

	for(i=0;(2*i)<(N-1.0);i++)
	{	
		teta=M_PI*(N+1.0+2.0*i)/(2.0*N);

		a=-2.0*cos(teta);

		for(j=0;j<Nel;j++)
		{
			if(N>=(2+j))	
			{
				hnum->V[N-j]=	1.0*hnum->V[N-j] +
						2.0*hnum->V[N-j-1] +
						1.0*hnum->V[N-j-2];

				hden->V[N-j]=	(1.0+a*b+b*b)*hden->V[N-j] +
						2.0*(1.0-b*b)*hden->V[N-j-1] +
						(1.0-a*b+b*b)*hden->V[N-j-2];
			}
			else 
			{
				if(N>=(j+1))
				{
					hnum->V[N-j]=	1.0*hnum->V[N-j] +
							2.0*hnum->V[N-j-1];
					hden->V[N-j]=	(1.0+a*b+b*b)*hden->V[N-j] +
							2.0*(1.0-b*b)*hden->V[N-j-1];
				}
				else
				{
					if(N>=j)
					{
						hnum->V[N-j]=	1.0*hnum->V[N-j];
						hden->V[N-j]=	(1.0+a*b+b*b)*hden->V[N-j];
					}
				}
			}
		}
	}

	pds_vector_mul_value(hnum,1/hden->V[0]);
	pds_vector_mul_value(hden,1/hden->V[0]);
	return TRUE;
}


/** \fn PdsVector *pds_df_butterworth_hnum_new(PdsDfNatural N,PdsDfReal factor)
 *  \brief Encuentra el numerador de un filtro Butterworth pasa bajo de orden N. 
 *  Usando una transformación bi-linear y frecuencia de corte igual a Wc=factor*pi. 
 *  (\f$ W_c=factor \pi \f$)
 *  \param[in] N Orden del filtro Butterworth.
 *  \param[in] factor Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital.
 *  \return Un puntero a un vector numerador o NULL en caso de error. 
 *  (ej. N==0 factor>1 o factor<0)
 *  \ingroup PdsDfUtilsGroup
 */
PdsVector *pds_df_butterworth_hnum_new(PdsDfNatural N,PdsDfReal factor)
{
	PdsDfNatural i,j,Nel;
	PdsRaReal a,b;
	PdsRaReal teta;
	PdsRaReal T;
	PdsVector *hnum=NULL;

	if(N==0)	return NULL;
	if(factor>1.0)	return NULL;
	if(factor<0.0)	return NULL;

	Nel=N+1;

	hnum=pds_vector_new(Nel);
	if(hnum==NULL)	return NULL;

	T=2.0*tan(factor*M_PI/2.0);
	b=2.0/T;

	hnum->V[0]=1.0;	

	if( (N%2)==1 )	
	{
		hnum->V[0]=1.0;		hnum->V[1]=1.0;
	}

	for(i=0;(2*i)<(N-1.0);i++)
	{	
		teta=M_PI*(N+1.0+2.0*i)/(2.0*N);

		a=-2.0*cos(teta);

		for(j=0;j<Nel;j++)
		{
			if(N>=(2+j))	
			{
				hnum->V[N-j]=	1.0*hnum->V[N-j] +
						2.0*hnum->V[N-j-1] +
						1.0*hnum->V[N-j-2];
			}
			else 
			{
				if(N>=(j+1))
				{
					hnum->V[N-j]=	1.0*hnum->V[N-j] +
							2.0*hnum->V[N-j-1];
				}
				else
				{
					if(N>=j)
					{
						hnum->V[N-j]=	1.0*hnum->V[N-j];
					}
				}
			}
		}
	}

	return hnum;
}

/** \fn PdsVector *pds_df_butterworth_hden_new(PdsDfNatural N,PdsDfReal factor)
 *  \brief Encuentra el denominador de un filtro Butterworth pasa bajo de orden N. 
 *  Usando una transformación bi-linear y frecuencia de corte igual a Wc=factor*pi. 
 *  (\f$ W_c=factor \pi \f$)
 *  \param[in] N Orden del filtro Butterworth.
 *  \param[in] factor Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital.
 *  \return Un puntero a un vector numerador o NULL en caso de error. 
 *  (ej. N==0 factor>1 o factor<0)
 *  \ingroup PdsDfUtilsGroup
 */
PdsVector *pds_df_butterworth_hden_new(PdsDfNatural N,PdsDfReal factor)
{
	PdsDfNatural i,j,Nel;
	PdsRaReal a,b;
	PdsRaReal teta;
	PdsRaReal T;
	PdsVector *hden=NULL;

	if(N==0)	return NULL;
	if(factor>1.0)	return NULL;
	if(factor<0.0)	return NULL;

	Nel=N+1;

	hden=pds_vector_new(Nel);
	if(hden==NULL)	return NULL;

	T=2.0*tan(factor*M_PI/2.0);
	b=2.0/T;

	hden->V[0]=1.0;	

	if( (N%2)==1 )	
	{
		hden->V[0]=1.0+b;	hden->V[1]=1.0-b;
	}

	for(i=0;(2*i)<(N-1.0);i++)
	{	
		teta=M_PI*(N+1.0+2.0*i)/(2.0*N);

		a=-2.0*cos(teta);

		for(j=0;j<Nel;j++)
		{
			if(N>=(2+j))	
			{
				hden->V[N-j]=	(1.0+a*b+b*b)*hden->V[N-j] +
						2.0*(1.0-b*b)*hden->V[N-j-1] +
						(1.0-a*b+b*b)*hden->V[N-j-2];
			}
			else 
			{
				if(N>=(j+1))
				{
					hden->V[N-j]=	(1.0+a*b+b*b)*hden->V[N-j] +
							2.0*(1.0-b*b)*hden->V[N-j-1];
				}
				else
				{
					if(N>=j)
					{
						hden->V[N-j]=	(1.0+a*b+b*b)*hden->V[N-j];
					}
				}
			}
		}
	}

	return hden;
}

/** \fn int pds_df_frequency_response_iir(const PdsVector *hnum,const PdsVector *hden,PdsVector *H)
 *  \brief Encuentra el módulo de la respuesta en frecuencia, normalizada 
 *  de 0 a 2*pi del filtro digital conformado por el numerador hnum y denominador hden.
 *  \param[in] hnum Coeficientes del numerador del filtro digital.
 *  \param[in] hden Coeficientes del denominador del filtro digital.
 *  \param[out] H Donde se guardará el módulo cuadrado de la respuesta en 
 *  frecuencia del filtro digital.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. hnum==NULL, hden==NULL o H==NULL)
 *  \ingroup PdsDfUtilsGroup
 */
int pds_df_frequency_response_iir(const PdsVector *hnum,const PdsVector *hden,PdsVector *H)
{
	PdsDfNatural i,k,N;
	PdsDfReal NumReal;
	PdsDfReal NumImag;
	PdsDfReal DenReal;
	PdsDfReal DenImag;
	PdsDfReal Modulo2;
	PdsDfReal teta;

	if(hnum==NULL)	return FALSE;
	if(hden==NULL)	return FALSE;
	if(H==NULL)	return FALSE;

	N=H->Nel;

	for(i=0;i<N;i++)
	{
		teta=i*2.0*M_PI/N;

		for(NumReal=0.0,k=0;k<hnum->Nel;k++)
		{
			NumReal=NumReal+hnum->V[k]*cos(teta*k);
		}

		for(NumImag=0.0,k=0;k<hnum->Nel;k++)
		{
			NumImag=NumImag-hnum->V[k]*sin(teta*k);
		}

		for(DenReal=0.0,k=0;k<hden->Nel;k++)
		{
			DenReal=DenReal+hden->V[k]*cos(teta*k);
		}

		for(DenImag=0.0,k=0;k<hden->Nel;k++)
		{
			DenImag=DenImag-hden->V[k]*sin(teta*k);
		}

		Modulo2=(NumReal*NumReal+NumImag*NumImag)/(DenReal*DenReal+DenImag*DenImag);

		H->V[i]=sqrt(Modulo2);
	}

	return TRUE;
}

/** \fn int pds_df_rectangular_window(PdsVector *h,PdsDfReal factor)
 *  \brief Encuentra los coeficientes de un filtro FIR pasa bajo. 
 *  Usando el método de la ventana, con ventana rectangular.
 *  \param[out] h Donde se guardarán los coeficientes del filtro digital.
 *  \param[in] factor Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. h==NULL, factor<0 o factor>1)
 *  \ingroup PdsDfUtilsGroup
 */
int pds_df_rectangular_window(PdsVector *h,PdsDfReal factor)
{
	PdsDfNatural i,Nel;
	PdsDfReal alpha;

	if(h==NULL)	return FALSE;
	if(factor>1.0)			return FALSE;
	if(factor<0.0)			return FALSE;

	Nel=h->Nel;
	alpha=(Nel-1.0)/2.0;

	for(i=0;i<Nel;i++)
	{
		if(i*1.0!=alpha)
		h->V[i]=factor*sin(factor*M_PI*(i-alpha))/(factor*M_PI*(i-alpha));
		else
		h->V[i]=factor;
	}

	return TRUE;
}


/** \fn PdsVector *pds_df_butterworth_hnum_new(PdsDfNatural N,PdsDfReal factor)
 *  \brief Devuelve los coeficientes de un filtro FIR pasa bajo. 
 *  Usando el método de la ventana, con ventana rectangular
 *  \param[in] N Orden del filtro FIR.
 *  \param[in] factor Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital.
 *  \return Un puntero a un vector de coeficientes de filtro FIR o NULL en caso de error. 
 *  (ej. N==0 factor>1 o factor<0)
 *  \ingroup PdsDfUtilsGroup
 */
PdsVector *pds_df_rectangular_window_new(PdsDfNatural N,PdsDfReal factor)
{
	PdsDfNatural i,Nel;
	PdsDfReal alpha;
	PdsVector *H=NULL;

	if(N==0)	return NULL;
	if(factor>1.0)	return NULL;
	if(factor<0.0)	return NULL;

	Nel=N+1;
	H=pds_vector_new(Nel);
	if(H==NULL)	return NULL;

	alpha=(Nel-1.0)/2.0;

	for(i=0;i<Nel;i++)
	{
		if(i*1.0!=alpha)
		H->V[i]=factor*sin(factor*M_PI*(i-alpha))/(factor*M_PI*(i-alpha));
		else
		H->V[i]=factor;
	}
	return H;
}


/** \fn int pds_df_hamming_window(PdsVector *h,PdsDfReal factor)
 *  \brief Encuentra los coeficientes de un filtro FIR pasa bajo. 
 *  Usando el método de la ventana, con ventana de Hamming.
 *  \param[out] h Donde se guardarán los coeficientes del filtro digital.
 *  \param[in] factor Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. h==NULL, factor<0 o factor>1)
 *  \ingroup PdsDfUtilsGroup
 */
int pds_df_hamming_window(PdsVector *h,PdsDfReal factor)
{
	PdsDfNatural i,Nel;
	PdsDfReal alpha;

	if(h==NULL)	return FALSE;
	if(factor>1.0)			return FALSE;
	if(factor<0.0)			return FALSE;

	Nel=h->Nel;
	alpha=(Nel-1.0)/2.0;

	for(i=0;i<Nel;i++)
	{
		if(i*1.0!=alpha)
		h->V[i]=factor*sin(factor*M_PI*(i-alpha))/(factor*M_PI*(i-alpha));
		else
		h->V[i]=factor;

		h->V[i]=h->V[i]*(0.54-0.46*cos(2*M_PI*i/(Nel-1.0)));
	}

	return TRUE;
}

/** \fn PdsVector *pds_df_hamming_window_new(PdsDfNatural N,PdsDfReal factor)
 *  \brief Devuelve los coeficientes de un filtro FIR pasa bajo. 
 *  Usando el método de la ventana, con ventana de Hamming.
 *  \param[in] N Orden del filtro FIR.
 *  \param[in] factor Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital.
 *  \return Un puntero a un vector de coeficientes de filtro FIR o NULL en caso de error. 
 *  (ej. N==0 factor>1 o factor<0)
 *  \ingroup PdsDfUtilsGroup
 */
PdsVector *pds_df_hamming_window_new(PdsDfNatural N,PdsDfReal factor)
{
	PdsDfNatural i,Nel;
	PdsDfReal alpha;
	PdsVector *H=NULL;

	if(N==0)	return NULL;
	if(factor>1.0)	return NULL;
	if(factor<0.0)	return NULL;

	Nel=N+1;
	H=pds_vector_new(Nel);
	if(H==NULL)	return NULL;

	alpha=(Nel-1.0)/2.0;

	for(i=0;i<Nel;i++)
	{
		if(i*1.0!=alpha)
		H->V[i]=factor*sin(factor*M_PI*(i-alpha))/(factor*M_PI*(i-alpha));
		else
		H->V[i]=factor;

		H->V[i]=H->V[i]*(0.54-0.46*cos(2*M_PI*i/(Nel-1.0)));
	}
	return H;
}


/** \fn int pds_df_hanning_window(PdsVector *h,PdsDfReal factor)
 *  \brief Encuentra los coeficientes de un filtro FIR pasa bajo. 
 *  Usando el método de la ventana, con ventana de Hanning.
 *  \param[out] h Donde se guardarán los coeficientes del filtro digital.
 *  \param[in] factor Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. h==NULL, factor<0 o factor>1)
 *  \ingroup PdsDfUtilsGroup
 */
int pds_df_hanning_window(PdsVector *h,PdsDfReal factor)
{
	PdsDfNatural i,Nel;
	PdsDfReal alpha;

	if(h==NULL)	return FALSE;
	if(factor>1.0)			return FALSE;
	if(factor<0.0)			return FALSE;

	Nel=h->Nel;
	alpha=(Nel-1.0)/2.0;

	for(i=0;i<Nel;i++)
	{
		if(i*1.0!=alpha)
		h->V[i]=factor*sin(factor*M_PI*(i-alpha))/(factor*M_PI*(i-alpha));
		else
		h->V[i]=factor;

		h->V[i]=h->V[i]*(0.5-0.5*cos(2*M_PI*i/(Nel-1.0)));
	}

	return TRUE;
}

/** \fn PdsVector *pds_df_hanning_window_new(PdsDfNatural N,PdsDfReal factor)
 *  \brief Devuelve los coeficientes de un filtro FIR pasa bajo. 
 *  Usando el método de la ventana, con ventana de Hanning.
 *  \param[in] N Orden del filtro FIR.
 *  \param[in] factor Número entre 0 y 1.0 que indica la proporción de PI, para 
 *  la frecuencia de corte del filtro digital.
 *  \return Un puntero a un vector de coeficientes de filtro FIR o NULL en caso de error. 
 *  (ej. N==0 factor>1 o factor<0)
 *  \ingroup PdsDfUtilsGroup
 */
PdsVector *pds_df_hanning_window_new(PdsDfNatural N,PdsDfReal factor)
{
	PdsDfNatural i,Nel;
	PdsDfReal alpha;
	PdsVector *H=NULL;

	if(N==0)	return NULL;
	if(factor>1.0)	return NULL;
	if(factor<0.0)	return NULL;

	Nel=N+1;
	H=pds_vector_new(Nel);
	if(H==NULL)	return NULL;

	alpha=(Nel-1.0)/2.0;

	for(i=0;i<Nel;i++)
	{
		if(i*1.0!=alpha)
		H->V[i]=factor*sin(factor*M_PI*(i-alpha))/(factor*M_PI*(i-alpha));
		else
		H->V[i]=factor;

		H->V[i]=H->V[i]*(0.5-0.5*cos(2*M_PI*i/(Nel-1.0)));
	}
	return H;
}

/** \fn int pds_df_frequency_response_fir(const PdsVector *h,PdsVector *H)
 *  \brief Encuentra el módulo de la respuesta en frecuencia, normalizada 
 *  de 0 a 2*pi del filtro digital conformado por el numerador h.
 *  \param[in] h Coeficientes del filtro digital.
 *  \param[out] H Donde se guardará el módulo cuadrado de la respuesta en 
 *  frecuencia del filtro digital.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. h==NULL o H==NULL)
 *  \ingroup PdsDfUtilsGroup
 */
int pds_df_frequency_response_fir(const PdsVector *h,PdsVector *H)
{
	PdsDfNatural i,k,N;
	PdsDfReal NumReal;
	PdsDfReal NumImag;
	PdsDfReal Modulo2;
	PdsDfReal teta;

	if(h==NULL)	return FALSE;
	if(H==NULL)	return FALSE;

	N=H->Nel;

	for(i=0;i<N;i++)
	{
		teta=i*2.0*M_PI/N;

		for(NumReal=0.0,k=0;k<h->Nel;k++)
		{
			NumReal=NumReal+h->V[k]*cos(teta*k);
		}

		for(NumImag=0.0,k=0;k<h->Nel;k++)
		{
			NumImag=NumImag-h->V[k]*sin(teta*k);
		}

		Modulo2=(NumReal*NumReal+NumImag*NumImag);

		H->V[i]=sqrt(Modulo2);
	}

	return TRUE;
}



/** \fn int pds_df_lowpass_to_highpass(PdsVector *H)
 *  \brief Tranforma un filtro pasa bajo con un corte en Wc a un filtro pasa alto
 *  con un corte en PI-Wc. En realidad lo que hace es un corrimiento de PI de la
 *  respuesta en frecuencia. Que es equivalente a cambiar el valor de Z por -Z.
 *  \param[in,out] H Donde se encuentra el filtro pasa bajo y donde se guardará 
 *  el filtro pasa alto.
 *  \return TRUE si todo fue bien o FALSE si no. (ej. H==NULL)
 *  \ingroup PdsDfUtilsGroup
 */
int pds_df_lowpass_to_highpass(PdsVector *H)
{
	PdsDfNatural i;

	if(H==NULL)	return FALSE;

	for(i=1;i<H->Nel;i=i+2)
	{
		H->V[i]=-H->V[i];
	}

	return TRUE;
}


/** \fn PdsVector *pds_df_lowpass_to_bandpass(const PdsVector *H)
 *  \brief Crea a partir de un filtro pasa bajo con un corte en Wc un filtro pasa banda
 *  centrado en (PI/2) con ancho de banda Wc. 

 *  En realidad lo que hace es comprimir
 *  toda respuesta en frecuencia de [-PI,PI] a [-PI/2,PI/2] y dislocarla a PI/2 y
 *  -PI/2. Que es equivalente a cambiar el valor de Z por -Z^2. Conociendo que Nel
 *  es el numero de elementos de H, entonces el vector entregado tendrá 2N-1 elementos.
 *  \param[in] H Donde se encuentra el filtro pasa bajo.
 *  \return Si todo fue bien retorna un puntero a un vector con el filtro pasa 
 *  banda, o NULL si no. (ej. H==NULL)
 *  \ingroup PdsDfUtilsGroup
 */
PdsVector *pds_df_lowpass_to_bandpass(const PdsVector *H)
{
	PdsDfNatural i;

	PdsVector *Hband=NULL;

	if(H==NULL)	return NULL;
	
	Hband=pds_vector_new(2*H->Nel-1);

	for(i=0;i<H->Nel;i++)
	{
		if(i%2==0)	Hband->V[2*i]=H->V[i];
		else		Hband->V[2*i]=-H->V[i];
	}

	return Hband;
}


/** \fn PdsVector *pds_df_lowpass_to_bandreject(const PdsVector *H)
 *  \brief Crea a partir de un filtro pasa bajo con un corte en Wc un filtro rechaza banda
 *  centrado en (PI/2) con ancho de rechaza banda PI-Wc. 

 *  En realidad lo que hace es comprimir
 *  toda respuesta en frecuencia de [-PI,PI] a [-PI/2,PI/2] y dislocarla a 0, PI y
 *  -PI. Que es equivalente a cambiar el valor de Z por Z^2. Conociendo que Nel
 *  es el numero de elementos de H, entonces el vector entregado tendrá 2N-1 elementos.
 *  \param[in] H Donde se encuentra el filtro pasa bajo.
 *  \return Si todo fue bien retorna un puntero a un vector con el filtro rechaza 
 *  banda, o NULL si no. (ej. H==NULL)
 *  \ingroup PdsDfUtilsGroup
 */
PdsVector *pds_df_lowpass_to_bandreject(const PdsVector *H)
{
	PdsDfNatural i;

	PdsVector *Hreject=NULL;

	if(H==NULL)	return NULL;
	
	Hreject=pds_vector_new(2*H->Nel-1);

	for(i=0;i<H->Nel;i++)
	{
		Hreject->V[2*i]=H->V[i];
	}

	return Hreject;
}

