/*
 * pdslstring.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdslstring.h
 *  \author Fernando Pujaico Rivera
 *  \date 30-07-2016
 *  \brief Funciones que trabajan con listas doblemente enlazadas que almacenan 
 *  cadenas de texto.
 *   
 */

#ifndef __PDSLSTRING_H__
#define __PDSLSTRING_H__


#ifdef __cplusplus
extern "C" {
#endif 

#include <stdio.h>


/** \defgroup PdsListStringGroup Funciones del módulo PdsListString.
 *
 *  Estas funciones manipulan una lista con elementos de tipo cadena.
 *
 * <h1>\#include \<pds/pdsdatafunc.h\></h1>
 *
 * @{
 */

#ifndef PDS_OK
/*!
  \def PDS_OK
  Valor retornado en caso de exito, este es un valor diferente que cero.
*/
	#define PDS_OK 1
#endif

#ifndef PDS_WRONG
/*!
  \def PDS_WRONG
  Valor retornado en caso de fracaso, este valor es igual a cero.
*/
	#define PDS_WRONG 0
#endif

/*! \typedef struct PdsListString
 *  \brief Defino el tipo de variable PdsListString.
 *  <br>
 *
 *  \ingroup PdsListStringGroup
 */
typedef struct PdsListString PdsListString;

/*! \struct PdsListString
 *  \brief Una estructura tipo  PdsListString .
 *  Esta estructura genera una doblemente enlazada lista de cadenas te texto.
 *  <br>
 *
 *  \ingroup PdsListStringGroup
 *  \author Fernando Pujaico Rivera
 */
struct PdsListString
{
	// Dirección del nodo anterios de la lista.
	struct PdsListString *AddPrev;
	// Strings real asociado; dentro;del nodo.
	char *S;
	// Direccion del nodo siguiente en la lista.
	struct PdsListString *AddNext;
};



/** @name pds_list_string_<new_family>
 *  Funciones para crear nuevos list strings.
 *
 * @{
 */

/** \fn PdsListString *pds_list_string_new(void)
 *  \brief Crea una lista de tipo PdsListString vacia.
 *  \return Un puntero a un elemento de tipo PdsListString.
 *  \ingroup PdsListStringGroup
 */
PdsListString *pds_list_string_new(void);

/** \fn PdsListString *pds_list_string_new_with_string(const char* data)
 *  \brief Crea una lista de tipo PdsListString con el dato data.
 *  \return Un puntero a un elemento de tipo PdsListString.
 *  \ingroup PdsListStringGroup
 */
PdsListString *pds_list_string_new_with_string(const char* data);

//@}

/** @name pds_list_string_<adding_family>
 *  Funciones para agregar o insertar list strings.
 *
 * @{
 */

/** \fn int pds_list_string_push(PdsListString **Lnode,const char* string_data)
 *  \brief Agrega un elemento a la cima de la lista. O al nodo actual si el 
 *  string es null. Si un nuevo nodo es agregado Lnode apuntara a este nodo.
 *  Si no le das el último nodo de la lista da error no hace nada y retorna PDS_WRONG.
 *  \param[in,out] Lnode El nodo de la cima de la lista.
 *  \param[in] string_data La cadena a escribir.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no. Si no le das el último nodo da error.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_push(PdsListString **Lnode,const char* string_data);

/** \fn int pds_list_string_unshift(PdsListString **Lnode,const char *string_data)
 *  \brief Agrega un elemento al inicio de la lista. O al nodo actual si el 
 *  string es null. Si un nuevo nodo es agregado Lnode apuntara a este nodo.
 *  Si no le das el primer nodo de la lista da error y no hace nada.
 *  \param[in,out] Lnode El nodo inicial de la lista.
 *  \param[in] string_data La cadena a escribir.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no. Si no le das el primer nodo da error.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_unshift(PdsListString **Lnode,const char *string_data);



/** \fn int pds_list_string_add_top_node(PdsListString **Lnode,const char* string_data)
 *  \brief Agrega un elemento a la cima de la lista. Si la lista no está en la cima,
 *  la lista es llevada a la cima y es dejada en la cima luego de agregar esta nueva posición.
 *  \param[in,out] Lnode Un nodo cualquiera de la lista.
 *  \param[in] string_data La cadena a escrever.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_add_top_node(PdsListString **Lnode,const char* string_data);


//@}

/** @name pds_list_string_<removing_family>
 *  Funciones para eliminar o recuperar list strings.
 *
 * @{
 */

/** \fn int pds_list_string_pop(PdsListString **Lnode)
 *  \brief Quita un nodo al final de la lista. Si el nodo es removido, Lnode
 *  apuntará al nuevo nodo en la cima.
 *  Si la lista tiene un solo nodo, este es limpiado y Lnode no cambia de dirección.
 *  Si no le das el último nodo de la lista da error, no hace nada y retorna PDS_WRONG.
 *  \param[in,out] Lnode El nodo de la cima de la lista.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no. Si no le das el último nodo da error.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_pop(PdsListString **Lnode);


/** \fn int pds_list_string_shift(PdsListString **Lnode)
 *  \brief Quita un nodo al inicio de la lista. Si el nodo es removido, Lnode
 *  apuntará al nuevo nodo inicial.
 *  Si la lista tiene un solo nodo, este es limpiado y Lnode no cambia de dirección.
 *  Si no le das el último nodo de la lista da error, no hace nada y retorna PDS_WRONG.
 *  \param[in,out] Lnode El nodo al inicio de la lista.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no. Si no le das el primer nodo da error.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_shift(PdsListString **Lnode);


/** \fn int pds_list_string_delete_top_node(PdsListString **Lnode)
 *  \brief Elimina un elemento a la cima de la lista. Si la lista no está en la cima,
 *  la lista es llevada a la cima y es dejada en la cima luego de eliminar la ultima posición.
 *  Si la lista tiene un solo nodo, este es limpiado y Lnode no cambia de dirección
 *  \param[in,out] Lnode Un nodo cualquiera de la lista.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_delete_top_node(PdsListString **Lnode);


//@}

/** @name pds_list_string_<read_family>
 *  Funciones para leer datos de list strings.
 *
 * @{
 */

/** \fn int pds_list_string_read(const PdsListString *L,char** string_data)
 *  \brief Lee la cadena de texto del nodo actual de la lista.
 *  \param[in] L Un nodo cualquiera de la lista.
 *  \param[out] string_data La dirección de memoria interna en el PdsListString 
 *  de la cadena a leer (NO liberar esta dirección de memoria).
 *  \return PDS_OK si todo fue bien o PDS_WRONG si (*L)==NULL.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_read(const PdsListString *L,char** string_data);


/** \fn int pds_list_string_read_and_advance(PdsListString **L,char** string_data)
 *  \brief Lee una cadena de texto del nodo actual de la lista y se coloca en el siguiente 
 *  nodo de la lista, si está ya en el último nodo no se mueve al siguiente nodo.
 *  \param[in,out] L Un nodo cualquiera de la lista.
 *  \param[out] string_data La dirección de memoria interna en el PdsListString  
 *  de la cadena a leer (NO liberar esta dirección de memoria).
 *  \return PDS_OK si fue posible moverse al siguiente nodo o PDS_WRONG si no pudo 
 *  moverse al siguiente nodo o si (*L)==NULL.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_read_and_advance(PdsListString **L,char** string_data);


/** \fn char* pds_list_string_get_string(const PdsListString *L)
 *  \brief Lee una cadena de texto de un nodo en la lista actual y retorna una 
 *  nueva cadena con el contenido.
 *  \param[in] L Un nodo cualquiera de la lista.
 *  \return Una nueva cadena con el contenido del PdsListString. O NULL en caso
 *  de cadena vacia o L==NULL o error de allocacion de memoria. Esta cadena debe
 *  ser liberada despues de usada.
 *  \ingroup PdsListStringGroup
 */
char* pds_list_string_get_string(const PdsListString *L);

//@}

/** @name pds_list_string_<location_family>
 *  Funciones para seleccionar posición del nodo en la list string.
 *
 * @{
 */


/** \fn int pds_list_string_go_top(PdsListString **L)
 *  \brief Busca el elemento final; superior; de la lista.
 *  \param[in,out] L Un nodo de la lista, en donde se cargará el último nodo.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no. 
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_go_top(PdsListString **L);


/** \fn int pds_list_string_go_bottom(PdsListString **L)
 *  \brief Busca el elemento inicial; inferior; de la lista.
 *  \param[in,out] L Un nodo de la lista, en donde se cargará el primer nodo.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no. 
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_go_bottom(PdsListString **L);

//@}




/** @name pds_list_string_<printf_family>
 *  Funciones para imprimir en pantalla datos de list strings.
 *
 * @{
 */


/** \fn int pds_list_string_printf(const PdsListString *L)
 *  \brief Imprime en pantalla los datos de un nodo de tipo puntero PdsListString.
 *  \param[in] L El nodo a imprimir en pantalla.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_printf(const PdsListString *L);


/** \fn int pds_list_string_detailed_printf(const PdsListString *L)
 *  \brief Imprime en pantalla detalladamente los datos de un nodo de tipo puntero PdsListString.
 *  \param[in] L El nodo a imprimir en pantalla.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_detailed_printf(const PdsListString *L);


/** \fn int pds_list_string_all_printf(const PdsListString *L)
 *  \brief Imprime en pantalla todos los datos de la lista.
 *  \param[in] L Un nodo de la lista a imprimir en pantalla.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_all_printf(const PdsListString *L);


/** \fn int pds_list_string_all_detailed_printf(const PdsListString *L)
 *  \brief Imprime en pantalla todos los datos de la lista detalladamente.
 *  \param[in] L Un nodo de la lista a imprimir en pantalla.
 *  \return PDS_OK si todo fue bien o PDS_WRONG si no.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_all_detailed_printf(const PdsListString *L);

//@}

/** @name pds_list_string_<free_family>
 *  Funciones para liberar memoria de list strings.
 *
 * @{
 */

/** \fn void pds_list_string_free(PdsListString *L)
 *  \brief Libera una lista entera de tipo puntero PdsListString. 
 *  \param[in,out] L La lista a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsListStringGroup
 */
void pds_list_string_free(PdsListString *L);


/** \fn void pds_list_string_destroy(PdsListString **L)
 *  \brief Libera una lista de tipo puntero PdsListString, y limpia el puntero con NULL.
 *  \param[in,out] L La lista a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsListStringGroup
 */
void pds_list_string_destroy(PdsListString **L);

//@}


/** @name pds_list_string_<methods>
 *  Funciones extras para las list string.
 *
 * @{
 */

/** \fn int pds_list_string_is_top(const PdsListString *L)
 *  \brief Retorna PDS_OK si L apunta el nodo que esta en el top de la lista.
 *  \param[in] L Un nodo cualquiera de la lista.
 *  \return PDS_OK si L apunta al final de la lista o PDS_WRONG si no.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_is_top(const PdsListString *L);


/** \fn int pds_list_string_is_bottom(const PdsListString *L)
 *  \brief Retorna PDS_OK si L apunta el nodo que esta en el bottom de la lista.
 *  \param[in] L Un nodo cualquiera de la lista.
 *  \return PDS_OK si L apunta al inicio de la lista o PDS_WRONG si no.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_is_bottom(const PdsListString *L);


/** \fn int pds_list_string_get_number_of_nodes(const PdsListString *L)
 *  \brief Cuenta la cantidad de nodos de la lista.
 *  \param[in] L Un nodo cualquiera de la lista a consultar.
 *  \return Retorna la cantidad de nodos de la lista, 0 si L==NULL o un número  
 *  menor que cero en caso de error.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_get_number_of_nodes(const PdsListString *L);

/** \fn int pds_list_string_get_number_of_no_null_nodes(const PdsListString *L)
 *  \brief Cuenta la cantidad de nodos con strings no nulos en la lista.
 *  \param[in] L Un nodo cualquiera de la lista a consultar.
 *  \return Retorna la cantidad de nodos no nulos de la lista, 0 si L==NULL o un 
 *  número menor que cero en caso de error.
 *  Si se encuentra un nodo con L->AddNext==L->AddPrev==NULL
 *  es considerado un error (exepto para el nodo inicial). 
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_get_number_of_no_null_nodes(const PdsListString *L);


/** \fn int pds_list_string_is_empty(const PdsListString *L)
 *  \brief Indica si el nodo está vacío L->S==L->AddNextL->AddPrev==NULL.
 *  \param[in] L Un nodo a consultar.
 *  \return Retorna PDS_OK si L==NULL o si todos los elementos del nodo son nulos.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_is_empty(const PdsListString *L);

//@}

/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif	/* __PDSLSTRING_H__ */ 

