/*
 * pdslstring.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <stdlib.h>
#include <string.h>
#include <pds/pdslstring.h>


/** \fn PdsListString *pds_list_string_new(void)
 *  \brief Crea una lista de tipo PdsListString vacia.
 *  \return Un puntero a la cadena de tipo PdsListString.
 *  \ingroup PdsListStringGroup
 */
PdsListString *pds_list_string_new(void)
{
	PdsListString *L=NULL;
	L=(PdsListString *)calloc(1,sizeof(PdsListString));
	if(L==NULL)	return NULL;
	
	L->AddPrev=NULL;
	L->S=NULL;
	L->AddNext=NULL;

	return L;
}


/** \fn int pds_list_string_push(PdsListString **Lprev,const char* string_data)
 *  \brief Agrega un elemento a la cima de la lista.
 *  \param[in,out] Lprev El nodo de la cima de la lista.
 *  \param[in] string_data La cadena a escrever.
 *  \return TRUE si todo fue bien o FALSE si no. Si no le das el último nodo da error
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_push(PdsListString **Lprev,const char* string_data)
{
	PdsListString *L=NULL;

	if((*Lprev)==NULL)		return FALSE;
	if((*Lprev)->AddNext!=NULL)	return FALSE;

	if((*Lprev)->S==NULL)
	{
		(*Lprev)->S=(char*)calloc(sizeof(char),strlen(string_data)+1);
        sprintf((*Lprev)->S,"%s",string_data);
		return TRUE;
	}
	else
	{
		L=(PdsListString *)calloc(1,sizeof(PdsListString));
		if(L==NULL)	return FALSE;
	
		L->AddPrev=(struct PdsListString *)(*Lprev);
		L->S=(char*)calloc(sizeof(char),strlen(string_data)+1);
		if(L->S==NULL)
		{
			free(L);
			return FALSE;
		}
        sprintf(L->S,"%s",string_data);

		L->AddNext=NULL;

		(*Lprev)->AddNext=(struct PdsListString *)L;

		(*Lprev)=L;
	
		return TRUE;
	}
}


/** \fn int pds_list_string_pop(PdsListString **L)
 *  \brief Quita un elemento de la lista. Si no hay elementos retorna FALSE.
 *  \param[in,out] L El nodo de la cima de la lista.
 *  \return TRUE si todo fue bien o FALSE si no. Si no le das el último nodo da error
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_pop(PdsListString **L)
{
	PdsListString *Lprev=NULL;

	if((*L)==NULL)		return FALSE;
	if((*L)->AddNext!=NULL)	return FALSE;
	
	if((*L)->AddPrev==NULL)
	{
		if((*L)->S==NULL)
		{
			return FALSE;
		}
		else
		{
			(*L)->AddNext=NULL;
			free((*L)->S);
			(*L)->S=NULL;
			(*L)->AddPrev=NULL;
			return TRUE;
		}
	}
	else
	{
		Lprev=(PdsListString *)((*L)->AddPrev);
	
		(*L)->AddNext=NULL;
		free((*L)->S);
		(*L)->S=NULL;
		(*L)->AddPrev=NULL;
		free((*L));

		(*L)=Lprev;
		(*L)->AddNext=NULL;
		return TRUE;
	}
}


/** \fn int pds_list_string_shift(PdsListString **L)
 *  \brief Quita el elemento inicial;mas antiguo; de la lista, si no hay elementos retorna FALSE.
 *  \param[in,out] L El primer nodo de la lista.
 *  \return TRUE si todo fue bien o FALSE si no. Si no le das el primer nodo da error.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_shift(PdsListString **L)
{
	PdsListString *Lnext=NULL;

	if((*L)==NULL)		return FALSE;
	if((*L)->AddPrev!=NULL)	return FALSE;
	
	if((*L)->AddNext==NULL)
	{
		if((*L)->S==NULL)
		{
			return FALSE;
		}
		else
		{
			free((*L)->S);
			(*L)->S=NULL;
			return TRUE;
		}
	}
	else
	{
		Lnext=(PdsListString *)((*L)->AddNext);
	
		free((*L)->S);
		(*L)->S=NULL;
		(*L)->AddNext=NULL;
		free((*L));

		(*L)=Lnext;
		(*L)->AddPrev=NULL;
		return TRUE;
	}
}


/** \fn int pds_list_string_unshift(PdsListString **Lnext,const char *string_data)
 *  \brief Agrega un elemento al inicio de la lista.
 *  \param[in,out] Lnext El primer nodo de la lista.
 *  \param[in] string_data La cadena a escrever.
 *  \return TRUE si todo fue bien o FALSE si no. Si no le das el primer nodo da error.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_unshift(PdsListString **Lnext,const char *string_data)
{
	PdsListString *L=NULL;

	if((*Lnext)==NULL)		return FALSE;
	if((*Lnext)->AddPrev!=NULL)	return FALSE;

	if((*Lnext)->S==NULL)
	{
		(*Lnext)->S=(char*)calloc(sizeof(char),strlen(string_data)+1);
        sprintf((*Lnext)->S,"%s",string_data);
		return TRUE;
	}
	else
	{
		L=(PdsListString *)calloc(1,sizeof(PdsListString));
		if(L==NULL)	return FALSE;
	
		L->AddNext=(struct PdsListString *)(*Lnext);
		L->S=(char*)calloc(sizeof(char),strlen(string_data)+1);
		if(L->S==NULL)
		{
			free(L);
			return FALSE;
		}
        sprintf(L->S,"%s",string_data);
		L->AddPrev=NULL;

		(*Lnext)->AddPrev=(struct PdsListString *)L;

		(*Lnext)=L;
	
		return TRUE;
	}
}


/** \fn int pds_list_string_top(PdsListString **L)
 *  \brief Busca el elemento final; superior; de la lista.
 *  \param[in,out] L Un nodo de la lista, en donde se cargará el último nodo.
 *  \return TRUE si todo fue bien o FALSE si no. 
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_top(PdsListString **L)
{
	if((*L)==NULL)	return FALSE;

	while((*L)->AddNext!=NULL)
	{
		(*L)=(PdsListString*)((*L)->AddNext);
	}
	return TRUE;
}


/** \fn int pds_list_string_bottom(PdsListString **L)
 *  \brief Busca el elemento inicial; inferior; de la lista.
 *  \param[in,out] L Un nodo de la lista, en donde se cargará el primer nodo.
 *  \return TRUE si todo fue bien o FALSE si no. 
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_bottom(PdsListString **L)
{
	if((*L)==NULL)	return FALSE;

	while((*L)->AddPrev!=NULL)
	{
		(*L)=(PdsListString*)((*L)->AddPrev);
	}
	return TRUE;
}


/** \fn int pds_list_string_printf(const PdsListString *L)
 *  \brief Imprime en pantalla los datos de un nodo de tipo puntero PdsListString.
 *  \param[in] L El nodo a imprimir en pantalla.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_printf(const PdsListString *L)
{
	if(L==NULL)		return FALSE;

	printf("%p\t",L);
	printf("AddPrev : %p\n",L->AddPrev);
	if(L->S!=NULL)	
	{
		printf("\t\tS       : ");
		printf("%s\n",L->S);
	}
	else		printf("\t\tS       :\n");
	printf("\t\tAddNext : %p\n",L->AddNext);
	
	return TRUE;
}


/** \fn int pds_list_string_all_printf(const PdsListString *L)
 *  \brief Imprime en pantalla todos los datos de la lista.
 *  \param[in] L Un nodo de la lista a imprimir en pantalla.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_all_printf(const PdsListString *L)
{
	PdsListString *Lbottom=NULL;
	int id;

	if(L==NULL)		return FALSE;

	Lbottom=(PdsListString *)L;

	id=pds_list_string_bottom(&Lbottom);
	if(id==FALSE)	return FALSE;

	
	while(1)
	{
		id=pds_list_string_printf(Lbottom);
		if(id==FALSE)	return FALSE;

		if(Lbottom->AddNext!=NULL)	Lbottom=(PdsListString*)Lbottom->AddNext;
		else				break;
	}
		
	return TRUE;
}


/** \fn void pds_list_string_free(PdsListString *L)
 *  \brief Libera una lista entera de tipo puntero PdsListString. 
 *  \param[in,out] L La lista a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsListStringGroup
 */
void pds_list_string_free(PdsListString *L)
{
	PdsListString *Lbottom=NULL;
	int id;

	if(L!=NULL)
	{
		Lbottom=L;

		id=pds_list_string_bottom(&Lbottom);
		if(id==TRUE)
		{
			while(1)
			{
				pds_list_string_shift(&Lbottom);
				if(Lbottom->AddNext==NULL)	break;
			}

			free(Lbottom);
		}
	}
}


/** \fn void pds_list_string_destroy(PdsListString **L)
 *  \brief Libera una lista de tipo puntero PdsListString, y limpia el puntero con NULL.
 *  \param[in,out] L La lista a liberar y limpiar.
 *  \return No retorna valor.
 *  \ingroup PdsListStringGroup
 */
void pds_list_string_destroy(PdsListString **L)
{
	PdsListString *Lbottom=NULL;
	int id;

	if(L!=NULL)
	if((*L)!=NULL)
	{
		Lbottom=*L;

		id=pds_list_string_bottom(&Lbottom);
		if(id==TRUE)
		{
			while(1)
			{
				pds_list_string_shift(&Lbottom);
				if(Lbottom->AddNext==NULL)	break;
			}

			free(Lbottom);
			*L=NULL;
		}
	}
}




/** \fn int pds_list_string_add(PdsListString **Lprev,const char* string_data)
 *  \brief Agrega un elemento a la cima de la lista. Si la lista no esta en la cima,
 *  la lista es llevada a la cima y es dejada luego de agregar en esta posición.
 *  \param[in,out] Lprev Un nodo cualquiera de la lista.
 *  \param[in] string_data La cadena a escrever.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_add(PdsListString **Lprev,const char* string_data)
{
    int id;

	if((*Lprev)==NULL)  return FALSE;

	if((*Lprev)->AddNext!=NULL) 
    {
        id=pds_list_string_top(Lprev);
        if (id==FALSE)  return FALSE;
    }

    id=pds_list_string_push(Lprev,string_data);
    return id;
}


/** \fn int pds_list_string_read(PdsListString **Lprev,char** string_data)
 *  \brief Lee una cadena de texto, en la lista actual y se coloca en el siguiente 
 *  nodo de la lista, si está ya en el último nodo no se mueve al siguiente nodo.
 *  \param[in,out] Lprev Un nodo cualquiera de la lista.
 *  \param[out] string_data La dirección de la cadena a leer (NO liberar esta 
 *  dirección de memoria).
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_read(PdsListString **Lprev,char** string_data)
{
    int id;
    PdsListString *L=NULL;

	if((*Lprev)==NULL)		    return FALSE;


    (*string_data)=(*Lprev)->S;

	if((*Lprev)->AddNext!=NULL) 
    {
        (*Lprev)=(PdsListString*)((*Lprev)->AddNext);
    }

    return TRUE;
}


/** \fn int pds_list_string_is_top(const PdsListString *L)
 *  \brief Retorna TRUE si L apunta el nodo que esta en el top de la lista.
 *  \param[in] L Un nodo cualquiera de la lista.
 *  \return TRUE si L apunta al final de la lista o FALSE si no.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_is_top(const PdsListString *L)
{
	if(L==NULL)		    return FALSE;

	if(L->AddNext==NULL)    return TRUE;
    else                    return FALSE;
}


/** \fn int pds_list_string_is_bottom(const PdsListString *L)
 *  \brief Retorna TRUE si L apunta el nodo que esta en el bottom de la lista.
 *  \param[in] L Un nodo cualquiera de la lista.
 *  \return TRUE si L apunta al inicio de la lista o FALSE si no.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_is_bottom(const PdsListString *L)
{
	if(L==NULL)		    return FALSE;

	if(L->AddPrev==NULL)    return TRUE;
    else                    return FALSE;
}


/** \fn int pds_list_string_get_length(const PdsListString *L)
 *  \brief Cuenta la cantidad de nodos de la lista.
 *  Es considerada una lista con cero nodos cuando no exiten nodos antes, nodos 
 *  despues y el valor actual de la lista NULL.
 *  \param[in] L Un nodo cualquiera de la lista a consultar.
 *  \return Retorna la cantidad de nodos de la lista o un número menor que cero 
 *  en caso de error.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_get_length(const PdsListString *L)
{
	PdsListString *Lbottom=NULL;
	int N=0;
	int id;

	if(L==NULL)		return -1;

	Lbottom=(PdsListString *)L;

	id=pds_list_string_bottom(&Lbottom);
	if(id==FALSE)	return -1;

	
	while(1)
	{
		if (Lbottom->S==NULL)
        if (Lbottom->AddNext==NULL)
        if (Lbottom->AddPrev==NULL) 
        {
            return 0;
        }

        N=N+1;

		if(Lbottom->AddNext!=NULL)	Lbottom=(PdsListString*)Lbottom->AddNext;
		else				break;
	}
		
	return N;
}



/** \fn int pds_list_string_is_empty(const PdsListString *L)
 *  \brief Indica si la lista está vacía.
 *  \param[in] L Un nodo cualquiera de la lista a consultar.
 *  \return Retorna TRUE si L==NULL o si todos los elementos del nodo son nulos.
 *  \ingroup PdsListStringGroup
 */
int pds_list_string_is_empty(const PdsListString *L)
{
	if(L==NULL)		return TRUE;
	
	if (L->S==NULL)
    if (L->AddNext==NULL)
    if (L->AddPrev==NULL) 
    {
        return TRUE;
    }
		
	return FALSE;
}
