/*
 * pdscstring.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <stdlib.h>
#include <string.h>
#include <fnmatch.h>

#include <pds/pdscstring.h>
#include <pds/pdsstring.h>
#include <pds/pdsfilesfunc.h>


/** \fn PdsCellString *pds_cell_string_new(int N)
 *  \brief Crea un arreglo de células de tipo PdsCellString con N elementos 
 *  vacíos (NULL).
 *  \param[in] N Número de elementos de elementos del arreglo de células.
 *  \return Un puntero a una estructura de tipo PdsCellString.
 *  \ingroup PdsCellStringGroup
 */
PdsCellString *pds_cell_string_new(int N)
{
	PdsCellString *C=NULL;
    int i;

    if (N<0)   return NULL;

	C=(PdsCellString *)calloc(1,sizeof(PdsCellString));
	if(C==NULL)	return NULL;
	
    C->Ncell=N;
    
    C->data=(char **)calloc(N+1,sizeof(char*));
    if(C->data==NULL)
    {
        free(C);
        return NULL;
    }

    for(i=0;i<=N;i++)
    {
        C->data[i]=NULL;
    }

	return C;
}


/** \fn PdsCellString *pds_cell_string_copy_new(const PdsCellString *Cin)
 *  \brief Crea un arreglo de células de tipo PdsCellString copiando los datos de otra.
 *  \param[in] Cin La célula a ser copiada.
 *  \return Un puntero a una estructura de tipo PdsCellString o NULL en caso de error.
 *  \ingroup PdsCellStringGroup
 */
PdsCellString *pds_cell_string_copy_new(const PdsCellString *Cin)
{
	PdsCellString *C=NULL;
    int i,j;

    if (Cin==NULL)   return NULL;

	C=(PdsCellString *)calloc(1,sizeof(PdsCellString));
	if(C==NULL)	return NULL;
	
    C->Ncell=Cin->Ncell;
    
    C->data=(char **)calloc(C->Ncell+1,sizeof(char*));
    if(C->data==NULL)
    {
        free(C);
        return NULL;
    }


    for(i=0;i<=C->Ncell;i++)
    {
        if(Cin->data[i]==NULL)
        {
            C->data[i]=NULL;
        }
        else
        {
            C->data[i]=(char *)calloc(strlen(Cin->data[i])+1,sizeof(char));
            if(C->data[i]==NULL)
            {
                for(j=0;j<i;j++)    free(C->data[j]);
                free(C->data);
                free(C);
                return NULL;
            }
            sprintf(C->data[i],"%s",Cin->data[i]);
        }
    }

	return C;
}


/** \fn int pds_cell_string_write(PdsCellString *C,int id,const char* string_data)
 *  \brief Escribe un dato de tipo char* en el elemento id del arreglo de células.
 *  \param[in,out] C El arreglo de células a escribir.
 *  \param[in] id Posición en donde escribir el dato.
 *  \param[in] string_data La cadena a escribir.
 *  \return TRUE si todo fue bien o FALSE si no. Retorna FALSE si el id está fuera 
 *  de rango.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_write(PdsCellString *C,int id,const char* string_data)
{
    char *tmp=NULL;

	if(C==NULL)		return FALSE;

    if( (id>=C->Ncell)||(id<0) )    return FALSE;
    
    tmp=(char *)calloc(strlen(string_data)+1,sizeof(char));
    if(tmp==NULL)   FALSE;
    sprintf(tmp,"%s",string_data);

    free(C->data[id]);
    C->data[id]=tmp;

    return TRUE;
}


/** \fn int pds_cell_string_all_printf(const PdsCellString *C)
 *  \brief Imprime en pantalla los datos del arreglo de células de tipo puntero 
 *  PdsCellString. Escribe un dato por cada linea.
 *  \param[in] C El arreglo de células a imprimir en pantalla.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_all_printf(const PdsCellString *C)
{
    int i;
	if(C==NULL)		return FALSE;

    for(i=0;i<C->Ncell;i++)
    {
	    printf("%s\n",C->data[i]);
    }
	
	return TRUE;
}


/** \fn int pds_cell_string_all_fprintf(FILE* fd,const PdsCellString *C)
 *  \brief Imprime en el descriptor de fichero los datos del arreglo de células 
 *  de tipo puntero PdsCellString. Escribe un dato por cada linea.
 *  \param[in,out] fd El descriptor de fichero sobre el cual escribir.
 *  \param[in] C El arreglo de células a escribir.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_all_fprintf(FILE* fd,const PdsCellString *C)
{
    int i;
	if(C==NULL)     return FALSE;
	if(fd==NULL)	return FALSE;

    for(i=0;i<C->Ncell;i++)
    {
	    fprintf(fd,"%s\n",C->data[i]);
    }
	
	return TRUE;
}

/** \fn int pds_cell_string_save(FILE* fd,const PdsCellString *C)
 *  \brief Imprime en el descriptor de fichero los datos del arreglo de células 
 *  de tipo puntero PdsCellString. Escribe un dato por cada linea.
 *  \param[in] pathname La dirección de fichero sobre el cual escribir.
 *  \param[in] C El arreglo de células a escribir.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_save(const char* pathname,const PdsCellString *C)
{
    int i;
    FILE *fd=NULL;

	if(C==NULL)		    return FALSE;
	if(pathname==NULL)  return FALSE;

    fd=fopen(pathname,"w");
    if(fd==NULL)    return FALSE;

    for(i=0;i<C->Ncell;i++)
    {
	    fprintf(fd,"%s\n",C->data[i]);
    }

    fclose(fd);
	return TRUE;
}


/** \fn void pds_cell_string_free(PdsCellString *C)
 *  \brief Libera un arreglo de células de tipo puntero PdsCellString. 
 *  \param[in,out] C El arreglo de células a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsCellStringGroup
 */
void pds_cell_string_free(PdsCellString *C)
{
	int i;

	if(C!=NULL)
	{
        for(i=0;i<C->Ncell;i++) free(C->data[i]);
        free(C->data);
        free(C);        
	}
}


/** \fn void pds_cell_string_destroy(PdsCellString **C)
 *  \brief Libera un arreglo de células de tipo puntero PdsCellString y carga con NULL.
 *  \param[in,out] C El arreglo de células a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsCellStringGroup
 */
void pds_cell_string_destroy(PdsCellString **C)
{
	int i;

	if(C!=NULL)
	if((*C)!=NULL)
	{
        for(i=0;i<(*C)->Ncell;i++) free((*C)->data[i]);
        free((*C)->data);
        free(*C);     
		*C=NULL;   
	}
}




/** \fn int pds_cell_string_add(PdsCellString *C,const char* string_data)
 *  \brief Agrega un elemento en la ultima posición vacía(NULL) del arreglo de células. 
 *  O agrega un nuevo registro de memoria.
 *  \param[in,out] C El arreglo de células para agregar datos.
 *  \param[in] string_data La cadena a escribir.
 *  \return TRUE si todo fue bien o FALSE si no. Si string_data==NULL 
 *  no es considerado error y se finaliza la función sin hacer ningún cambio.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_add(PdsCellString *C,const char* string_data)
{
    int i,id,dat=TRUE;
    char **temp=NULL;

	if(C==NULL)             return FALSE;
	if(string_data==NULL)   return TRUE;

    id=pds_cell_string_get_last_empty(C);

    if(id<0)        return FALSE;
    if(id>C->Ncell) return FALSE;

    if(id==C->Ncell)
    {
        temp=(char **)calloc(C->Ncell+2,sizeof(char*));
        if(temp==NULL)   return FALSE;

        for(i=0;i<=C->Ncell;i++)    temp[i]=C->data[i];
        temp[C->Ncell+1]=NULL;

        temp[C->Ncell]=(char *)calloc(strlen(string_data)+1,sizeof(char));
        if(temp[C->Ncell]==NULL)
        {
            free(temp);
            return FALSE;
        }
        sprintf(temp[C->Ncell],"%s",string_data);

        free(C->data);
        C->data=temp;
        C->Ncell=C->Ncell+1;
    }
    else
    {
        dat=pds_cell_string_write(C,id,string_data);
    }
    
    return dat;
}



/** \fn int pds_cell_string_add_cell_string(PdsCellString *C,const PdsCellString *Cin)
 *  \brief Agrega un conjunto de elementos a partir de la última posición 
 *  vacía (NULL) del arreglo de células (Y/O agrega nuevos registros de memoria).
 *  Si el sumando Cin es NULL no agrega nada y sale sin decir error y sin alterar 
 *  el acumulador C.
 *
 *  ESTA FUNCION DEBE SER HECHA DE NUEVO SIN USAR pds_cell_string_add PARA 
 *  NO LIBERAR Y RESERVAR MEMORIA TAN ALEGREMENTE. ASSI SE GANARIA VELOCIDAD.
 *  \param[in,out] C El arreglo de células a agregar datos.
 *  \param[in] Cin El arreglo de células de donde se leen los datos.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_add_cell_string(PdsCellString *C,const PdsCellString *Cin)
{
    int i,dat;
    PdsCellString *Ctmp=NULL;

	if(C==NULL)       return FALSE;
	if(Cin==NULL)     return TRUE;

    Ctmp=pds_cell_string_copy_new(C);

    for(i=0;i<Cin->Ncell;i++)
    {
        dat=pds_cell_string_add(Ctmp,Cin->data[i]);
        if(dat==FALSE)  
        {
            pds_cell_string_free(Ctmp);
            return FALSE;
        }
    }

    free(C->data);
    C->data=Ctmp->data;
    C->Ncell=Ctmp->Ncell;
    free(Ctmp);

    return TRUE;
}

/** \fn int pds_cell_string_read(const PdsCellString* C,int id,char** string_data)
 *  \brief Lee una cadena de texto, en la posición id del arreglo de células.
 *  \param[in] C El arreglo de células a analizar.
 *  \param[in] id La posición a leer.
 *  \param[out] string_data La dirección de la cadena a leer (NO debe liberarse esta 
 *  dirección de memoria).
 *  \return TRUE si todo fue bien o FALSE si no.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_read(const PdsCellString* C,int id,char** string_data)
{
    char *tmp=NULL;

	if(C==NULL)		                return FALSE;
	if(string_data==NULL)           return FALSE;
    if( (id>=C->Ncell)||(id<0) )    return FALSE;

    (*string_data)=C->data[id];

    return TRUE;
}



/** \fn int pds_cell_string_get_ncell(const PdsCellString *C)
 *  \brief Cuenta la cantidad de células incluyendo las vacías.
 *  \param[in] C El arreglo de células a analizar.
 *  \return Retorna la cantidad de células o un número menor que cero 
 *  en caso de error.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_get_ncell(const PdsCellString *C)
{
	if(C==NULL)		                return -1;
	return C->Ncell;
}



/** \fn int pds_cell_string_get_last_empty(const PdsCellString *C)
 *  \brief Retorna el id de la última célula libre (osea con NULL).
 *  \param[in] C El arreglo de células a analizar.
 *  \return Retorna el id de la última célula libre, o un número menor que cero 
 *  en caso de error. En caso de que el arreglo esté lleno la función retornará
 *  C->Ncell.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_get_last_empty(const PdsCellString *C)
{
    int i,id;
	if(C==NULL)		                return -1;

    id=C->Ncell;
    for(i=0;i<C->Ncell;i++)
    {
        if(C->data[C->Ncell-1-i]==NULL)     id=C->Ncell-1-i;
        else                                break;
    }
	return id;
}

/** \fn int pds_cell_string_find(const PdsCellString *C,const char *str)
 *  \brief Retorna el id de la primera ocurrencia de la cadena str en el arreglo de 
 *  células C.
 *  \param[in] C El arreglo de células a analizar.
 *  \param[in] str La cadena a buscar.
 *  \return Retorna el id de la primera ocurrencia de la cadena str, o un número 
 *  menor que cero en caso de error.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_find(const PdsCellString *C,const char *str)
{
    int i,id=-1;
	if(C==NULL)		                return id;
	if(str==NULL)                   return id;

    for(i=0;i<C->Ncell;i++)
    {
        if(strcmp(C->data[i],str)==0)
            return i;
    }
	return id;
}


/** \fn int pds_cell_string_findci(const PdsCellString *C,const char *str)
 *  \brief Retorna el id de la primera ocurrencia de la cadena str en el arreglo de 
 *  células C. La función es case-insensitive.
 *  \param[in] C El arreglo de células a analizar.
 *  \param[in] str La cadena a buscar.
 *  \return Retorna el id de la primera ocurrencia de la cadena str, o un número 
 *  menor que cero en caso de error.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_findci(const PdsCellString *C,const char *str)
{
    int i,id=-1;
	if(C==NULL)		                return id;
	if(str==NULL)                   return id;

    for(i=0;i<C->Ncell;i++)
    {
        if(pds_strcicmp(C->data[i],str)==0)
            return i;
    }
	return id;
}



/** \fn PdsCellString *pds_cell_string_new_load_data(const char*filepath)
 *  \brief Crea un arreglo de células de tipo PdsCellString con los datos de cada 
 *  linea del archivo.
 *  \param[in] filepath La dirección del archivo de donde se cargarán los datos.
 *  \return Un puntero a una estructura de tipo PdsCellString.
 *  \ingroup PdsCellStringGroup
 */
PdsCellString *pds_cell_string_new_load_data(const char*filepath)
{
	PdsCellString *C=NULL;
    char *line;
    int dat,i;
    FILE *fd=NULL;

    if (filepath==NULL)         return NULL;
    if (strlen(filepath)==0)    return NULL;

	C=pds_cell_string_new(0);
    if(C==NULL) return NULL;
	
    fd=fopen(filepath,"r");
    if(fd==NULL)    
    {
        pds_cell_string_free(C);
        return NULL;
    }

    do{
        line=pds_read_line(fd);
        dat=pds_cell_string_add(C,line);
        free(line);
    }while(feof(fd)==0);

    fclose(fd);

	return C;
}



/** \fn int pds_cell_string_remove(PdsCellString *C,int id)
 *  \brief Remueve la célula de posición id.
 *
 *  \param[in,out] C El arreglo de células a remover datos.
 *  \param[in] id El id de la célula a remover.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  En caso de error no modifica el contenido de C.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_remove(PdsCellString *C,int id)
{
    int i;
    char **data=NULL;

	if(C==NULL)         return FALSE;
	if(id<0)            return FALSE;
	if(id>=C->Ncell)    return FALSE;

    data=(char**)calloc(C->Ncell,sizeof(char*));
    if(data==NULL)  return FALSE;
    
    for(i=0;i<id;i++)   data[i]=C->data[i];

    free(C->data[id]);

    for(i=(id+1);i<C->Ncell;i++)   data[i-1]=C->data[i];
    data[C->Ncell-1]=NULL;

    C->Ncell=C->Ncell-1;
    free(C->data);
    C->data=data;

    return TRUE;
}


/** \fn int pds_cell_string_filter_orinc(PdsCellString *C,const PdsCellString *pattern)
 *  \brief Remueve las células que no cumplen con por lo menos un patrón 
 *  especificado por pattern.
 *
 *  \param[in,out] C El arreglo de células a remover datos.
 *  \param[in] pattern El patrón de búsqueda.
 *  \return TRUE si todo fue bien o FALSE si no.
 *  En caso de error el contenido de C no es modificado. Si pattern es NULL no 
 *  es considerado error y se retorn TRUE sin hacer nada.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_filter_orinc(PdsCellString *C,const PdsCellString *pattern)
{
    PdsCellString *C0=NULL;
    int i,j,dat,fulfill_any_pattern;

    if(C==NULL)         return  FALSE;
    if(pattern==NULL)   return  TRUE;

    C0=pds_cell_string_copy_new(C);
    if(C0==NULL)    return FALSE;
    

    for(i=0;i<C0->Ncell;i++)
    {
        fulfill_any_pattern=FALSE;
        for(j=0;j<pattern->Ncell;j++)
        {
            if(fnmatch(pattern->data[j], C0->data[i], FNM_NOESCAPE)==0)
            {
                fulfill_any_pattern=TRUE;
                break;
            }
        }

        if(!fulfill_any_pattern)
        {
            dat=pds_cell_string_remove(C0,i);
            if(dat==FALSE)  return FALSE;

            i=i-1;
        }
    }



    for(i=0;i<C->Ncell;i++) free(C->data[i]);
    free(C->data);

    C->data=C0->data;
    C->Ncell=C0->Ncell;
    free(C0);

    return TRUE;
    
}


/** \fn int pds_cell_string_findci_and_add(PdsCellString *C,const char* string_data)
 *  \brief Agrega un nuevo registro de memoria si la cadena string_data no existe 
 *  en la estructura *C. Si esta ya existe la sobre escribe pues esta función no
 *  diferencia mayúsculas y minúsculas.
 *  \param[in,out] C El arreglo de células para agregar datos.
 *  \param[in] string_data La cadena a escribir.
 *  \return TRUE si todo fue bien o FALSE si no. Si string_data==NULL 
 *  no es considerado error y se finaliza la función sin hacer ningún cambio.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_findci_and_add(PdsCellString *C,const char* string_data)
{
    int i,id=-1;
    int dat;
    int RET=TRUE;
	if(C==NULL)		                return FALSE;
	if(string_data==NULL)           return TRUE;

    for(i=0;i<C->Ncell;i++)
    {
        if(pds_strcicmp(C->data[i],string_data)==0)
        {
            id=i;
            dat=pds_cell_string_write(C,i,string_data);
            if(dat==FALSE)
            {
                fprintf(stderr,"ERROR writing the data[%d]=%s",id,string_data);
                RET=FALSE;
            }
        }
    }

    if(id==-1)
    {
        dat=pds_cell_string_add(C,string_data);
        if(dat==FALSE)
        {
            fprintf(stderr,"ERROR adding the data=%s",string_data);
            return FALSE;
        }
    }

    return RET;
}


/** \fn int pds_cell_string_find_and_remove(const PdsCellString *C,const char *str)
 *  \brief Procura todas las células con cadenas igual a str y las remueve.
 *  \param[in,out] C El arreglo de células a analizar.
 *  \param[in] str La cadena a buscar.
 *  \return Retorna TRUE si todo fue bien o FALSE si no. Si C o str son nulos
 *  entonces retorna FALSE sin hacer nada sobre C.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_find_and_remove(PdsCellString *C,const char *str)
{
    int id;
    int dat;

    if(C==NULL)     return FALSE;
    if(str==NULL)   return FALSE;

    do{
        id=pds_cell_string_find(C,str);
        if(id>=0)
        {
            dat=pds_cell_string_remove(C,id);
            if(dat==FALSE)
            {
                fprintf(stderr,"ERROR removing the data[%d]=%s",id,str);
                fprintf(stderr,"Trying newly ...");
                //return FALSE;
            }
        }
    }while(id>=0);

    return TRUE;
}


/** \fn int pds_cell_string_findci_and_remove(const PdsCellString *C,const char *str)
 *  \brief Procura todas las células con cadenas igual a str y las remueve.
 *  \param[in,out] C El arreglo de células a analizar.
 *  La función es case-insensitive.
 *  \param[in] str La cadena a buscar.
 *  \return Retorna TRUE si todo fue bien o FALSE si no. Si C o str son nulos
 *  entonces retorna FALSE sin hacer nada sobre C.
 *  \ingroup PdsCellStringGroup
 */
int pds_cell_string_findci_and_remove(PdsCellString *C,const char *str)
{
    int id;
    int dat;

    if(C==NULL)     return FALSE;
    if(str==NULL)   return FALSE;

    do{
        id=pds_cell_string_findci(C,str);
        if(id>=0)
        {
            dat=pds_cell_string_remove(C,id);
            if(dat==FALSE)
            {
                fprintf(stderr,"ERROR removing the data[%d]=%s",id,str);
                fprintf(stderr,"Trying newly ...");
                //return FALSE;
            }
        }
    }while(id>=0);

    return TRUE;
}

