/*
 * pdsstring.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
 #include <ctype.h>
#include <pds/pdsstring.h>
#include <pds/pdscstring.h>

/** \fn char* pds_sprintf(const char *format, ...)
 *  \brief Similar a sprintf pero esta retorna un puntero a una nueva cadena con 
 *  el texto.
 *
 *  \param[in] format Formato de texto de salida.
 *  \param[in] ... Otros argumentos.
 *  \return Retorna un puntero a una nueva cadena con el texto.
 *  \ingroup PdsStringGroup
 */
char* pds_sprintf(const char *format, ...)
{
    char *char_string=NULL;
    char *p=NULL;
    int N=-1;
    int val;
    va_list args;               /* points to each unnamed arg in turn */

    va_start( args, format );   /* make args point to 1st unnamed arg */
    N=strlen(format);
    for (p = (char*)format; *p; p++) 
    {
        if (*p == '%') 
        {
            p++;
            if(*p=='s')         {N = N+strlen(va_arg(args, char *));}
            else if(*p=='c')    {val = va_arg(args, int); N=N+1;}
            else                {val = va_arg(args, int); N=N+21;}
        }
    }
    va_end( args );             /* clean up when done */

    if(N==-1)
    {
        return NULL;
    }

    char_string=(char*)calloc(N+1,sizeof(char));
    if(char_string==NULL)
    {
        return NULL;
    }


    va_start( args, format );   /* make args point to 1st unnamed arg */
    vsprintf( char_string, format, args );
    va_end( args );             /* clean up when done */


    char_string=(char*)realloc(char_string,strlen(char_string)+1);

    return char_string;
}


/** \fn PdsCellString *pds_strsep(const char *str, const char *delim)
 *  \brief Esta función es similar a la función strsep, donde la cadena str es 
 *  dividida usando algunos de los delimitadores espesificados en delim. Los 
 *  pedazos son cargados en una estructura PdsCellString* retornada por la función.
 *  \param[in] str Cadena a dividir, esta cadena no sera modificada.
 *  \param[in] delim Delimitadores a usar.
 *  \return Retorna un puntero a una nueva estructura de tipo PdsCellString *,
 *  con los tokens de la división.
 *  \ingroup PdsStringGroup
 */
PdsCellString *pds_strsep(const char *str, const char *delim)
{
    char *str0 =NULL; 
    char *token=NULL;
    char *str1=NULL;    
    PdsCellString *Tok=NULL;
    int dat;

    if(str==NULL)   return NULL;
    if(delim==NULL) return NULL;

    str0=strdup(str);
    if(str0==NULL)  return NULL;

    Tok=pds_cell_string_new(0);
    if(Tok==NULL)
    {
        return NULL;
    }

    str1=str0;
    do{
        token = strsep(&str0, delim);
        if(token!=NULL)
        {
            //if(strlen(token)>0) 
            {
                dat=pds_cell_string_add(Tok,token);
            }
            
        }
    }while(token!=NULL);

    free(str1);

    return Tok;
}


/** \fn int pds_strcicmp(char const *str0, char const *str1)
 *  \brief Esta función es similar a la función strcmp pero no distinguiendo 
 *  mayúsculas y minúsculas (Case Insensitive), donde si las cadenas son iguales
 *  la funcion retorna 0.
 *  \param[in] str0 Primera cadena a comparar.
 *  \param[in] str1 Segunda cadena a comparar.
 *  \return Retorna 0 si las cadenas son iguales (Case Insensitive).
 *  \ingroup PdsStringGroup
 */
int pds_strcicmp(char const *str0, char const *str1)
{
    int d;

    for (;; str0++, str1++) 
    {
        d = tolower(*str0) - tolower(*str1);
        if( (d != 0) || (*str0==0) )
            return d;
    }
}

