/*
 * pdsrecord.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsrecord.h
 *  \author Fernando Pujaico Rivera
 *  \date 23-01-2012
 * \brief Archivo de definición de una entrada de grabación de audio.
 *
 */
#ifndef __PDSRECORD_H__
#define __PDSRECORD_H__

#include <stdio.h>
#include <stdlib.h>
#include <pds/pdsra.h>
#include <pds/pdsdaglobal.h>
#include <alsa/asoundlib.h>

#ifdef __cplusplus
extern "C" {
#endif 

/** \defgroup PdsRecordGroup Funciones del módulo PdsRecord.
 * @{
 */

#ifndef TRUE
/*! \def TRUE
 *  Define la palabra TRUE equivalente con 1.
 *  \ingroup PdsRecordGroup
*/
	#define TRUE 1
#endif

#ifndef FALSE
/*! \def FALSE
 *  Define la palabra FALSE equivalente con 0.
 *  \ingroup PdsRecordGroup
*/
	#define FALSE 0
#endif

/*! \struct PdsRecord
 *  \brief La estructura tipo  PdsRecord .
 *  Esta estructura genera una metodo para grabar audio por la entrada de audio
 *  por defecto del sistema.
 *  Para usar incluir pds/pdsrecord.h.
 *  \ingroup PdsRecordGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct
{
	char		*buffer1B;
	short		*buffer2B;
	int		*buffer4B;

	PdsDaNatural	samples;

	PdsDaNatural	bits;
	PdsDaNatural	bytes;

	PdsDaNatural	freq;

	snd_pcm_t	*capture_handle;
}PdsRecord;


/** \fn PdsRecord *pds_record_new(PdsDaNatural *freq,PdsDaNatural *bits)
 *  \brief Crea una estructura para la grabación de audio digital.
 *  \param[in,out] freq Frecuencia de muestreo,si no es posible es cargada en freq
 *  la frecuencia mas próxima.
 *  \param[in,out] bits La cantidad de bits en el muestreo, si no es posible es cargada
 *  la cantidad de bits posible próxima o inmediata superior. Los valores posibles 
 *  son:8, 16,24 o32 bits.
 *  \return Un puntero a la estructura para la grabación de audio digital. o NULL
 *  en caso de error.
 *  \ingroup PdsRecordGroup
 */
PdsRecord *pds_record_new(PdsDaNatural *freq,PdsDaNatural *bits);


/** \fn int pds_record_set_samples_buffer(PdsRecord *A,PdsDaNatural samples)
 *  \brief Establece la cantidad de muestras en el buffer de lectura.
 *  \param[in,out] A estructura para la grabación de audio digital.
 *  \param[in] samples La cantidad de muestras en el buffer.
 *  \return TRUE si todo fue bien o FALSE si no. (ejem A==NULL o samples==0).
 *  \ingroup PdsRecordGroup
 */
int pds_record_set_samples_buffer(PdsRecord *A,PdsDaNatural samples);


/** \fn int pds_record_read(PdsRecord *A,PdsVector *X)
 *  \brief Lee los datos en el buffer de lectura. Los datos estarán comprendidos 
 *  entre [-1.0,1.0>.
 *  \param[in,out] A Estructura para la grabación de audio digital.
 *  \param[in] X Vector real donde se guardarán los datos del buffer.
 *  \return TRUE si todo fue bien o FALSE si no. (ejem A==NULL, X==NULL o el
 *  número de elementos de X es distinto a la cantidad de samples en el buffer.).
 *  \ingroup PdsRecordGroup
 */
int pds_record_read(PdsRecord *A,PdsVector *X);


/** \fn void pds_record_free(PdsRecord *A)
 *  \brief Libera la estructura para la grabación de audio digital.
 *  \param[in,out] A Estructura para la grabación de audio digital.
 *  \ingroup PdsRecordGroup
 */
void pds_record_free(PdsRecord *A);


/** \fn void pds_record_destroy(PdsRecord **A)
 *  \brief Libera la estructura para la grabación de audio digital, y carga con 
 *  NULL la estructura.
 *  \param[in,out] A Estructura para la grabación de audio digital.
 *  \ingroup PdsRecordGroup
 */
void pds_record_destroy(PdsRecord **A);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif 


#endif /* __PDSRECORD_H__ */
