/*
 * pdsplayback.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsplayback.h
 *  \author Fernando Pujaico Rivera
 *  \date 23-01-2012
 * \brief Archivo de definición de una salida de reproducción de audio.
 *
 */
#ifndef __PDSPLAYBACK_H__
#define __PDSPLAYBACK_H__

#include <stdio.h>
#include <stdlib.h>
#include <pds/pdsra.h>
#include <pds/pdsdaglobal.h>
#include <alsa/asoundlib.h>

#ifdef __cplusplus
extern "C" {
#endif 

/** \defgroup PdsPlaybackGroup Funciones del módulo PdsPlayback.
 * @{
 */

#ifndef TRUE
/*! \def TRUE
 *  Define la palabra TRUE equivalente con 1.
 *  \ingroup PdsPlaybackGroup
*/
	#define TRUE 1
#endif

#ifndef FALSE
/*! \def FALSE
 *  Define la palabra FALSE equivalente con 0.
 *  \ingroup PdsPlaybackGroup
*/
	#define FALSE 0
#endif

/*! \struct PdsPlayback
 *  \brief La estructura tipo  PdsPlayback .
 *  Esta estructura genera una metodo para reproducir audio por la salida de audio
 *  por defecto del sistema.
 *  Para usar incluir pds/pdsplayback.h.
 *  \ingroup PdsPlaybackGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct
{
	char		*buffer1B;
	short		*buffer2B;
	int		*buffer4B;

	PdsDaNatural	samples;

	PdsDaNatural	bits;
	PdsDaNatural	bytes;

	PdsDaNatural	freq;

	snd_pcm_t	*playback_handle;
}PdsPlayback;


/** \fn PdsPlayback *pds_playback_new(PdsDaNatural *freq,PdsDaNatural *bits)
 *  \brief Crea una estructura para la reproducción de audio digital.
 *  \param[in,out] freq Frecuencia de muestreo,si no es posible es cargada en freq
 *  la frecuencia mas próxima.
 *  \param[in,out] bits La cantidad de bits en el muestreo, si no es posible es cargada
 *  la cantidad de bits posible próxima o inmediata superior. Los valores posibles 
 *  son:8, 16,24 o32 bits.
 *  \return Un puntero a la estructura para la reproducción de audio digital. o NULL
 *  en caso de error.
 *  \ingroup PdsPlaybackGroup
 */
PdsPlayback *pds_playback_new(PdsDaNatural *freq,PdsDaNatural *bits);


/** \fn int pds_playback_set_samples_buffer(PdsPlayback *A,PdsDaNatural samples)
 *  \brief Establece la cantidad de muestras en el buffer de salida.
 *  \param[in,out] A estructura para la reproducción de audio digital.
 *  \param[in] samples La cantidad de muestras en el buffer.
 *  \return TRUE si todo fue bien o FALSE si no. (ejem A==NULL o samples==0).
 *  \ingroup PdsPlaybackGroup
 */
int pds_playback_set_samples_buffer(PdsPlayback *A,PdsDaNatural samples);


/** \fn int pds_playback_write(PdsPlayback *A,PdsVector *X)
 *  \brief Escribe los datos de X en el buffer de salida. Los datos estarán comprendidos 
 *  entre [-1.0,1.0>.
 *  \param[in,out] A Estructura para la reproducción de audio digital.
 *  \param[in] X Vector real de donde se leerán los datos del buffer.
 *  \return TRUE si todo fue bien o FALSE si no. (ejem A==NULL, X==NULL o el
 *  número de elementos de X es distinto a la cantidad de samples en el buffer.).
 *  \ingroup PdsPlaybackGroup
 */
int pds_playback_write(PdsPlayback *A,PdsVector *X);


/** \fn void pds_playback_free(PdsPlayback *A)
 *  \brief Libera la estructura para la reproducción de audio digital.
 *  \param[in,out] A Estructura para la reproducción de audio digital.
 *  \ingroup PdsPlaybackGroup
 */
void pds_playback_free(PdsPlayback *A);


/** \fn void pds_playback_destroy(PdsPlayback **A)
 *  \brief Libera la estructura para la reproducción de audio digital, y carga con 
 *  NULL la estructura.
 *  \param[in,out] A Estructura para la reproducción de audio digital.
 *  \ingroup PdsPlaybackGroup
 */
void pds_playback_destroy(PdsPlayback **A);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif 


#endif /* __PDSPLAYBACK_H__ */
