/*
 * pdscomplex.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdscomplex.h
 *  \author Fernando Pujaico Rivera
 *  \date 01-05-2011
 * \brief Archivo de definición de variable compleja.
 *
 */
#ifndef __PDSCOMPLEX_H__
#define __PDSCOMPLEX_H__


#include <pds/pdscnglobal.h>

#ifdef __cplusplus
extern "C" {
#endif 

/** \defgroup PdsComplexGroup Funciones del módulo PdsComplex.
 * @{
 */


/*! \struct PdsComplex
 *  \brief La estructura tipo  PdsComplex .
 *  Esta estructura genera unnumero complejo.
 *  Para usar incluir pdscomplex/pdscomplex.h.
 *  \ingroup PdsComplexGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! Parte real. */
	PdsCnReal Real;
	/*! Parte imaginaria. */
	PdsCnReal Imag;
}PdsComplex;



/*! \def PDS_CRECT(A,B)
 *  \brief Es equivalente a <b>(A+Bi)</b>.
 *  \ingroup PdsComplexGroup
*/
#define PDS_CRECT(A,B) pds_complex_rect(A,B)

/** \fn PdsComplex pds_complex_rect(PdsCnReal a,PdsCnReal b)
 *  \brief Crea la variable complejas. <b>a+bi</b>.
 *  \param[in] a Parte real.
 *  \param[in] b Parte imaginaria.
 *  \return La variable compleja.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_rect(PdsCnReal a,PdsCnReal b);



/*! \def PDS_CPOLAR(A,B)
 *  \brief Es equivalente a <b>(A*e^{Bi})</b>.
 *  \ingroup PdsComplexGroup
*/
#define PDS_CPOLAR(A,B) pds_complex_polar(A,B)

/** \fn PdsComplex pds_complex_polar(PdsCnReal a,PdsCnReal b)
 *  \brief Evalua de forma polar. <b>a*e^(bi)</b>.
 *  \param[in] a Módulo de la variable compleja.
 *  \param[in] b Ángulo de la variable compleja.
 *  \return El valor complejo equivalente.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_polar(PdsCnReal a,PdsCnReal b);



/*! \def PDS_CADDC(A,B)
 *  \brief Es equivalente a <b>(A+B)</b>.
 *  \ingroup PdsComplexGroup
*/
#define PDS_CADDC(A,B) pds_complex_add_complex(A,B)

/** \fn PdsComplex pds_complex_add_complex(PdsComplex a,PdsComplex b)
 *  \brief Suma dos variables complejas. <b>a+b</b>.
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable compleja.
 *  \return La suma de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_add_complex(PdsComplex a,PdsComplex b);



/*! \def PDS_CADDR(A,B)
 *  \brief Es equivalente a <b>(A+B)</b>.
 *  \ingroup PdsComplexGroup
*/
#define PDS_CADDR(A,B) pds_complex_add_real(A,B)

/** \fn PdsComplex pds_complex_add_real(PdsComplex a,PdsCnReal b)
 *  \brief Suma una variable complejas y una real. <b>a+b</b>.
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable real.
 *  \return La suma de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_add_real(PdsComplex a,PdsCnReal b);



/*! \def PDS_CADDI(A,B)
 *  \brief Es equivalente a <b>(A+Bi)</b>.
 *  \ingroup PdsComplexGroup
*/
#define PDS_CADDI(A,B) pds_complex_add_imag(A,B)

/** \fn PdsComplex pds_complex_add_imag(PdsComplex a,PdsCnReal b)
 *  \brief Suma una variable complejas y una imaginaria. <b>a+bi</b>.
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable imaginaria.
 *  \return La suma de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_add_imag(PdsComplex a,PdsCnReal b);



/*! \def PDS_CSUBC(A,B)
 *  \brief Es equivalente a <b>(A-B)</b>.
 *  \ingroup PdsComplexGroup
*/
#define PDS_CSUBC(A,B) pds_complex_sub_complex(A,B)

/** \fn PdsComplex pds_complex_sub_complex(PdsComplex a,PdsComplex b)
 *  \brief Resta dos variables complejas. <b>a-b</b>.
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable compleja.
 *  \return La resta de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_sub_complex(PdsComplex a,PdsComplex b);



/*! \def PDS_CSUBR(A,B)
 *  \brief Es equivalente a <b>(A-B)</b>.
 *  \ingroup PdsComplexGroup
*/
#define PDS_CSUBR(A,B) pds_complex_sub_real(A,B)

/** \fn PdsComplex pds_complex_sub_real(PdsComplex a,PdsCnReal b)
 *  \brief Resta una variable complejas y una real. <b>a-b</b>
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable real.
 *  \return La resta de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_sub_real(PdsComplex a,PdsCnReal b);



/*! \def PDS_CSUBI(A,B)
 *  \brief Es equivalente a <b>(A-Bi)</b>.
 *  \ingroup PdsComplexGroup
*/
#define PDS_CSUBI(A,B) pds_complex_sub_imag(A,B)

/** \fn PdsComplex pds_complex_sub_imag(PdsComplex a,PdsCnReal b)
 *  \brief Resta una variable complejas y una imaginaria. <b>a-bi</b>
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable imaginaria.
 *  \return La resta de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_sub_imag(PdsComplex a,PdsCnReal b);



/*! \def PDS_CMULC(A,B)
 *  \brief Es equivalente a <b>(A*B)</b>.
 *  \ingroup PdsComplexGroup
*/
#define PDS_CMULC(A,B) pds_complex_mul_complex(A,B)

/** \fn PdsComplex pds_complex_mul_complex(PdsComplex a,PdsComplex b)
 *  \brief Multiplica dos variables complejas. <b>a*b</b>.
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable compleja.
 *  \return La multiplicación de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_mul_complex(PdsComplex a,PdsComplex b);



/*! \def PDS_CMULR(A,B)
 *  \brief Es equivalente a <b>(A*B)</b>.
 *  \ingroup PdsComplexGroup
*/
#define PDS_CMULR(A,B) pds_complex_mul_real(A,B)

/** \fn PdsComplex pds_complex_mul_real(PdsComplex a,PdsCnReal b)
 *  \brief Multiplica una variable compleja y una real. <b>a*b</b>.
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable real.
 *  \return La multiplicación de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_mul_real(PdsComplex a,PdsCnReal b);



/*! \def PDS_CMULI(A,B)
 *  \brief Es equivalente a <b>(A*Bi)</b>.
 *  \ingroup PdsComplexGroup
*/
#define PDS_CMULI(A,B) pds_complex_mul_imag(A,B)

/** \fn PdsComplex pds_complex_mul_imag(PdsComplex a,PdsCnReal b)
 *  \brief Multiplica una variable compleja y una imaginaria. <b>a*bi</b>.
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable imaginaria.
 *  \return La multiplicación de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_mul_imag(PdsComplex a,PdsCnReal b);


/*! \def PDS_CDIVC(A,B)
 *  \brief Es equivalente a <b>(A/B)</b>.
 *  \ingroup PdsComplexGroup
*/
#define PDS_CDIVC(A,B) pds_complex_div_complex(A,B)

/** \fn PdsComplex pds_complex_div_complex(PdsComplex a,PdsComplex b)
 *  \brief Divide dos variables complejas. <b>a/b</b>
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable compleja.
 *  \return La división de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_div_complex(PdsComplex a,PdsComplex b);



/*! \def PDS_CDIVR(A,B)
 *  \brief Es equivalente a <b>(A/B)</b>.
 *  \ingroup PdsComplexGroup
*/
#define PDS_CDIVR(A,B) pds_complex_div_real(A,B)

/** \fn PdsComplex pds_complex_div_real(PdsComplex a,PdsCnReal b)
 *  \brief Divide una variable compleja y una real. <b>a/b</b>.
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable real.
 *  \return La división de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_div_real(PdsComplex a,PdsCnReal b);



/*! \def PDS_CDIVI(A,B)
 *  \brief Es equivalente a <b>(A/Bi)</b>.
 *  \ingroup PdsComplexGroup
*/
#define PDS_CDIVI(A,B) pds_complex_div_imag(A,B)

/** \fn PdsComplex pds_complex_div_imag(PdsComplex a,PdsCnReal b)
 *  \brief Divide una variable compleja y una imaginaria. <b>a/bi</b>.
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable imaginaria.
 *  \return La división de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_div_imag(PdsComplex a,PdsCnReal b);


/*! \def PDS_CINV(A)
 *  \brief Es equivalente a  <b>A^{-1}=conj(A)/|A|</b> tal que <b>|A|=sqrt(A.Real^2+A.Imag^2)</b>.
 *  \ingroup PdsComplexGroup
*/
#define PDS_CINV(A) pds_complex_inv(A)

/** \fn PdsComplex pds_complex_inv(PdsComplex a)
 *  \brief Evalua a^{-1}. <b>a^{-1}=conj(a)/|a|</b> tal que <b>|a|=sqrt(a.Real^2+a.Imag^2)</b>.
 *  \param[in] a Variable compleja.
 *  \return El a^{-1}.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_inv(PdsComplex a);


/*! \def PDS_CCONJ(A)
 *  \brief Es equivalente a  <b>b=conj(A)=A.Real-A.Imag i</b>.
 *  \ingroup PdsComplexGroup
*/
#define PDS_CCONJ(A) pds_complex_conj(A)

/** \fn PdsComplex pds_complex_conj(PdsComplex a)
 *  \brief Evalua el conjugado de a. <b>b=conj(a)=a.Real-a.Imag i</b>.
 *  \param[in] a Variable compleja.
 *  \return El conj(a).
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_conj(PdsComplex a);


/*! \def PDS_CEXP(B)
 *  \brief Es equivalente a <b>e^B=e^{B.Real}(cos(B.Imag)+sin(B.Imag)i)</b>.
 *  \ingroup PdsComplexGroup
*/
#define PDS_CEXP(B) pds_complex_exp(B)

/** \fn PdsComplex pds_complex_exp(PdsComplex a)
 *  \brief Evalua el e^a. <b>e^a=e^{a.Real}(cos(a.Imag)+sin(a.Imag)i)</b>.
 *  \param[in] a Variable compleja.
 *  \return El e^a.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_exp(PdsComplex a);


/*! \def PDS_CLOG(B)
 *  \brief Es equivalente a <b>log(B)=log(|B|)+acos(B.Real/M)i</b>.
 *  \ingroup PdsComplexGroup
*/
#define PDS_CLOG(B) pds_complex_log(B)

/** \fn PdsComplex pds_complex_log(PdsComplex a)
 *  \brief Evalua el log(a). <b>log(a)=log(|a|)+acos(a.Real/M)i</b>.
 *  \param[in] a Variable compleja.
 *  \return El e^a.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_log(PdsComplex a);


/*! \def PDS_CCIS(B)
 *  \brief Es equivalente a <b>CIS(B)=COS(B)+i SIN(B)</b>.
 *  \ingroup PdsComplexGroup
*/
#define PDS_CCIS(B) pds_complex_cis(B)

/** \fn PdsComplex pds_complex_cis(PdsCnReal b)
 *  \brief Evalua cis(b). <b>cos(b)+i sin(b)=cis(b)</b>.
 *  \param[in] b Variable real.
 *  \return El cis(b).
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_cis(PdsCnReal b);



/*! \def PDS_CNORM(A)
 *  \brief Es equivalente a <b>|A|=sqrt(A.Real^2+A.Imag^2)</b>.
 *  \ingroup PdsComplexGroup
*/
#define PDS_CNORM(A) pds_complex_norm(A)

/** \fn PdsCnReal pds_complex_norm(PdsComplex a)
 *  \brief Evalua |a|. <b>|a|=sqrt(a.Real^2+a.Imag^2)</b>.
 *  \param[in] a Variable compleja.
 *  \return El |a|.
 *  \ingroup PdsComplexGroup
 */
PdsCnReal pds_complex_norm(PdsComplex a);



/*! \def PDS_CNORM2(A)
 *  \brief Es equivalente a <b>|A|^2=(A.Real^2+A.Imag^2)</b>.
 *  \ingroup PdsComplexGroup
*/
#define PDS_CNORM2(A) pds_complex_norm2(A)

/** \fn PdsCnReal pds_complex_norm2(PdsComplex a)
 *  \brief Evalua |a|^2. <b>|a|^2=(a.Real^2+a.Imag^2)</b>.
 *  \param[in] a Variable compleja.
 *  \return El |a|^2.
 *  \ingroup PdsComplexGroup
 */
PdsCnReal pds_complex_norm2(PdsComplex a);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif 


#endif /* __PDSCOMPLEX_H__ */
