/*
 * pdsquaternion.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

#include <math.h>
#include <pds/pdsquaternion.h>


////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsQuaternion                                             ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsQuaternion pds_quaternion_rect(PdsCnReal w,PdsCnReal x,PdsCnReal y,PdsCnReal z)
 *  \brief Crea la variable complejas. <b>q=w+xi+yi+zi</b>.
 *
 *  \f[ q=w+xi+yj+zk\f]
 *
 *  \param[in] w Parte real.
 *  \param[in] x Parte imaginaria vinculada a i.
 *  \param[in] y Parte imaginaria vinculada a j.
 *  \param[in] z Parte imaginaria vinculada a k.
 *  \return Un quaternio.
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_rect(PdsCnReal w,PdsCnReal x,PdsCnReal y,PdsCnReal z)
{
	PdsQuaternion c;
	c.W=w;
	c.X=x;
	c.Y=y;
	c.Z=z;
	return c;
}


/** \fn PdsQuaternion pds_quaternion_polar(PdsCnReal a,PdsCnReal b,PdsCnReal c,PdsCnReal d)
 *  \brief Evalua de forma polar. <b>ae^{bi+cj+dk}</b>.
 *  
 *  \f[\sqrt{b^2+c^2+d^2}=M\f]
 *  \f[a~e^{bi+cj+dk}=a \left(cos(M)+sin(M)\frac{bi+cj+dk}{M}\right)\f]
 *
 *  \param[in] a parametro 1 del quaternio.
 *  \param[in] b parametro 2 del quaternio.
 *  \param[in] c parametro 3 del quaternio.
 *  \param[in] d parametro 4 del quaternio.
 *  \return El valor complejo equivalente.
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_polar(PdsCnReal a,PdsCnReal b,PdsCnReal c,PdsCnReal d)
{
	PdsQuaternion q;
    PdsCnReal M;

    M=sqrt(b*b+c*c+d*d);

	q.W=a*cos(M);
	q.X=a*sin(M)*b/M;
	q.Y=a*sin(M)*c/M;
	q.Z=a*sin(M)*d/M;
	return q;
}


/** \fn PdsQuaternion pds_quaternion_add_quaternion(PdsQuaternion a,PdsQuaternion b)
 *  \brief Suma dos quaternios. <b>a+b</b>.
 *  \param[in] a Quaternio.
 *  \param[in] b Quaternio.
 *  \return La suma de ambas variables.
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_add_quaternion(PdsQuaternion a,PdsQuaternion b)
{
	PdsQuaternion c;
	c.W=a.W+b.W;
	c.X=a.X+b.X;
	c.Y=a.Y+b.Y;
	c.Z=a.Z+b.Z;
	return c;
}


/** \fn PdsQuaternion pds_quaternion_add_real(PdsQuaternion a,PdsCnReal b)
 *  \brief Suma un quaternio y una real. <b>a+b</b>.
 *  \param[in] a Quaternio.
 *  \param[in] b Variable real.
 *  \return La suma de ambas variables.
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_add_real(PdsQuaternion a,PdsCnReal b)
{
	PdsQuaternion c;
	c.W=a.W+b;
	c.X=a.X;
	c.Y=a.Y;
	c.Z=a.Z;
	return c;
}


/** \fn PdsQuaternion pds_quaternion_sub_quaternion(PdsQuaternion a,PdsQuaternion b)
 *  \brief Resta dos quaternios. <b>a-b</b>.
 *  \param[in] a Quaternio.
 *  \param[in] b Quaternio.
 *  \return La resta de ambas variables.
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_sub_quaternion(PdsQuaternion a,PdsQuaternion b)
{
	PdsQuaternion c;
	c.W=a.W-b.W;
	c.X=a.X-b.X;
	c.Y=a.Y-b.Y;
	c.Z=a.Z-b.Z;
	return c;
}


/** \fn PdsQuaternion pds_quaternion_sub_real(PdsQuaternion a,PdsCnReal b)
 *  \brief Resta un quaternio y una real. <b>a-b</b>
 *  \param[in] a Quaternio.
 *  \param[in] b Variable real.
 *  \return La resta de ambas variables.
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_sub_real(PdsQuaternion a,PdsCnReal b)
{
	PdsQuaternion c;
	c.W=a.W-b;
	c.X=a.X;
	c.Y=a.Y;
	c.Z=a.Z;
	return c;
}


/** \fn PdsQuaternion pds_quaternion_mul_quaternion(PdsQuaternion a,PdsQuaternion b)
 *  \brief Multiplica dos quaternios. <b>a*b</b>.
 *  \param[in] a Quaternio.
 *  \param[in] b Quaternio.
 *  \return La multiplicación de ambas variables.
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_mul_quaternion(PdsQuaternion a,PdsQuaternion b)
{
	PdsQuaternion c;
	c.W = a.W*b.W - a.X*b.X - a.Y*b.Y - a.Z*b.Z;
	c.X = a.W*b.X + a.X*b.W + a.Y*b.Z - a.Z*b.Y;
	c.Y = a.W*b.Y - a.X*b.Z + a.Y*b.W + a.Z*b.X;
	c.Z = a.W*b.Z + a.X*b.Y - a.Y*b.X + a.Z*b.W;
	return c;
}


/** \fn PdsQuaternion pds_quaternion_mul_real(PdsQuaternion a,PdsCnReal b)
 *  \brief Multiplica un quaternio y una real. <b>a*b</b>.
 *  \param[in] a Quaternio.
 *  \param[in] b Variable real.
 *  \return La multiplicación de ambas variables.
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_mul_real(PdsQuaternion a,PdsCnReal b)
{
	PdsQuaternion c;
	c.W=a.W*b;
	c.X=a.X*b;
	c.Y=a.Y*b;
	c.Z=a.Z*b;
	return c;
}



/** \fn PdsQuaternion pds_quaternion_div_quaternion(PdsQuaternion a,PdsQuaternion b)
 *  \brief Divide dos quaternios. <b>a/b</b>
 *  \param[in] a Quaternio.
 *  \param[in] b Quaternio.
 *  \return La división de ambas variables.
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_div_quaternion(PdsQuaternion a,PdsQuaternion b)
{
	PdsQuaternion c;
	PdsCnReal M;
	M=b.W*b.W+b.X*b.X+b.Y*b.Y+b.Z*b.Z;

	c.W = (+a.W*b.W + a.X*b.X + a.Y*b.Y + a.Z*b.Z)/M;
	c.X = (-a.W*b.X + a.X*b.W - a.Y*b.Z + a.Z*b.Y)/M;
	c.Y = (-a.W*b.Y + a.X*b.Z + a.Y*b.W - a.Z*b.X)/M;
	c.Z = (-a.W*b.Z - a.X*b.Y + a.Y*b.X + a.Z*b.W)/M;
	return c;
}


/** \fn PdsQuaternion pds_quaternion_div_real(PdsQuaternion a,PdsCnReal b)
 *  \brief Divide un quaternio y una real. <b>a/b</b>.
 *  \param[in] a Quaternio.
 *  \param[in] b Variable real.
 *  \return La división de ambas variables.
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_div_real(PdsQuaternion a,PdsCnReal b)
{
	PdsQuaternion c;
	c.W=a.W/b;
	c.X=a.X/b;
	c.Y=a.Y/b;
	c.Z=a.Z/b;
	return c;
}


/** \fn PdsQuaternion pds_quaternion_inv(PdsQuaternion a)
 *  \brief Retona el inverso de un quaternio a. <b>b=a^{-1}</b>.
 * 
 *  \f[\sqrt{W^2+X^2+Y^2+Z^2}=|a|\f]
 *  \f[b=\frac{conj(a)}{|a|^2}\f]
 *
 *  \param[in] a Quaternio.
 *  \return el inverso de a.
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_inv(PdsQuaternion a)
{
	PdsQuaternion c;
    PdsCnReal Q;

    Q=sqrt(a.W*a.W + a.X*a.X + a.Y*a.Y + a.Z*a.Z);

	c.W=a.W/Q;
	c.X=-a.X/Q;
	c.Y=-a.Y/Q;
	c.Z=-a.Z/Q;
	return c;
}



/** \fn PdsQuaternion pds_quaternion_unit(PdsQuaternion a)
 *  \brief Retona un quaternio unitario de un quaternio a. <b>b=a/|a|</b>.
 * 
 *  \f[\sqrt{W^2+X^2+Y^2+Z^2}=|a|\f]
 *  \f[b=\frac{a}{|a|}\f]
 *
 *  \param[in] a Quaternio.
 *  \return el quaternio unitario de a.
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_unit(PdsQuaternion a)
{
	PdsQuaternion c;
    PdsCnReal Q;

    Q=sqrt(a.W*a.W + a.X*a.X + a.Y*a.Y + a.Z*a.Z);

	c.W=a.W/Q;
	c.X=a.X/Q;
	c.Y=a.Y/Q;
	c.Z=a.Z/Q;
	return c;
}


/** \fn PdsQuaternion pds_quaternion_conj(PdsQuaternion a)
 *  \brief Retona el conjugado de un quaternio. <b>b=a.W-a.X i-a.Y j-a.Z k</b>.
 *
 *  \f[a=W+Xi+Yj+Zk\f]
 *  \f[b=conj(a)=W-Xi-Yj-Zk\f]
 *
 *  \param[in] a Quaternio.
 *  \return el conjgado de a.
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_conj(PdsQuaternion a)
{
	PdsQuaternion c;
	c.W=a.W;
	c.X=-a.X;
	c.Y=-a.Y;
	c.Z=-a.Z;
	return c;
}


/** \fn PdsQuaternion pds_quaternion_exp(PdsQuaternion q)
 *  \brief Evalua <b>exp(q),q=W+Xi+Yj+Zk</b>.
 *
 *  \f[\sqrt{X^2+Y^2+Z^2}=M\f]
 *  \f[e^{q}=e^W \left(cos(M)+sin(M)\frac{Xi+Yj+Zk}{M}\right)\f]
 *
 *  \param[in] q Quaternio.
 *  \return El exp(q).
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_exp(PdsQuaternion q)
{
	PdsQuaternion c;

    PdsCnReal M,exp_qW,cos_M,exp_qW_sin_M;

    M=sqrt(q.X*q.X+q.Y*q.Y+q.Z*q.Z);
    exp_qW=exp(q.W);
    cos_M=cos(M);
    exp_qW_sin_M=exp_qW*sin(M);

	c.W=exp_qW*cos_M;
	c.X=exp_qW_sin_M*q.X/M;
	c.Y=exp_qW_sin_M*q.Y/M;
	c.Z=exp_qW_sin_M*q.Z/M;
	return c;
}

/** \fn PdsQuaternion pds_quaternion_log(PdsQuaternion q)
 *  \brief Evalua <b>ln(q),q=W+Xi+Yj+Zk</b>.
 *
 *  \f[\sqrt{W^2+X^2+Y^2+Z^2}=|q|\f]
 *  \f[\sqrt{X^2+Y^2+Z^2}=V\f]
 *  \f[e^{q}=ln(|q|)+arccos\left(\frac{W}{|q|}\right)\frac{Xi+Yj+Zk}{V} \f]
 *
 *  \param[in] q Quaternio.
 *  \return El ln(q).
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_log(PdsQuaternion q)
{
	PdsQuaternion c;

    PdsCnReal V,Q,acs;

    Q=sqrt(q.W*q.W+q.X*q.X+q.Y*q.Y+q.Z*q.Z);

    V=sqrt(q.X*q.X+q.Y*q.Y+q.Z*q.Z);

    acs=acos(q.W/Q);

	c.W=log(Q);
	c.X=acs*q.X/V;
	c.Y=acs*q.Y/V;
	c.Z=acs*q.Z/V;
	return c;
}

/** \fn PdsQuaternion pds_quaternion_pow(PdsQuaternion q,PdsCnReal a)
 *  \brief Evalua <b>pow(q,a),q=W+Xi+Yj+Zk</b>.
 *
 *  \f[\sqrt{W^2+X^2+Y^2+Z^2}=|q|\f]
 *  \f[\sqrt{X^2+Y^2+Z^2}=V\f]
 *  \f[ \frac{W}{|q|}=cos(\theta )\f]
 *  \f[q^a= |q|^a \left( cos(a \theta )+sin(a \theta ) \frac{Xi+Yj+Zk}{V} \right) \f]
 *
 *  \param[in] q Quaternio.
 *  \param[in] a Variable real.
 *  \return El q^a.
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_pow(PdsQuaternion q,PdsCnReal a)
{
	PdsQuaternion c;

    PdsCnReal V,Q,theta,cos_atheta,pow_Qa_sin_atheta,pow_Qa;

    Q=sqrt(q.W*q.W+q.X*q.X+q.Y*q.Y+q.Z*q.Z);

    V=sqrt(q.X*q.X+q.Y*q.Y+q.Z*q.Z);

    theta=acos(q.W/Q);
    pow_Qa=pow(Q,a);
    cos_atheta=cos(a*theta);
    pow_Qa_sin_atheta=pow_Qa*sin(a*theta);

	c.W=pow_Qa*cos_atheta;
	c.X=pow_Qa_sin_atheta*q.X/V;
	c.Y=pow_Qa_sin_atheta*q.Y/V;
	c.Z=pow_Qa_sin_atheta*q.Z/V;
	return c;
}

/** \fn PdsCnReal pds_quaternion_norm(PdsQuaternion a)
 *  \brief Evalua |a|. <b>|a|=sqrt(a.W^2+a.X^2+a.Y^2+a.Z^2)</b>.
 *
 *  \f[\sqrt{W^2+X^2+Y^2+Z^2}=|a|\f]
 *
 *  \param[in] a Quaternio.
 *  \return El |a|.
 *  \ingroup PdsQuaternionGroup
 */
PdsCnReal pds_quaternion_norm(PdsQuaternion a)
{
	PdsCnReal r;
	r=sqrt(a.W*a.W + a.X*a.X + a.Y*a.Y + a.Z*a.Z);
	return r;
}


/** \fn PdsCnReal pds_quaternion_norm2(PdsQuaternion a)
 *  \brief Evalua |a|^2. <b>|a|^2=(a.W^2+a.X^2+a.Y^2+a.Z^2)</b>.
 *
 *  \f[W^2+X^2+Y^2+Z^2=|a|^2\f]
 *
 *  \param[in] a Quaternio.
 *  \return El |a|^2.
 *  \ingroup PdsQuaternionGroup
 */
PdsCnReal pds_quaternion_norm2(PdsQuaternion a)
{
	PdsCnReal r2;
	r2=(a.W*a.W + a.X*a.X + a.Y*a.Y + a.Z*a.Z);
	return r2;
}

/** \fn PdsQuaternion pds_quaternion_gyro(PdsCnReal Wx,PdsCnReal Wy,PdsCnReal Wz,PdsCnReal dt)
 *  \brief Retorna un quaternio de giro q, provocado por el vetor de velocidad 
 *  angular <b>W = (Wx,Wy,Wz) = Wx i + Wy j + Wz k</b> con un tiempo hasta la muestra 
 *  anterior de 'dt'.
 *
 *  Para calcular el quaternio, es usado :
 *  \f[|W|=\sqrt{{W_x}^2+{W_y}^2+{W_z}^2}\f]
 *  \f[e_w=\frac{W}{|W|}\f]
 *  de modo que
 *  \f[\theta=|W|~dt\f]
 *  e
 *
 *  \f[q=e^{\frac{\theta}{2}~e_w}=cos(\frac{\theta}{2})+sin(\frac{\theta}{2})e_w\f]
 *
 *  \param[in] Wx Velocidad angular en el eje X.
 *  \param[in] Wy Velocidad angular en el eje Y.
 *  \param[in] Wz Velocidad angular en el eje Z.
 *  \param[in] dt Tiempo de muestreo desde la muestra anterior.
 *  \return El quaternio de rotación q.
 *  \ingroup PdsQuaternionGroup
 */
PdsQuaternion pds_quaternion_gyro(PdsCnReal Wx,PdsCnReal Wy,PdsCnReal Wz,PdsCnReal dt)
{
    PdsQuaternion q;
	PdsCnReal W;
	
    W=sqrt(Wx*Wx+Wy*Wy+Wz*Wz);

    q.W=cos(W*dt/2);
    q.X=sin(W*dt/2)*Wx/W;
    q.Y=sin(W*dt/2)*Wy/W;
    q.Z=sin(W*dt/2)*Wz/W;
	return q;
}


