/*
 * pdscomplex.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdscomplex.h
 *  \author Fernando Pujaico Rivera
 *  \date 01-05-2011
 * \brief Archivo de definición de variable compleja.
 *
 */
#ifndef __PDSCOMPLEX_H__
#define __PDSCOMPLEX_H__


#include <pds/pdscnglobal.h>

#ifdef __cplusplus
extern "C" {
#endif 

/** \defgroup PdsComplexGroup Funciones del módulo PdsComplex.
 * @{
 */


/*! \struct PdsComplex
 *  \brief La estructura tipo  PdsComplex .
 *  Esta estructura genera unnumero complejo.
 *  Para usar incluir pdscomplex/pdscomplex.h.
 *  \ingroup PdsMatrixGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! Parte real. */
	PdsCnReal Real;
	/*! Parte imaginaria. */
	PdsCnReal Imag;
}PdsComplex;



/*! \def _CRECT(A,B)
 *  \brief Es equivalente a <b>(A+Bi)</b>.
 *  \ingroup PdsMatrixGroup
*/
#define _CRECT(A,B) pds_complex_rect(A,B)

/** \fn PdsComplex pds_complex_rect(PdsCnReal a,PdsCnReal b)
 *  \brief Crea la variable complejas. <b>a+bi</b>.
 *  \param[in] a Parte real.
 *  \param[in] b Parte imaginaria.
 *  \return La variable compleja.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_rect(PdsCnReal a,PdsCnReal b);



/*! \def _CPOLAR(A,B)
 *  \brief Es equivalente a <b>(A*e^{Bi})</b>.
 *  \ingroup PdsMatrixGroup
*/
#define _CPOLAR(A,B) pds_complex_polar(A,B)

/** \fn PdsComplex pds_complex_polar(PdsCnReal a,PdsCnReal b)
 *  \brief Evalua de forma polar. <b>a*e^(bi)</b>.
 *  \param[in] a Módulo de la variable compleja.
 *  \param[in] b Ángulo de la variable compleja.
 *  \return El valor complejo equivalente.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_polar(PdsCnReal a,PdsCnReal b);



/*! \def _CADDC(A,B)
 *  \brief Es equivalente a <b>(A+B)</b>.
 *  \ingroup PdsMatrixGroup
*/
#define _CADDC(A,B) pds_complex_add_complex(A,B)

/** \fn PdsComplex pds_complex_add_complex(PdsComplex a,PdsComplex b)
 *  \brief Suma dos variables complejas. <b>a+b</b>.
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable compleja.
 *  \return La suma de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_add_complex(PdsComplex a,PdsComplex b);



/*! \def _CADDR(A,B)
 *  \brief Es equivalente a <b>(A+B)</b>.
 *  \ingroup PdsMatrixGroup
*/
#define _CADDR(A,B) pds_complex_add_real(A,B)

/** \fn PdsComplex pds_complex_add_real(PdsComplex a,PdsCnReal b)
 *  \brief Suma una variable complejas y una real. <b>a+b</b>.
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable real.
 *  \return La suma de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_add_real(PdsComplex a,PdsCnReal b);



/*! \def _CADDI(A,B)
 *  \brief Es equivalente a <b>(A+Bi)</b>.
 *  \ingroup PdsMatrixGroup
*/
#define _CADDI(A,B) pds_complex_add_imag(A,B)

/** \fn PdsComplex pds_complex_add_imag(PdsComplex a,PdsCnReal b)
 *  \brief Suma una variable complejas y una imaginaria. <b>a+bi</b>.
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable imaginaria.
 *  \return La suma de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_add_imag(PdsComplex a,PdsCnReal b);



/*! \def _CSUBC(A,B)
 *  \brief Es equivalente a <b>(A-B)</b>.
 *  \ingroup PdsMatrixGroup
*/
#define _CSUBC(A,B) pds_complex_sub_complex(A,B)

/** \fn PdsComplex pds_complex_sub_complex(PdsComplex a,PdsComplex b)
 *  \brief Resta dos variables complejas. <b>a-b</b>.
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable compleja.
 *  \return La resta de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_sub_complex(PdsComplex a,PdsComplex b);



/*! \def _CSUBR(A,B)
 *  \brief Es equivalente a <b>(A-B)</b>.
 *  \ingroup PdsMatrixGroup
*/
#define _CSUBR(A,B) pds_complex_sub_real(A,B)

/** \fn PdsComplex pds_complex_sub_real(PdsComplex a,PdsCnReal b)
 *  \brief Resta una variable complejas y una real. <b>a-b</b>
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable real.
 *  \return La resta de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_sub_real(PdsComplex a,PdsCnReal b);



/*! \def _CSUBI(A,B)
 *  \brief Es equivalente a <b>(A-Bi)</b>.
 *  \ingroup PdsMatrixGroup
*/
#define _CSUBI(A,B) pds_complex_sub_imag(A,B)

/** \fn PdsComplex pds_complex_sub_imag(PdsComplex a,PdsCnReal b)
 *  \brief Resta una variable complejas y una imaginaria. <b>a-bi</b>
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable imaginaria.
 *  \return La resta de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_sub_imag(PdsComplex a,PdsCnReal b);



/*! \def _CMULC(A,B)
 *  \brief Es equivalente a <b>(A*B)</b>.
 *  \ingroup PdsMatrixGroup
*/
#define _CMULC(A,B) pds_complex_mul_complex(A,B)

/** \fn PdsComplex pds_complex_mul_complex(PdsComplex a,PdsComplex b)
 *  \brief Multiplica dos variables complejas. <b>a*b</b>.
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable compleja.
 *  \return La multiplicación de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_mul_complex(PdsComplex a,PdsComplex b);



/*! \def _CMULR(A,B)
 *  \brief Es equivalente a <b>(A*B)</b>.
 *  \ingroup PdsMatrixGroup
*/
#define _CMULR(A,B) pds_complex_mul_real(A,B)

/** \fn PdsComplex pds_complex_mul_real(PdsComplex a,PdsCnReal b)
 *  \brief Multiplica una variable compleja y una real. <b>a*b</b>.
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable real.
 *  \return La multiplicación de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_mul_real(PdsComplex a,PdsCnReal b);



/*! \def _CMULI(A,B)
 *  \brief Es equivalente a <b>(A*Bi)</b>.
 *  \ingroup PdsMatrixGroup
*/
#define _CMULI(A,B) pds_complex_mul_imag(A,B)

/** \fn PdsComplex pds_complex_mul_imag(PdsComplex a,PdsCnReal b)
 *  \brief Multiplica una variable compleja y una imaginaria. <b>a*bi</b>.
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable imaginaria.
 *  \return La multiplicación de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_mul_imag(PdsComplex a,PdsCnReal b);


/*! \def _CDIVC(A,B)
 *  \brief Es equivalente a <b>(A/B)</b>.
 *  \ingroup PdsMatrixGroup
*/
#define _CDIVC(A,B) pds_complex_div_complex(A,B)

/** \fn PdsComplex pds_complex_div_complex(PdsComplex a,PdsComplex b)
 *  \brief Divide dos variables complejas. <b>a/b</b>
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable compleja.
 *  \return La división de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_div_complex(PdsComplex a,PdsComplex b);



/*! \def _CDIVR(A,B)
 *  \brief Es equivalente a <b>(A/B)</b>.
 *  \ingroup PdsMatrixGroup
*/
#define _CDIVR(A,B) pds_complex_div_real(A,B)

/** \fn PdsComplex pds_complex_div_real(PdsComplex a,PdsCnReal b)
 *  \brief Divide una variable compleja y una real. <b>a/b</b>.
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable real.
 *  \return La división de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_div_real(PdsComplex a,PdsCnReal b);



/*! \def _CDIVI(A,B)
 *  \brief Es equivalente a <b>(A/Bi)</b>.
 *  \ingroup PdsMatrixGroup
*/
#define _CDIVI(A,B) pds_complex_div_imag(A,B)

/** \fn PdsComplex pds_complex_div_imag(PdsComplex a,PdsCnReal b)
 *  \brief Divide una variable compleja y una imaginaria. <b>a/bi</b>.
 *  \param[in] a Variable compleja.
 *  \param[in] b Variable imaginaria.
 *  \return La división de ambas variables.
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_div_imag(PdsComplex a,PdsCnReal b);


/*! \def _CCIS(B)
 *  \brief Es equivalente a <b>CIS(B)=COS(B)+i SIN(B)</b>.
 *  \ingroup PdsMatrixGroup
*/
#define _CCIS(B) pds_complex_cis(B)

/** \fn PdsComplex pds_complex_cis(PdsCnReal b)
 *  \brief Evalua cis(b). <b>cos(b)+i sin(b)=cis(b)</b>.
 *  \param[in] b Variable real.
 *  \return El cis(b).
 *  \ingroup PdsComplexGroup
 */
PdsComplex pds_complex_cis(PdsCnReal b);



/*! \def _CNORM(A)
 *  \brief Es equivalente a <b>|A|=sqrt(A.Real^2+A.Imag^2)</b>.
 *  \ingroup PdsMatrixGroup
*/
#define _CNORM(A) pds_complex_norm(A)

/** \fn PdsCnReal pds_complex_norm(PdsComplex a)
 *  \brief Evalua |a|. <b>|a|=sqrt(a.Real^2+a.Imag^2)</b>.
 *  \param[in] a Variable compleja.
 *  \return El |a|.
 *  \ingroup PdsComplexGroup
 */
PdsCnReal pds_complex_norm(PdsComplex a);



/*! \def _CNORM2(A)
 *  \brief Es equivalente a <b>|A|^2=(A.Real^2+A.Imag^2)</b>.
 *  \ingroup PdsMatrixGroup
*/
#define _CNORM2(A) pds_complex_norm2(A)

/** \fn PdsCnReal pds_complex_norm2(PdsComplex a)
 *  \brief Evalua |a|^2. <b>|a|^2=(a.Real^2+a.Imag^2)</b>.
 *  \param[in] a Variable compleja.
 *  \return El |a|^2.
 *  \ingroup PdsComplexGroup
 */
PdsCnReal pds_complex_norm2(PdsComplex a);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif 


#endif /* __PDSCOMPLEX_H__ */
