/*
 * pdsbscblock.h
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */

/** \file pdsbscblock.h
 *  \author Fernando Pujaico Rivera
 *  \date 01-09-2016
 *  \brief Funciones que trabajan con un bloque de canales BSC.
 *  
 *  Estas funciones trabajan con canales BSC así:.<br>
 *  \image html bscblock.png "Bloque de canales BSC"
 *  \b X0 es la entrada real del canal
 */

#ifndef __PDSBSCBLOCKH__
#define __PDSBSCBLOCKH__

#include <stdio.h>

#ifdef __cplusplus
extern "C" {
#endif 

#include <pds/pdscmglobal.h>
#include <pds/pdsuniform.h>
#include <pds/pdsbvector.h>
#include <pds/pdsbsc.h>

#ifndef TRUE
	#define TRUE 1
#endif

#ifndef FALSE
	#define FALSE 0
#endif

/** \defgroup PdsBscBlockGroup Funciones del módulo PdsBscBlock.
 * @{
 */


/*! \struct PdsBscBlock
 *  \brief La estructura tipo  PdsBscBlock .
 *  Esta estructura genera una bloque de canales BSC.
 *  Para usar incluir pds/pdsbscblock.h.
 *  \ingroup PdsBscBlockGroup
 *  \author Fernando Pujaico Rivera
 */
typedef struct 
{
	/*! Vector de canales BSC */
	PdsBsc **BSC;

	/*! Cantidad de canales BSC*/
	PdsCmNatural Nel;
}PdsBscBlock;


/** \fn PdsBscBlock *pds_bscblock_new(PdsVector *P)
 *  \brief Crea un bloque de canales de tipo PdsBscBlock.
 *  \param[in] P Las probabilidades de error de los canales. Cualquier valor 
 *  mayor que 1.0 y menor que 0.0 retornan un vector NULL.
 *  \return Un puntero al canal de tipo PdsBscBlock, o NULL si hubo un error.
 *  \ingroup PdsBscBlockGroup
 */
PdsBscBlock *pds_bscblock_new(PdsVector *P);


/** \fn PdsBscBlock *pds_bscblock_new_symmetric(PdsCmNatural M,PdsCmReal rho)
 *  \brief Crea un bloque de canales de tipo PdsBscBlock.
 *  \param[in] M es el número de canales BSC en el bloque.
 *  \param[in] rho Las probabilidades de error de todos los canales. Cualquier valor 
 *  mayor que 1.0 y menor que 0.0 retornan un vector NULL.
 *  \return Un puntero al canal de tipo PdsBscBlock, o NULL si hubo un error.
 *  \ingroup PdsBscBlockGroup
 */
PdsBscBlock *pds_bscblock_new_symmetric(PdsCmNatural M,PdsCmReal rho);



/** \fn int pds_bscblock_evaluate(PdsBscBlock *BLOCK,const PdsBVector *BVectorInput,PdsBVector *BVectorOutput)
 *  \brief Evalua un bit u0 en el bloque de canales BSC.
 *  \param[in,out] BLOCK Bloque de canales BSC.
 *  \param[in] u0 Es el bit a evaluar
 *  \param[out] BVectorOutput Es el vector binario a la salida de los canales BSC.
 *  \return TRUE si todo fue bien o FALSE si nó, por ejemplo un vector nulo.
 *  \ingroup PdsBscBlockGroup
 */
int pds_bscblock_evaluate_val(PdsBscBlock *BLOCK,PdsBaBit u0,PdsBVector *BVectorOutput);


/** \fn void pds_bscblock_free(PdsBscBlock *BLOCK)
 *  \brief Libera un canal de tipo puntero PdsBscBlock.
 *  \param[in,out] BLOCK El bloque de canales a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsBscBlockGroup
 */
void pds_bscblock_free(PdsBscBlock *BLOCK);


/** \fn void pds_bscblock_destroy(PdsBscBlock **BLOCK)
 *  \brief Libera un bloque de canales de tipo puntero PdsBscBlock y lo carga con NULL.
 *  \param[in,out] BLOCK El bloque de canales a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsBscBlockGroup
 */
void pds_bscblock_destroy(PdsBscBlock **BLOCK);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif 

#endif
/* __PDSBSCBLOCKH__*/ 

