/*
 * pdsbmatrix.c
 * 
 * Copyright 2011 Fernando Pujaico Rivera <fernando.pujaico.rivera@gmail.com>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 * 
 */


#include <stdlib.h>
#include <math.h>
#include <pds/pdsbmatrix.h>



////////////////////////////////////////////////////////////////////////////////
////  Trabajando con PdsBMatrix                                             ////
////////////////////////////////////////////////////////////////////////////////


/** \fn PdsBMatrix *pds_bmatrix_new(PdsBaNatural Nlin,PdsBaNatural Ncol)
 *  \brief Crea una matriz de tipo PdsBMatrix.
 *  \param[in] Nlin Es el número de lineas de la matriz binaria.
 *  \param[in] Ncol Es el número de columnas de la matriz binaria.
 *  \return Un puntero a la matriz de tipo PdsBMatrix, o NULL si hubo un error.
 *  \ingroup PdsBMatrixGroup
 */
PdsBMatrix *pds_bmatrix_new(PdsBaNatural Nlin,PdsBaNatural Ncol)
{
	PdsBMatrix *tmp=NULL;
	int bloques_ancho,i;
	int FaltoMemoria=0;

	bloques_ancho=Ncol/8;
	if(Ncol%8>0) bloques_ancho=bloques_ancho+1;

	tmp=calloc(1,sizeof(PdsBMatrix));
	if(tmp==NULL)	return NULL;

	tmp->Nlin=Nlin;
	tmp->Nbytes=bloques_ancho;
	tmp->Ncol=Ncol;

	tmp->M=calloc(tmp->Nlin,sizeof(PdsBaByte *));
	if(tmp->M==NULL)	
	{
		free(tmp);
		return NULL;
	}

	for(i=0;i<tmp->Nlin;i++)	
	{
		tmp->M[i]=calloc(tmp->Nbytes,sizeof(PdsBaByte));
		if(tmp->M[i]==NULL)	FaltoMemoria=1;
	}

	if(FaltoMemoria==1)
	{
		for(i=0;i<tmp->Nlin;i++)	free(tmp->M[i]);
		free(tmp->M);
		free(tmp);
		return NULL;
	}

	return tmp;
}


/** \fn PdsBMatrix *pds_bmatrix_new_bmatrix(const PdsBMatrix *M)
 *  \brief Crea una matriz de tipo PdsBMatrix que es una copia de outra matriz.
 *  \param[in] M Matriz de origen.
 *  \return Un puntero a una matriz de tipo PdsBMatrix, o NULL si hubo un error.
 *  \ingroup PdsBMatrixGroup
 */
PdsBMatrix *pds_bmatrix_new_bmatrix(const PdsBMatrix *M)
{	
	PdsBMatrix *T;
	PdsBaNatural i,j;
	PdsBaBit m;

	T=pds_bmatrix_new(M->Nlin,M->Ncol);
	if(T==NULL) return NULL;

	for(i=0;i<T->Nlin;i++)
	{
		for(j=0;j<T->Ncol;j++)
		{
			m=0;
			pds_bmatrix_get_bit(M,i,j,&m);
			pds_bmatrix_set_bit(T,i,j,m);
		}
	}

	return T;	
}


/** \fn int pds_bmatrix_get_bit(const PdsBMatrix *BMatrix,PdsBaNatural lin,PdsBaNatural col,PdsBaBit *m)
 *  \brief Obtiene el bit de la posición n del vector BMatrix y lo carga en m.
 *  \param[in] BMatrix Es la matriz en donde se pedirá el bit.
 *  \param[in] lin Es la fila del bit a pedir.
 *  \param[in] col Es la columna del bit a pedir.
 *  \param[out] m Es el bit pedido.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BMatrix==NULL, o lin/col fuera de rango.
 *  \ingroup PdsBMatrixGroup
 */
int pds_bmatrix_get_bit(const PdsBMatrix *BMatrix,PdsBaNatural lin,PdsBaNatural col,PdsBaBit *m)
{
	unsigned char b;

	*m=0;
	if(BMatrix==NULL)	return FALSE;
	if(lin>=BMatrix->Nlin)	return FALSE;
	if(col>=BMatrix->Ncol)	return FALSE;
 
	b=col%8;
	*m=(BMatrix->M[lin][col/8]&(1<<b))>>b;


	return TRUE;
}


/** \fn int pds_bmatrix_set_bit(PdsBMatrix *BMatrix,PdsBaNatural lin,PdsBaNatural col,PdsBaBit value)
 *  \brief Escribe el valor binario value en la posición (lin,col) de la matriz BMatrix.
 *  \param[in,out] BMatrix Es la matriz en donde se escribirá el bit.
 *  \param[in] lin Es el número de fila del bit a escribir.
 *  \param[in] col Es el número de columna del bit a escribir.
 *  \param[in] value Es el bit escrito, cero es 0 diferente de cero es 1.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BMatrix==NULL, o lin/col fuera de rango.
 *  \ingroup PdsBMatrixGroup
 */
int pds_bmatrix_set_bit(PdsBMatrix *BMatrix,PdsBaNatural lin,PdsBaNatural col,PdsBaBit value)
{
	unsigned char b;

	if(BMatrix==NULL)	return FALSE;
	if(lin>=BMatrix->Nlin)	return FALSE;
	if(col>=BMatrix->Ncol)	return FALSE;

	b=col%8;
	if(value==0)	BMatrix->M[lin][col/8]=BMatrix->M[lin][col/8]&(~(1<<b));
	else		BMatrix->M[lin][col/8]=BMatrix->M[lin][col/8]|(1<<b);
	
	
	return TRUE;
}


/** \fn int pds_bmatrix_xor_row(PdsBMatrix *BMatrix,PdsBaNatural row1,PdsBaNatural row2)
 *  \brief Ejejcuta la siguiente operacion BMatrix[row1]=BMatrix[row1] XOR BMatrix[row2].
 *  \param[in,out] BMatrix Es la matriz en operación.
 *  \param[in] row1 Primera linea.
 *  \param[in] row2 Segunda linea.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BMatrix==NULL, o row1/row2 fuera de rango.
 *  \ingroup PdsBMatrixGroup
 */
int pds_bmatrix_xor_row(PdsBMatrix *BMatrix,PdsBaNatural row1,PdsBaNatural row2)
{
	PdsBaNatural i;

	if(BMatrix==NULL)	return FALSE;
	if(row1>=BMatrix->Nlin)	return FALSE;
	if(row2>=BMatrix->Nlin)	return FALSE;
 
	for(i=0;i<BMatrix->Nbytes;i++)
	BMatrix->M[row1][i]=(BMatrix->M[row1][i])^(BMatrix->M[row2][i]);

	return TRUE;
}

/** \fn int pds_bmatrix_row_weight(const PdsBMatrix *BMatrix,PdsBaNatural lin,PdsBaNatural *m)
 *  \brief Retorna la cantidad de unos en la linea lin de la matriz BMatrix.
 *  \param[in] BMatrix Es la matriz en anilisis.
 *  \param[in] lin Linea a contar.
 *  \param[out] m Es la cantidad de unos en la linea.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BMatrix==NULL.
 *  \ingroup PdsBMatrixGroup
 */
int pds_bmatrix_row_weight(const PdsBMatrix *BMatrix,PdsBaNatural lin,PdsBaNatural *m)
{
	PdsBaNatural i;
	PdsBaBit s;
	int id;

	*m=0;
	if(BMatrix==NULL)	return FALSE;

	for(i=0;i<BMatrix->Ncol;i++)
	{
		id=pds_bmatrix_get_bit(BMatrix,lin,i,&s);
		if(id==FALSE)	{*m=0;return FALSE;}
		*m=*m+s;	
	}
	return TRUE;
}


/** \fn int pds_bmatrix_column_weight(const PdsBMatrix *BMatrix,PdsBaNatural col,PdsBaNatural *m)
 *  \brief Retorna la cantidad de unos en la columna col de la matriz BMatrix.
 *  \param[in] BMatrix Es la matriz en anilisis.
 *  \param[in] col Columna a contar.
 *  \param[out] m Es la cantidad de unos en la columna.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BMatrix==NULL.
 *  \ingroup PdsBMatrixGroup
 */
int pds_bmatrix_column_weight(const PdsBMatrix *BMatrix,PdsBaNatural col,PdsBaNatural *m)
{
	PdsBaNatural i;
	PdsBaBit s;
	int id;

	*m=0;
	if(BMatrix==NULL)	return FALSE;

	for(i=0;i<BMatrix->Nlin;i++)
	{
		id=pds_bmatrix_get_bit(BMatrix,i,col,&s);
		if(id==FALSE)	{*m=0;return FALSE;}
		*m=*m+s;	
	}
	return TRUE;
}


/** \fn int pds_bmatrix_find_first_one(PdsBMatrix *M,PdsBaNatural j)
 *  \brief Encuentra una linea que inicie en 1 en la columna "j" 
 *  a partir de la fila "j"(inclusive). La primera fila que encuentre 
 *  la coloca en la fila j y la función finaliza. 
 *  \param[in,out] M Es la matriz en anilisis.
 *  \param[in] j Columna a buscar.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: M==NULL.
 *  \ingroup PdsBMatrixGroup
 */
int pds_bmatrix_find_first_one(PdsBMatrix *M,PdsBaNatural j)
{
	PdsBaBit m;
	PdsBaByte *temp=NULL;
	unsigned short int i;

	if(j>=M->Nlin)	return FALSE;
	if(j>=M->Ncol)	return FALSE;


	for(i=j,m=0;i<M->Nlin;i++)
	{
		pds_bmatrix_get_bit(M,i,j, &m);
		if(m==1)
		{
			temp=M->M[j];
			M->M[j]=M->M[i];
			M->M[i]=temp;
			return TRUE;
		}

	}
	return FALSE;
}


/** \fn int  pds_bmatrix_set_first_zero(PdsBMatrix *M,PdsBaNatural j)
 *  \brief Obliga con XOR de la linea j que todos los vectores abajo 
 *  y arriba de la fila j sean 0 en la columna j.
 *  \param[in,out] M Es la matriz en anilisis.
 *  \param[in] j Linea a ejecutar.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: M==NULL.
 *  \ingroup PdsBMatrixGroup
 */
int  pds_bmatrix_set_first_zero(PdsBMatrix *M,PdsBaNatural j)
{
	PdsBaBit m;
	PdsBaNatural i;

	if(M==NULL)	return FALSE;
	if(j>=M->Nlin)	return FALSE;
	if(j>=M->Ncol)	return FALSE;

	for(i=0;i<M->Nlin;i++)
	{
		pds_bmatrix_get_bit(M,i,j, &m);
		if( (j!=i)&&(m==1) )	pds_bmatrix_xor_row(M,i,j);
	}
	return TRUE;
}


/** \fn PdsBMatrix * pds_bmatrix_systematize(const PdsBMatrix *S,PdsBaNatural *pos)
 *  \brief Retorna una matriz systemática o hasta donde se pueda sistematizar, carga 
 *  en pos la linea del ultimo bit de la matriz identidad. Para garantizar una matriz
 *  sistemática *pos tiene que ser igual a S->Nlin-1.
 *  \param[in] S Es la matriz en análisis.
 *  \param[out] pos La linea del ultimo bit de la matriz identidad.
 *  \return Retorna una matriz systemática. Retorna NULL si no fue posible crear la matriz
 *  \ingroup PdsBMatrixGroup
 */
PdsBMatrix * pds_bmatrix_systematize(const PdsBMatrix *S,PdsBaNatural *pos)
{
	int id;
	PdsBaNatural i,j;
	PdsBMatrix *M=NULL;

	*pos=0;
	if(S==NULL)	return NULL;

	M=pds_bmatrix_new_bmatrix(S);
	if(M==NULL)	return NULL;


	for(j=0;j<M->Nlin;j++)
	{
		//encuentra un vector que inicie en 1 en la columna "j" a partir de la 
		//columna "j" y lo pongo en la fila j.
		id=pds_bmatrix_find_first_one(M,j);

		
		//Obliga con XOR de M[j] que todos los vectores abajo y arriba de la fila j sean 0
		//en la columna j
		if(id==TRUE)	
		{
			*pos=j;
			pds_bmatrix_set_first_zero(M,j);
		}
		else		
		{
			return M;
		}

	}

	return  M;
}

/** \fn int pds_bmatrix_alist_save(const PdsBMatrix *m,const char *filename)
 *  \brief Escribe la matriz en un archivo con nombre filename en formato Alist.
 *  <a href="http://www.inference.phy.cam.ac.uk/mackay/codes/alist.html">Mackay</a>, 
 *  \param[in] m Es la matriz a gruadar.
 *  \param[in] filename Es el nombre del archivo donde será guardada la matriz.
 *  \return TRUE si todo fue bien, o FALSE sino. Ejem: BMatrix==NULL.
 *  \ingroup PdsBMatrixGroup
 */
int pds_bmatrix_alist_save(const PdsBMatrix *m,const char *filename)
{
	PdsBaNatural i,j,maxlin,maxcol,p,q;
	int id;
	PdsBaBit b;
	FILE *fd=NULL;

	if(m==NULL)	return FALSE;

	fd=fopen(filename,"w");
	if(fd==NULL)		  return FALSE;

	////////////////////////////////////////////////////////////////////////
	id=fprintf(fd,"%d %d\n",m->Nlin,m->Ncol);
	if(id==FALSE)	return FALSE;
	//printf("%32s [OK]\n","Escrito Tamanho matriz");

	////////////////////////////////////////////////////////////////////////
	id=pds_bmatrix_row_weight(m,0,&maxlin);
	if(id==FALSE)	return FALSE;
	for(i=1;i<m->Nlin;i++) 
	{
		id=pds_bmatrix_row_weight(m,i,&p);
		if(id==FALSE)	return FALSE;
		if(p>maxlin) maxlin=p;
	}
	pds_bmatrix_column_weight(m,0,&maxcol);
	for(i=1;i<m->Ncol;i++) 
	{
		id=pds_bmatrix_column_weight(m,i,&q);
		if(id==FALSE)	return FALSE;
		if(q>maxcol) maxcol=q;		
	}
	fprintf(fd,"%d %d\n",maxlin,maxcol);
	//printf("%32s [OK]\n","Escrito Tamanho maximo unos");
		
	////////////////////////////////////////////////////////////////////////
	for(i=0;i<m->Nlin;i++)
	{
		id=pds_bmatrix_row_weight(m,i,&p);
		if(id==FALSE)	return FALSE;
		fprintf(fd,"%d ",p);
	}
	fprintf(fd,"\n");
	//printf("%32s [OK]\n","Escrito Tamanho unos linea");

	////////////////////////////////////////////////////////////////////////	
	for(i=0;i<m->Ncol;i++)
	{
		id=pds_bmatrix_column_weight(m,i,&q);
		if(id==FALSE)	return FALSE;
		fprintf(fd,"%d ",q);
	}
	fprintf(fd,"\n");
	//printf("%32s [OK]\n","Escrito Tamanho unos columna");	

	////////////////////////////////////////////////////////////////////////	
	for(i=0;i<m->Nlin;i++) 
	{
		p=0;
		for(j=0;j<m->Ncol;j++)
		{
			id=pds_bmatrix_get_bit(m,i,j,&b);
			if(id==FALSE)	return FALSE;
			if(b==1)
			{
				fprintf(fd,"%d ",j+1);
				p++;
			}
		}
		for(j=p;j<maxlin;j++) fprintf(fd,"%d ",0);
		fprintf(fd,"\n");
	}
	//printf("%32s [OK]\n","Escrito unos fila");

	////////////////////////////////////////////////////////////////////////	
	for(j=0;j<m->Ncol;j++) 
	{
		q=0;
		for(i=0;i<m->Nlin;i++)
		{
			id=pds_bmatrix_get_bit(m,i,j,&b);
			if(id==FALSE)	return FALSE;
			if(b==1)
			{
				fprintf(fd,"%d ",i+1);
				q++;
			}
		}
		for(i=q;i<maxcol;i++) fprintf(fd,"%d ",0);
		fprintf(fd,"\n");
	}
	//printf("%32s [OK]\n","Escrito unos columna");
	//printf("\n");

	fclose(fd);
	return TRUE;
}


/** \fn int pds_bmatrix_is_zero(const PdsBMatrix *BMatrix, PdsBaNatural Lin)
 *  \brief Averigua si la matriz es nula desde la linea Lin (incluindo) hasta el final.
 *  \param[in] BMatrix La matriz a consultar.
 *  \param[in] Lin La linea inicial a consultar.
 *  \return TRUE si esta lleno de zeros desde la linea pos o FALSE sino. Si la
 *  matriz es nula también retorna FALSE.
 *  \ingroup PdsBMatrixGroup
 */
int pds_bmatrix_is_zero(const PdsBMatrix *BMatrix, PdsBaNatural Lin)
{
	PdsBaNatural i;
	PdsBaNatural j;
	PdsBaBit m;

	if(BMatrix==NULL)	return FALSE;

	for(i=Lin;i<BMatrix->Nlin;i++)
	{
		for(j=0;j<BMatrix->Ncol;j++)
		{
			pds_bmatrix_get_bit (BMatrix,i,j,&m);
			if(m==1)	return FALSE;
		}
	}
	return TRUE;
}


/** \fn int pds_bmatrix_printf(const PdsBMatrix *BMatrix)
 *  \brief Imprime en pantallala la matriz BMatrix.
 *  \param[in] BMatrix La matriz a imprimir.
 *  \return TRUE si todo fue bien o FALSE sino. Ejem: BMatrix==NULL
 *  \ingroup PdsBMatrixGroup
 */
int pds_bmatrix_printf(const PdsBMatrix *BMatrix)
{
	PdsBaNatural i,j,k;
	PdsBaBit m;

	if(BMatrix==NULL)	return FALSE;

	for(i=0;i<BMatrix->Nlin;i++)
	{
		for(j=0,k=0;j<BMatrix->Ncol;j++,k++)
		{
			pds_bmatrix_get_bit(BMatrix,i,j,&m);
			printf("%d",m);
			if(k==7)	printf(" ");
		}
		printf("\n");
	}

	return TRUE;
}


/** \fn void pds_bmatrix_free(PdsBMatrix *BMatrix)
 *  \brief Libera una matriz de tipo puntero PdsBMatrix.
 *  \param[in,out] BMatrix El puntero matriz a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsBMatrixGroup
 */
void pds_bmatrix_free(PdsBMatrix *BMatrix)
{
	PdsBaNatural i;

	if(BMatrix!=NULL)
	{
		for(i=0;i<BMatrix->Nlin;i++)	free(BMatrix->M[i]);

		free(BMatrix->M);
		free(BMatrix);
	}
}


/** \fn void pds_bmatrix_destroy(PdsBMatrix **BMatrix)
 *  \brief Libera una matriz de tipo puntero PdsBMatrix y ademas carga NULL en el puntero.
 *  \param[in,out] BMatrix La direccion del puntero matriz a liberar.
 *  \return No retorna valor.
 *  \ingroup PdsBMatrixGroup
 */
void pds_bmatrix_destroy(PdsBMatrix **BMatrix)
{
	PdsBaNatural i;

	if((*BMatrix)!=NULL)
	{
		for(i=0;i<(*BMatrix)->Nlin;i++)	free((*BMatrix)->M[i]);

		free((*BMatrix)->M);
		free(*BMatrix);
		(*BMatrix)=NULL;
	}
}

