// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;

#include "../libpdrx/datatypes.h"
#include "../libpdrx/xception.h"
#include "config_impl.h"
#include "db.h"
#include "out_impl.h"
#include "out_ftree.h"
#include "diagram.h"

//=== OutputDiagram ========================================================
OutputDiagram::OutputDiagram (const string& option_key)
	: OutputImpl(option_key)
{
}

void OutputDiagram::Do (const Config& config, const Database& database) const throw (Xception)
{
	const string& input_file = config.GetFilenameOption(m_option_key + ".input_file");
	if (input_file.empty())
		THROW(format("missing input_file specification in diagram configuration: %s") % m_option_key);

	const string& output_file = config.GetFilenameOption(m_option_key + ".output_file");
	if (output_file.empty())
		THROW(format("missing output_file specification in diagram configuration: %s") % m_option_key);

	Poco::TextEncoding* pEncoding = NULL;
	if (m_option_key.empty())
		pEncoding = &NativeEncoding();
	else
		pEncoding = &SpecificEncoding(config.GetStringOption(m_option_key + ".encoding"));

	bool antialias = config.GetBoolOption(m_option_key + ".antialias");

	// very easy: take the whole input file as a definition
	string definition;
	{
		ifstream is(input_file.c_str(), ios::in);
		if (!is.good())
			THROW(format("cannot open input file: %s") % input_file);

		string line;
		while (getline(is, line))
			definition += Decode(line, *pEncoding) + '\n';

		trim(definition);
	}

	// make a diagram from the definition
	if (!definition.empty())
	{
		auto_ptr<Diagram> diagram(DiagramFactory::Create(output_file, antialias));
		{
			auto_ptr<FTree> ft(Parse(definition));
			FDiagramContext context(config, database, *diagram);
			ft->Evaluate(context);
		}
		diagram->SaveAs(output_file);
	}
}
