// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;
using namespace boost::filesystem;

#include "../libpdrx/xception.h"
#include "../libpdrx/config.h"
#include "http.h"
#include "hmac_sha1.h"

#ifdef _WIN32
#include <shellapi.h>
#endif

//#define DEBUG_REQUEST		DEBUG
//#define DEBUG_HEADERS		DEBUG
//#define DEBUG_RESPONSE		DEBUG

// note: we don't use the encoded streams here for debug output, we want to
// see everything exactly "as is"

//=== helper ===============================================================
namespace HttpClientLocal {

	class HttpException: public Xception
	{
		public:
		HttpException (const string& function, system::error_code error);
		virtual void Rethrow ();
	};

	HttpException::HttpException (const string& function, system::error_code error)
		: Xception(function, string("HTTP: ") + system::system_error(error).what())
	{
	}

	void HttpException::Rethrow ()
	{
		throw *this;
	}

	static char read_byte (boost::asio::ip::tcp::socket& socket)
	{
		char ch;
		system::error_code error;
		while (asio::read(socket, asio::buffer(&ch, sizeof(ch)), asio::transfer_at_least(1), error) < sizeof(ch))
		{
			if (error != asio::error::eof)
				throw HttpException(CURRENT_FUNCTION_NAME, error);
			else
#ifdef _WIN32
				::Sleep(250);
#else
				sleep(1); // give the socket some time
#endif
		}
		return ch;
	}

	static void read_line (asio::ip::tcp::socket& socket, string& line)
	{
		char ch;
		while ((ch = read_byte(socket)) != '\r')
			line += ch;
		read_byte(socket); // \n
	}

	static void read_block (asio::ip::tcp::socket& socket, size_t len, string& output)
	{
		size_t pos = output.size();
		output.resize(pos + len);
		while (true)
		{
			system::error_code error;
			size_t n = asio::read(socket, asio::buffer(((char*)output.c_str()) + pos, len), asio::transfer_at_least(len), error);
			if (n == len)
				break;
			if (error != asio::error::eof)
				throw HttpException(CURRENT_FUNCTION_NAME, error);
			else
			{
#ifdef _WIN32
				::Sleep(250);
#else
				sleep(1); // give the socket some time
#endif
				pos += n;
				len -= n;
			}
		}
/*
		output.reserve(output.size() + len);
		for (size_t i = 0; i < len; i++)
			output.push_back(read_byte(socket));
*/
	}
} // namespace HttpClientLocal

//=== HttpClient ===========================================================
HttpClient::HttpClient ()
	: m_io_service()
	, m_socket(m_io_service)
	, m_host()
{
}

HttpClient::~HttpClient ()
{
	if (m_socket.is_open())
		Close();
}

void HttpClient::Open (const string& host, const string& proxy)
{
	m_host.clear();

	system::error_code error;

	if (!proxy.empty())
	{
		// split the proxy, the proxy has normally a port
		static const regex rx("([^:]+):(.+)?");
		smatch mr;
		if (!regex_match(proxy, mr, rx))
			THROW(format("invalid proxy specified: %s") % proxy);
		const string proxy_server(mr[1]);
		const string proxy_port(mr[2]);

		asio::ip::tcp::resolver resolver(m_io_service);
		asio::ip::tcp::resolver::query query(
			proxy_server,
			(proxy_port.empty())
				? string("http")
				: proxy_port,
			(proxy_port.empty())
				? (asio::ip::tcp::resolver::query::passive | asio::ip::tcp::resolver::query::address_configured)
				: (asio::ip::tcp::resolver::query::passive | asio::ip::tcp::resolver::query::address_configured | asio::ip::tcp::resolver::query::numeric_service)
		);
		asio::ip::tcp::resolver::iterator endpoint_iterator = resolver.resolve(query, error);
		if (error != 0)
			throw HttpClientLocal::HttpException(CURRENT_FUNCTION_NAME, error);
		asio::ip::tcp::resolver::iterator end;
		while (!m_socket.is_open() && endpoint_iterator != end)
			m_socket.connect(*endpoint_iterator++, error);
		if (error != 0)
			throw HttpClientLocal::HttpException(CURRENT_FUNCTION_NAME, error);
	}
	else
	{
		asio::ip::tcp::resolver resolver(m_io_service);
		asio::ip::tcp::resolver::query query(host, "http");
		asio::ip::tcp::resolver::iterator endpoint_iterator = resolver.resolve(query, error);
		if (error != 0)
			throw HttpClientLocal::HttpException(CURRENT_FUNCTION_NAME, error);
		asio::ip::tcp::resolver::iterator end;

		while (!m_socket.is_open() && endpoint_iterator != end)
			m_socket.connect(*endpoint_iterator++, error);
		if (error != 0)
			throw HttpClientLocal::HttpException(CURRENT_FUNCTION_NAME, error);
	}

	// (at this point the connection IS open)
	m_host = host;
}

void HttpClient::Close ()
{
	m_socket.close();
	m_host.clear();
}

void HttpClient::Request (const string& method, const string& filename, const string& additionalHeaders /*= ""*/, const string& body /*= ""*/)
{
	string r;
	r += method + " " + filename + " HTTP/1.1\r\n";
	r += "Host: " + m_host + "\r\n";
	r += "Accept: */*\r\n";
	r += "Connection: keep-alive\r\n";
	r += "Content-Length: " + lexical_cast<string>(body.size()) + "\r\n";
	if (!additionalHeaders.empty())
		r += additionalHeaders;
	trim(r);
	r += "\r\n\r\n"; // yes, twice
	r += body;

#ifdef DEBUG_REQUEST
	cout << "----------" << endl;
	cout << r;
	cout << "----------" << endl;
#endif
	asio::streambuf request;
	std::ostream request_stream(&request);
	request_stream << r;

	system::error_code error;
	asio::write(m_socket, request, error);
	if (error != 0)
		throw HttpClientLocal::HttpException(CURRENT_FUNCTION_NAME, error);
}

string HttpClient::Response ()
{
	// we don't use a stream here because of the possibility of a
	// chunked response

	system::error_code error;

	// check the first line -> "HTTP/1.1 200 OK"
	{
		string line;
		HttpClientLocal::read_line(m_socket, line);
#ifdef DEBUG_HEADERS
		cout << line << endl;
#endif
		static const regex rx("HTTP/[^ ]+ ([0-9]+) (.+)");
		smatch mr;
		if (regex_match(line, mr, rx))
		{
			int status_code = lexical_cast<int>(mr[1]);
			if (status_code != 200)
				THROW(format("HTTP server returned: %d (%d)") % status_code % mr[2]);
		}
		else
			THROW("HTTP: invalid response from HTTP server (no HTTP!)");
	}

	// check the header lines
	bool chunked = false;
	size_t content_length = (size_t)-1;
	while (true)
	{
		string line;
		HttpClientLocal::read_line(m_socket, line);
		trim(line);
#ifdef DEBUG_HEADERS
		cout << line << endl;
#endif
		if (line.empty())
			break;
		if (line.find("Transfer-Encoding:") != string::npos)
			chunked = (line.find("chunked") != string::npos);
		else
		{
			if (line.find("Content-Length:") != string::npos)
			{
				line.erase(0, 16);
				content_length = lexical_cast<size_t>(line);
			}
		}
	}

	// read the body
	string output;
	if (chunked)
	{
		int chunks = 0;
		while (true)
		{
			size_t chunk_len = 0;
			{
				string line;
				HttpClientLocal::read_line(m_socket, line);
				stringstream ss;
				ss << hex << line;
				ss >> chunk_len;
			}
			if (chunk_len == 0)
				break;
			else
				chunks++;

			HttpClientLocal::read_block(m_socket, chunk_len, output);

			HttpClientLocal::read_byte(m_socket); // \r
			HttpClientLocal::read_byte(m_socket); // \n
		}
	}
	else
	{
		if (content_length != (size_t)-1)
		{
			// read a fix length block
			HttpClientLocal::read_block(m_socket, content_length, output);
		}
		else
		{
			// read until eof
			while (true)
			{
				char ch;
				asio::read(m_socket, asio::buffer(&ch, sizeof(ch)), asio::transfer_at_least(1), error);
				if (error == asio::error::eof)
					break;
				output.push_back(ch);
			}
		}
	}

#ifdef DEBUG_RESPONSE
	cout << output << endl;
#endif

	return output;
}

//=== OAuthHttpClient ======================================================
namespace OAuth
{
	string UrlEncode (const string& s)
	{
		string result;
		foreach(char c, s)
		{
			if (('0' <= c && c <= '9') || ('A' <= c && c <= 'Z') || ('a' <= c && c <= 'z') || c == '~' || c == '-' || c == '_' || c == '.')
				result += c;
			else
				result += (format("%%%02X") % (int)c).str();
		}
		return result;
	}

	string Base64Encode (const string& s)
	{
		static const string base64_chars("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/");

		string result;
		unsigned char char_array_3[3];
		unsigned char char_array_4[4];

		int i = 0;
		foreach(char c, s)
		{
			char_array_3[i++] = c;
			if (i == 3)
			{
				char_array_4[0] = (char_array_3[0] & 0xfc) >> 2;
				char_array_4[1] = ((char_array_3[0] & 0x03) << 4) + ((char_array_3[1] & 0xf0) >> 4);
				char_array_4[2] = ((char_array_3[1] & 0x0f) << 2) + ((char_array_3[2] & 0xc0) >> 6);
				char_array_4[3] = char_array_3[2] & 0x3f;
				for(i = 0; (i <4) ; i++)
					result += base64_chars[char_array_4[i]];
				i = 0;
			}
		}

		if (i)
		{
			int j = 0;
			for(j = i; j < 3; j++)
				char_array_3[j] = '\0';
			char_array_4[0] = (char_array_3[0] & 0xfc) >> 2;
			char_array_4[1] = ((char_array_3[0] & 0x03) << 4) + ((char_array_3[1] & 0xf0) >> 4);
			char_array_4[2] = ((char_array_3[1] & 0x0f) << 2) + ((char_array_3[2] & 0xc0) >> 6);
			char_array_4[3] = char_array_3[2] & 0x3f;
			for (j = 0; (j < i + 1); j++)
				result += base64_chars[char_array_4[j]];
			while (i++ < 3)
				result += '=';
		}

		return result;
	}

	class Params
	{
		typedef map<string, string> Map;

		Map m_map;

		public:

		Params ();

		Params& operator () (const string& key, const string& value);

		string AsBaseString (const string& method, const string& endpoint) const;
		string AsAuthorizationHeader () const;
	};

	Params::Params ()
		: m_map()
	{
	}

	Params& Params::operator () (const string& key, const string& value)
	{
		m_map.insert(Map::value_type(key, value));
		return *this;
	}

	string Params::AsBaseString (const string& method, const string& endpoint) const
	{
		string result(method);
		result += '&';
		result += UrlEncode(endpoint);
		result += '&';
		string t;
		foreach(const Map::value_type& vt, m_map)
		{
			if (!t.empty())
				t += '&';
			t += vt.first + '=' + vt.second;
		}
		result += UrlEncode(t);
		return result;
	}

	string Params::AsAuthorizationHeader () const
	{
		string result("OAuth ");
		string t;
		foreach(const Map::value_type& vt, m_map)
		{
			if (!t.empty())
				t += ", ";
			t += vt.first + "=\"" + vt.second + "\"";
		}
		return result + t;
	}

	string CreateNonce () // a nonce (just a random string)
	{
		string nonce;
		const char* t = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";
		for (int i = 0; i <= 16; i++)
			nonce += t[rand() % (sizeof(t) - 1)];
		return nonce;
	}

	string CreateTimestamp ()
	{
		time_t t;
		time(&t);
		return (format("%d") % t).str();
	}
} // namespace OAuth

OAuthHttpClient::OAuthHttpClient (const string& consumer_key, const string& consumer_secret, const Config& config, const string& option_key)
	: HttpClient()
	, m_consumer_key(consumer_key)
	, m_consumer_secret(consumer_secret)
	, m_private_keys_filename()
#ifndef _WIN32
	, m_browser_executable(config.GetStringOption("browser"))
#endif
	, m_access_token()
	, m_access_token_secret()
	, m_create_private_keys_file()
{
	path cfg(config.GetConfigFile());
	string fn;
	if (cfg.filename().find(".") == 0)
		fn = ".pdr_";
	fn += option_key + ".keys";
	m_private_keys_filename = (cfg.parent_path() / fn).string();

	// we try to open the private keys file here, if we don't have such
	// a file at the moment the keys remain empty, this causes an OAuth
	// authentication later to get these keys
	ifstream ifs(m_private_keys_filename.c_str(), ios::in);
	if (ifs.good())
	{
		string line;
		getline(ifs, line);
		trim(line);
		static const regex rx("([^,]+),(.+)");
		smatch mr;
		if (regex_match(line, mr, rx))
		{
			m_access_token = mr[1];
			m_access_token_secret = mr[2];
		}
	}
}

OAuthHttpClient::~OAuthHttpClient ()
{
	// if we created the private keys during this session save them here
	// for later use
	if (m_create_private_keys_file)
	{
		ofstream ofs(m_private_keys_filename.c_str(), ios::out);
		if (ofs.good())
			ofs << m_access_token << "," << m_access_token_secret << endl;
	}
}

void OAuthHttpClient::Open (const string& host, const string& proxy)
{
	HttpClient::Open(host, proxy);

	if (m_access_token.empty() || m_access_token_secret.empty())
	{
		cout << "    no private keys found, interactive authentication needed" << endl;

		const string& nonce = OAuth::CreateNonce();

		// step 1: get a request token
		string request_token, request_token_secret;
		{
			const string endpoint("http://api.twitter.com/oauth/request_token");

			OAuth::Params params;
			params
				("oauth_callback",		"oob")
				("oauth_consumer_key",		m_consumer_key)
				("oauth_nonce",			nonce)
				("oauth_signature_method",	"HMAC-SHA1")
				("oauth_timestamp",		OAuth::CreateTimestamp())
				("oauth_version",		"1.0")
			;
			// cout << params.AsBaseString("POST", endpoint) << endl;

			string oauth_signature;
			{
				const string& text = params.AsBaseString("POST", endpoint);
				const string& key = m_consumer_secret + '&';
				char signature[20];
				CHMAC_SHA1().HMAC_SHA1((BYTE*)text.c_str(), text.size(), (BYTE*)key.c_str(), key.size(), (BYTE*)signature);
				oauth_signature = OAuth::UrlEncode(OAuth::Base64Encode(string(signature, 20)));
			}
			// cout << oauth_signature << endl;

			params
				("oauth_signature",		oauth_signature);

			string header("Authorization: ");
			header += params.AsAuthorizationHeader();
			// cout << header << endl;

			HttpClient::Request("POST", endpoint, header);
			const string& response = Response();
			// cout << response << endl;

			const regex rx("oauth_token=([^&]+)&oauth_token_secret=([^&]+)&oauth_callback_confirmed=true");
			smatch mr;
			if (!regex_match(response, mr, rx))
				THROW("could not get request_token");
			request_token = mr[1];
			request_token_secret = mr[2];
			// cout << "request_token: " << request_token << endl << "request_token_secret: " << request_token_secret << endl;
		}

		// because we don't know what the user does and how long
		// this will take we close the socket here for being sure
		// that it will not time out
		HttpClient::Close();

		// step 2: authorize the user (the interactive thing)
		string oauth_verifier;
		{
			const string endpoint("http://api.twitter.com/oauth/authorize?oauth_token=" + request_token);

#ifdef _WIN32
			::ShellExecute(0, "open", endpoint.c_str(), "", "", SW_SHOW);
#else
			if (m_browser_executable.empty())
			{
				THROW(
					"You didn't specify a browser executable. Twitter requires a browser to show "
					"an interactive authentication web page. Use the pdr command line parameter "
					"--browser (or -? for general information)"
				);
			}

			std::system((m_browser_executable + " " + endpoint).c_str());
#endif

			cout << "        PIN (displayed in browser): ";
			getline(cin, oauth_verifier);
			trim(oauth_verifier);
		}

		// now we open again using the same parameters
		HttpClient::Open(host, proxy);

		// step 3: change the request_token against an access token
		{
			const string endpoint("http://api.twitter.com/oauth/access_token");

			OAuth::Params params;
			params
				("oauth_consumer_key",		m_consumer_key)
				("oauth_nonce",			nonce)
				("oauth_signature_method",	"HMAC-SHA1")
				("oauth_token",			request_token)
				("oauth_timestamp",		OAuth::CreateTimestamp())
				("oauth_verifier",		oauth_verifier)
				("oauth_version",		"1.0")
			;
			// cout << params.AsBaseString("POST", endpoint) << endl;

			string oauth_signature;
			{
				const string& text = params.AsBaseString("POST", endpoint);
				const string& key = m_consumer_secret + '&' + request_token_secret;
				char signature[20];
				CHMAC_SHA1().HMAC_SHA1((BYTE*)text.c_str(), text.size(), (BYTE*)key.c_str(), key.size(), (BYTE*)signature);
				oauth_signature = OAuth::UrlEncode(OAuth::Base64Encode(string(signature, 20)));
			}
			// cout << oauth_signature << endl;

			params
				("oauth_signature",		oauth_signature);

			string header("Authorization: ");
			header += params.AsAuthorizationHeader();
			// cout << header << endl;

			HttpClient::Request("POST", endpoint, header);
			const string& response = Response();
			// cout << response << endl;

			const regex rx("oauth_token=([^&]+)&oauth_token_secret=([^&]+)&user_id=([^&]+)&screen_name=(.+)");
			smatch mr;
			if (!regex_match(response, mr, rx))
				THROW("could not get access_token");
			m_access_token = mr[1];
			m_access_token_secret = mr[2];

			cout	<< "        access_token: " << m_access_token << endl
				<< "        access_token_secret: " << m_access_token_secret << endl
				<< "        userid (verified): " << mr[3] << endl
				<< "        username (verified): " << mr[4] << endl
				<< "    success" << endl;

			m_create_private_keys_file = true;
		}
	}
}

void OAuthHttpClient::Request (const string& method, const string& filename, const string& additionalHeaders /*= ""*/, const string& body /*= ""*/)
{
	// first we need to separate the query parameters behind a '?' from
	// the filename for signing, so we get a short_filename without any
	// parameters and query_params
	string short_filename;
	vector<string> query_params;
	{
		string::size_type pos = filename.find("?");
		if (pos != string::npos)
		{
			short_filename = string(filename, 0, pos);
			string s(filename, pos + 1);
			split(query_params, s, is_any_of("&"));
		}
		else
			short_filename = filename;
	}

	// now we generate a signed Authorization header, the signature is
	// computed over the HTTP-method, the short (!) filename and the
	// OAuth parameters combined with the separated query parameters
	string authorization_header;
	{
		OAuth::Params params;
		params
			("oauth_consumer_key",		m_consumer_key)
			("oauth_nonce",			OAuth::CreateNonce())
			("oauth_signature_method",	"HMAC-SHA1")
			("oauth_timestamp",		OAuth::CreateTimestamp())
			("oauth_token",			m_access_token)
			("oauth_version",		"1.0")
		;
		foreach(const string& query_param, query_params)
		{
			static const regex rx("([^=]+)=(.+)");
			smatch mr;
			if (regex_match(query_param, mr, rx))
			{
				params
					(mr[1], mr[2]);
			}
		}
		// cout << params.AsBaseString(method, short_filename) << endl;

		string oauth_signature;
		{
			const string& text = params.AsBaseString(method, short_filename);
			const string& key = m_consumer_secret + '&' + m_access_token_secret;
			char signature[20];
			CHMAC_SHA1().HMAC_SHA1((BYTE*)text.c_str(), text.size(), (BYTE*)key.c_str(), key.size(), (BYTE*)signature);
			oauth_signature = OAuth::UrlEncode(OAuth::Base64Encode(string(signature, 20)));
		}
		// cout << oauth_signature << endl;

		params
			("oauth_signature",		oauth_signature);

		authorization_header = "Authorization: " + params.AsAuthorizationHeader() + "\r\n";
		// cout << authorization_header << endl;
	}

	// combine the Authorization header with user defined headers
	string headers(additionalHeaders);
	trim(headers);
	if (!headers.empty())
		headers += "\r\n";
	headers += authorization_header;

	// request!
	HttpClient::Request(method, filename, headers, body);
}

string OAuthHttpClient::GetUniqueFeedIdentifier () const
{
	return m_access_token + ',' + m_access_token_secret;
}
