// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;

#include "xception.h"

//=== Xception =============================================================
Xception::Xception (const string& function, const string& message)
	: m_stack()
{
	m_stack.push_back(Stack::value_type(function, message));
}

Xception::Xception (const string& function, const string& message, const string& technical)
	: m_stack()
{
	stringstream ss;
	if (!message.empty())
		ss << message;
	if (!technical.empty())
	{
		if (!message.empty())
			ss << endl;
		ss << "  " << technical;
	}
	ss << ends;

	m_stack.push_back(Stack::value_type(function, ss.str()));
}

Xception::Xception (const string& function, const format& message)
	: m_stack()
{
	m_stack.push_back(Stack::value_type(function, message.str()));
}

Xception::Xception (const string& function, const format& message, const string& technical)
	: m_stack()
{
	stringstream ss;
	const string& t = message.str();
	if (!t.empty())
		ss << t;
	if (!technical.empty())
	{
		if (!t.empty())
			ss << endl;
		ss << "  " << technical;
	}
	ss << ends;

	m_stack.push_back(Stack::value_type(function, ss.str()));
}

Xception::~Xception ()
{
}

	static bool detailed = false;

string Xception::Message (Xception::MessageStyle style) const
{
	stringstream ss;
	switch (style)
	{
		case brief:
		{
			// only the last, top most message
			ss << m_stack.back().second;
			break;
		}
		default:
		{
			// the messages in reverse order,
			// if detailed also the function names
			bool first = true;
			Stack::const_reverse_iterator I = m_stack.rbegin();
			while (I != m_stack.rend())
			{
				const Stack::value_type& vt = *I++;
				if (!vt.second.empty() || detailed)
				{
					if (!first)
						ss << endl;
					first = false;

					if (!vt.second.empty())
						ss << vt.second;

					if (detailed)
					{
						if (!vt.second.empty())
							ss << endl;
						ss << "    [" << vt.first << "]";
					}
				}
			}
			break;
		}
	}
	ss << ends;

	return ss.str();
}

void Xception::SetDetailedMode ()
{
	detailed = true;
}

void Xception::Rethrow (const string& function, const string& message)
{
	m_stack.push_back(Stack::value_type(function, message));

	Rethrow();
}

void Xception::Rethrow (const string& function, const format& message)
{
	m_stack.push_back(Stack::value_type(function, message.str()));

	Rethrow();
}

void Xception::Rethrow ()
{
	throw *this;
}
