// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifdef USE_MYSQL

#include "../libpdrx/common.h"

#include <Poco/Data/Common.h>
#include <Poco/Data/MySQL/Connector.h>
#include <Poco/Data/MySQL/MySQLException.h>
#include <Poco/Data/RecordSet.h>

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;
using namespace Poco::Data;

#include "../libpdrx/datatypes.h"
#include "../libpdrx/config.h"
#include "db_impl.h"

// the MySQL connect string is a bit delicate, it looks like this:
//
//   user=my_db_user_name;password=my_db_user_password;db=my_db
//
// don't place anything else in this string, no spaces, no quotation marks
// and especially no semicolon at the end

//=== MySQLDatabase ========================================================
MySQLDatabase::MySQLDatabase (const string& connect, bool verbose)
	: PocoDatabaseImpl(connect, verbose, MySQL::Connector::KEY, new MySQL::Connector())
{
}

void MySQLDatabase::NoSchemaCallback () throw (Xception)
{
	if (m_verbose)
		cout << "creating new schema" << endl;

	try
	{
		DBTransactor transactor(m_pSession, m_transactionCounter);

		(*m_pSession) << "create table TCollections "
				 "("
					"name varchar(64) not null primary key,"
					"type char(1) not null,"
					"tbl_name varchar(64) not null,"
					"purpose varchar(256)",
					"unit varchar(16)"
				 ") engine=InnoDB;", now;

		(*m_pSession) << "create table TRejected "
				 "("
					"i int primary key default null auto_increment,"
					"t char(19) not null,"
					"expr varchar(1024) not null"
				 ") engine=InnoDB;", now;

		(*m_pSession) << "create table C0 "
				 "("
					"t char(19) not null primary key,"
					"v double not null"
				 ") engine=InnoDB;", now;

		(*m_pSession) << "create table C1 "
				 "("
					"t char(19) not null primary key,"
					"v varchar(1024) not null"
				 ") engine=InnoDB;", now;

		(*m_pSession) << "insert into TCollections values ('*','n','C0',NULL,NULL);", now;
		(*m_pSession) << "insert into TCollections values ('#','t','C1',NULL,NULL);", now;

		transactor.Commit();
	}
	catch (...)
	{
		throw Xception("could not create schema");
	}
}

void MySQLDatabase::NoPurposeColumnCallback () throw (Xception)
{
	if (m_verbose)
		cout << "updating schema" << endl;

	try
	{
		(*m_pSession) << "alter table TCollections add column purpose varchar(256);", now;
	}
	catch (...)
	{
		throw Xception("could not update schema");
	}
}

void MySQLDatabase::NoUnitColumnCallback () throw (Xception)
{
	if (m_verbose)
		cout << "updating schema" << endl;

	try
	{
		(*m_pSession) << "alter table TCollections add column unit varchar(16);", now;
	}
	catch (...)
	{
		throw Xception("could not update schema");
	}
}

void MySQLDatabase::CreateCollectionInSchema (const string& name, char type, const string& tblname, const string& purpose, const string& unit) throw (Xception)
{
	try
	{
		DBTransactor transactor(m_pSession, m_transactionCounter);

		string sql("create table ");
		sql += tblname;
		sql += " (t char(19) not null primary key, ";
		switch (type)
		{
			case 'n':	sql += "v double not null"; break;
			case 'r':	sql += "n double not null, d double not null"; break;
			case 't':	sql += "v varchar(1024) not null"; break;
		}
		sql += ") engine=InnoDB;";
		(*m_pSession) << sql, now;

		string t;
		t = type;
		(*m_pSession) << "insert into TCollections values (?,?,?,?,?);", use(name), use(t), use(tblname), use(purpose), use(unit), now;

		transactor.Commit();
	}
	catch (...)
	{
		throw Xception("could not create collection");
	}
}

void MySQLDatabase::DropCollectionFromSchema (const string& tblname) throw (Xception)
{
	try
	{
		DBTransactor transactor(m_pSession, m_transactionCounter);

		(*m_pSession) << "delete from TCollections where tbl_name=?;", use(tblname), now;
		(*m_pSession) << "drop table " << tblname << ";", now;

		transactor.Commit();
	}
	catch (...)
	{
		throw Xception("could not delete collection");
	}
}

void MySQLDatabase::InsertOrUpdateCollectionItem (const CollectionMetaInfo& cmi, const CollectionItem& item)
{
	try
	{
		// insert or update?
		const string& t = lexical_cast<string, ptime>(item.first);
		int count;
		(*m_pSession) << "select count(*) from " << cmi.m_tblname << " where t=?;", use(t), into(count), now;

		switch (count)
		{
			case 0: // insert
			{
				switch (cmi.m_type)
				{
					case 'n':
					{
						double data = any_cast<double>(item.second);
						(*m_pSession) << "insert into " << cmi.m_tblname << " values (?,?);", use(t), use(data), now;
						break;
					}
					case 'r':
					{
						Ratio data = any_cast<Ratio>(item.second);
						(*m_pSession) << "insert into " << cmi.m_tblname << " values (?,?,?);", use(t), use(data.m_numerator), use(data.m_denominator), now;
						break;
					}
					case 't':
					{
						string data = any_cast<string>(item.second);
						(*m_pSession) << "insert into " << cmi.m_tblname << " values (?,?);", use(t), use(data), now;
						break;
					}
					default:
						break;
				}
				break;
			}
			case 1: // update
			{
				switch (cmi.m_type)
				{
					case 'n':
					{
						double data = any_cast<double>(item.second);
						(*m_pSession) << "update " << cmi.m_tblname << " set v=? where t=?;", use(data), use(t), now;
						break;
					}
					case 'r':
					{
						Ratio data = any_cast<Ratio>(item.second);
						(*m_pSession) << "update " << cmi.m_tblname << " set n=?,d=? where t=?;", use(data.m_numerator), use(data.m_denominator), use(t), now;
						break;
					}
					case 't':
					{
						string data = any_cast<string>(item.second);
						(*m_pSession) << "update " << cmi.m_tblname << " set v=? where t=?;", use(data), use(t), now;
						break;
					}
					default:
						break;
				}
				break;
			}
			default:
				break;
		}
	}
	catch (...)
	{
		throw Xception("could not insert or update collection item");
	}
}

void MySQLDatabase::UpdatePurpose (const string& name, const string& purpose) throw (Xception)
{
	try
	{
		(*m_pSession) << "update TCollections set purpose=? where name=?;", use(purpose), use(name), now;
	}
	catch (...)
	{
		throw Xception("could not update collection purpose");
	}
}

#endif // USE_MYSQL
