// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;

#include "../libpdrx/datatypes.h"
#include "../libpdrx/config.h"
#include "../libpdrx/encoding.h"
#include "db.h"
#include "in_impl.h"

#ifdef USE_ETPAN

// on WIN32 libetpan conflicts with boost, so we must define some
// identifiers here before libetpan makes conflicting typedefs
#ifdef WIN32
#define uint16_t unsigned __int16
#define uint32_t unsigned __int32
#define int32_t __int32
#endif

#include <libetpan/libetpan.h>

//=== ImapMailClient =======================================================
ImapMailClient::ImapMailClient (const string& option_key)
	: InputImpl(option_key)
{
}

	static void decode_quoted_printable (string& s)
	{
		size_t index = 0;
		char* buffer;
		size_t len;
		if (mailmime_quoted_printable_body_parse(s.c_str(), s.length(), &index, &buffer, &len, 0) == 0)
			s = buffer;
		mmap_string_unref(buffer);
	}

	static void convert_charset (string& s, const string& from_charset)
	{
		char* buffer;
		size_t len;
		if (charconv_buffer("utf-8", from_charset.c_str(), s.c_str(), s.length(), &buffer, &len) == 0)
			s = buffer;
		charconv_buffer_free(buffer);
	}

	static void handle_message (mailmessage* pMessage, const string& subject, Database& database, bool verbose, bool keep, bool& data, bool& rejects)
	{
		struct mail_flags* pFlags;
		mailmessage_get_flags(pMessage, &pFlags);
		if (pFlags->fl_flags & (MAIL_FLAG_SEEN | MAIL_FLAG_DELETED | MAIL_FLAG_CANCELLED))
			return;

		// check subject and get header data
		string sbjct;
		ptime timestamp(not_a_date_time);
		int encoding = MAILMIME_MECHANISM_ERROR;
		string charset("us-ascii");
		{
			char* header;
			size_t len;
			if (mailmessage_fetch_header(pMessage, &header, &len) == 0)
			{
				size_t index = 0;
				struct mailimf_fields* pImfFields;
				if (mailimf_fields_parse(header, len, &index, &pImfFields) == 0)
				{
					struct mailimf_single_fields* pImfSingleFields = mailimf_single_fields_new(pImfFields);

					struct mailimf_subject* fld_subject = pImfSingleFields->fld_subject;
					if (fld_subject)
						sbjct = InputImpl::ConvertMailSubject(fld_subject->sbj_value, decode_quoted_printable, convert_charset);

					struct mailimf_orig_date* fld_orig_date = pImfSingleFields->fld_orig_date;
					if (fld_orig_date)
					{
						struct mailimf_date_time& d = *fld_orig_date->dt_date_time;
						timestamp = ptime(date(d.dt_year, d.dt_month, d.dt_day), time_duration(d.dt_hour, d.dt_min, d.dt_sec,0));
						InputImpl::LocalizeMailDate(d.dt_zone, timestamp);
					}

					mailimf_single_fields_free(pImfSingleFields);

					for (clistiter* pCur = clist_begin(pImfFields->fld_list) ; pCur != NULL ; pCur = clist_next(pCur))
					{
						struct mailimf_field* pField = (mailimf_field*)clist_content(pCur);
						if (pField->fld_type == MAILIMF_FIELD_OPTIONAL_FIELD)
						{
							string fld_name = pField->fld_data.fld_optional_field->fld_name;
							to_lower(fld_name);
							if (fld_name == "content-transfer-encoding")
							{
								struct mailmime_mechanism* pEncoding;
								size_t current_index = 0;
								string fld_value(pField->fld_data.fld_optional_field->fld_value);
								if (mailmime_encoding_parse(fld_value.c_str(), fld_value.length(), &current_index, &pEncoding) == 0)
								{
									encoding = pEncoding->enc_type;
									mailmime_mechanism_free(pEncoding);
								}
							}
							else
							{
								if (fld_name == "content-type")
									charset = InputImpl::ExtractCharsetFromContentType(pField->fld_data.fld_optional_field->fld_value);
							}
						}
					}
				}
				mailimf_fields_free(pImfFields);
			}
			mailmessage_fetch_result_free(pMessage, header);
		}

		// ignore mails with wrong subject
		if (sbjct != subject)
			return;

		// parse message body
		{
			char* body;
			size_t len;
			if (mailmessage_fetch_body(pMessage, &body, &len) == 0)
			{
				istringstream ss(body);
				string line;
				while (getline(ss, line))
				{
					trim(line);
					if (line.empty())
						continue;

					// decode the line
					switch (encoding)
					{
						case MAILMIME_MECHANISM_BASE64:
						{
							size_t index = 0;
							char* buffer;
							size_t len;
							if (mailmime_base64_body_parse(line.c_str(), line.length(), &index, &buffer, &len) == 0)
								line = buffer;
							mmap_string_unref(buffer);
							break;
						}
						case MAILMIME_MECHANISM_QUOTED_PRINTABLE:
						{
							size_t index = 0;
							char* buffer;
							size_t len;
							if (mailmime_quoted_printable_body_parse(line.c_str(), line.length(), &index, &buffer, &len, 0) == 0)
								line = buffer;
							mmap_string_unref(buffer);
							break;
						}
						case MAILMIME_MECHANISM_BINARY:
						{
							size_t index = 0;
							char* buffer;
							size_t len;
							if (mailmime_binary_body_parse(line.c_str(), line.length(), &index, &buffer, &len) == 0)
								line = buffer;
							mmap_string_unref(buffer);
							break;
						}
						// MAILMIME_MECHANISM_ERROR
						// MAILMIME_MECHANISM_7BIT
						// MAILMIME_MECHANISM_8BIT
						// MAILMIME_MECHANISM_TOKEN
						default:
							// leave the line unchanged, good luck!
							break;
					}

					// transform charset into UTF-8
					if (!charset.empty() && charset != "utf-8")
					{
						char* buffer;
						size_t len;
						if (charconv_buffer("utf-8", charset.c_str(), line.c_str(), line.length(), &buffer, &len) == 0)
							line = buffer;
						charconv_buffer_free(buffer);
					}

					// insert into database
					try
					{
						data = true;
						Database::CollectionsItems items;
						InputImpl::Parse(line, timestamp, verbose, items);
						database.AddCollectionsItems(items);
					}
					catch (const Xception& )
					{
						database.AddRejected(timestamp, line);
						rejects = true;
					}
				}
			}
			mailmessage_fetch_result_free(pMessage, body);
		}

		pFlags->fl_flags |= MAIL_FLAG_SEEN;
		if (!keep)
			pFlags->fl_flags |= MAIL_FLAG_DELETED;
		mailmessage_check(pMessage);
	}

void ImapMailClient::Do (const Config& config, Database& database) const throw (Xception)
{
	bool verbose = config.GetBoolOption("verbose");
	if (verbose)
		cout << "looking for mail (IMAP)" << endl;

	// get configuration data
	const string& server = config.GetStringOption(m_option_key + ".server");
	if (server.empty())
		throw Xception(format("missing specification in configuration file: %s.server)") % m_option_key);

	int port = (config.HasOption(m_option_key + ".port"))
		? (int)config.GetDoubleOption(m_option_key + ".port")
		: 143;

	const string& account = config.GetStringOption(m_option_key + ".account");
	if (account.empty())
		throw Xception(format("missing specification in configuration file: %s.account)") % m_option_key);

	const string& password = config.GetStringOption(m_option_key + ".password");

	const string& folder = (config.HasOption(m_option_key + ".folder"))
		? config.GetStringOption(m_option_key + ".folder")
		: "INBOX";

	const string& subject = config.GetStringOption(m_option_key + ".subject");

	bool keep = config.GetBoolOption(m_option_key + ".keep");
	bool expunge = config.GetBoolOption(m_option_key + ".expunge");

	// request the IMAP server
	string msg;
	bool data = false;
	bool rejects = false;
	{
		struct mailstorage* pMailStorage = mailstorage_new(NULL);
		if (imap_mailstorage_init(pMailStorage, server.c_str(), port, NULL, CONNECTION_TYPE_STARTTLS, IMAP_AUTH_TYPE_PLAIN, account.c_str(), password.c_str(), 0, NULL) == 0 &&
		    mailstorage_connect(pMailStorage) == 0)
		{
			struct mailfolder* pMailFolder = mailfolder_new(pMailStorage, folder.c_str(), NULL);
			if (mailfolder_connect(pMailFolder) == 0)
			{
				struct mailmessage_list* pMailMessagesList;
				if (mailfolder_get_messages_list(pMailFolder, &pMailMessagesList) == 0)
				{
					size_t n = carray_count(pMailMessagesList->msg_tab);
					for (size_t i = 0; i < n; i++)
					{
						mailmessage* pMessage = (mailmessage*)carray_get(pMailMessagesList->msg_tab, i);
						handle_message(pMessage, subject, database, verbose, keep, data, rejects);
						mailmessage_flush(pMessage);
					}
				}
				else
					msg = "could not retrieve messages from imap server";
				mailmessage_list_free(pMailMessagesList);

				if (expunge)
					mailfolder_expunge(pMailFolder);

				mailfolder_disconnect(pMailFolder);
			}
			else
				msg = (format("could not open mail folder %s on imap server") % folder).str();

			mailfolder_free(pMailFolder);
			mailstorage_disconnect(pMailStorage);
		}
		else
			msg = (format("could not connect to imap server %s on port %d") % server % port).str();
		mailstorage_free(pMailStorage);
	}

	if (verbose && !data)
		cout << "    no data on server" << endl;

	if (rejects)
		throw Xarning("!!! at least one expression has been rejected, try -r to list rejections !!!");

	if (!msg.empty())
		throw Xception(msg);
}

#endif
