// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;

#include "../libpdrx/datatypes.h"
#include "../libpdrx/config.h"
#include "db.h"
#include "out_impl.h"
#include "out_ftree.h"

//#define	BOOST_SPIRIT_DEBUG

#include <boost/spirit/include/classic_core.hpp>
#include <boost/spirit/include/classic_chset.hpp>
#include <boost/spirit/include/classic_regex.hpp>
#include <boost/spirit/include/classic_rule.hpp>
#include <boost/spirit/include/classic_push_back_actor.hpp>

using namespace BOOST_SPIRIT_CLASSIC_NS;

//=== OutputImpl (abstract base class) =====================================
OutputImpl::OutputImpl (const string& option_key)
	: m_option_key(option_key)
{
}

string OutputImpl::GetVerboseName () const
{
	return m_option_key;
}

namespace Parser {

#define EXEC(fo)	Grammar::fo(const_cast<Grammar&>(self))
#define EXEC2(fo,x)	Grammar::fo(const_cast<Grammar&>(self), x)

	// the grammar of the spirit parser
	struct Grammar: public grammar<Grammar> {

		FTree*		m_result;
		vector<FExpr*>	m_stack;

		Grammar ()
			: m_result(NULL)
			, m_stack()
		{
		}

		struct fo_double {
			Grammar& m_g;
			fo_double (Grammar& g)
				: m_g(g)
			{
			}
			void operator () (string::const_iterator begin, string::const_iterator end) const
			{
				m_g.m_stack.back()->AddParam(new FDouble(lexical_cast<double>(string(begin, end))));
			}
		};

		struct fo_datetime {
			Grammar& m_g;
			fo_datetime (Grammar& g)
				: m_g(g)
			{
			}
			void operator () (string::const_iterator begin, string::const_iterator end) const
			{
				static const regex rx("(?:([0-9]+)\\-)?([0-9]+)\\-([0-9]+)\\-([0-9]+)\\:([0-9]+)(?:\\:([0-9]+))?");
				smatch mr;
				string s(begin, end);
				regex_match(s, mr, rx);
				int year, month, day, hour, minute, second;
				if (mr.length(1) > 0)
					year = lexical_cast<int>(mr[1]);
				else
					year = day_clock::local_day().year();
				month = lexical_cast<int>(mr[2]);
				day = lexical_cast<int>(mr[3]);
				hour = lexical_cast<int>(mr[4]);
				minute = lexical_cast<int>(mr[5]);
				if (mr.length(6) > 0)
					second = lexical_cast<int>(mr[6]);
				else
					second = 0;
				m_g.m_stack.back()->AddParam(new FDateTime(ptime(date(year, month, day), time_duration(hour, minute, second, 0))));
			}
		};

		struct fo_date {
			Grammar& m_g;
			fo_date (Grammar& g)
				: m_g(g)
			{
			}
			void operator () (string::const_iterator begin, string::const_iterator end) const
			{
				static const regex rx("(?:([0-9]+)\\-)?([0-9]+)\\-([0-9]+)");
				smatch mr;
				string s(begin, end);
				regex_match(s, mr, rx);
				int year, month, day;
				if (mr.length(1) > 0)
					year = lexical_cast<int>(mr[1]);
				else
					year = day_clock::local_day().year();
				month = lexical_cast<int>(mr[2]);
				day = lexical_cast<int>(mr[3]);
				m_g.m_stack.back()->AddParam(new FDate(date(year, month, day)));
			}
		};

		struct fo_time {
			Grammar& m_g;
			fo_time (Grammar& g)
				: m_g(g)
			{
			}
			void operator () (string::const_iterator begin, string::const_iterator end) const
			{
				static const regex rx("([0-9]+)\\:([0-9]+)(?:\\:([0-9]+))?");
				smatch mr;
				string s(begin, end);
				regex_match(s, mr, rx);
				int hour, minute, second;
				hour = lexical_cast<int>(mr[1]);
				minute = lexical_cast<int>(mr[2]);
				if (mr.length(3) > 0)
					second = lexical_cast<int>(mr[3]);
				else
					second = 0;
				m_g.m_stack.back()->AddParam(new FTime(time_duration(hour, minute, second, 0)));
			}
		};

		struct fo_string {
			Grammar& m_g;
			fo_string (Grammar& g)
				: m_g(g)
			{
			}
			void operator () (string::const_iterator begin, string::const_iterator end) const
			{
				string s(++begin, --end);
				replace_all(s, "\"\"", "\"");
				m_g.m_stack.back()->AddParam(new FString(s));
			}
		};

		struct fo_formatspec {
			Grammar& m_g;
			fo_formatspec (Grammar& g)
				: m_g(g)
			{
			}
			void operator () (string::const_iterator begin, string::const_iterator end) const
			{
				m_g.m_stack.back()->AddParam(new FFormatSpec(string(++begin, --end)));
			}
		};

		struct fo_int {
			Grammar& m_g;
			fo_int (Grammar& g)
				: m_g(g)
			{
			}
			void operator () (int i) const
			{
				m_g.m_stack.back()->AddParam(new FInt(i));
			}
		};

		struct fo_color {
			Grammar& m_g;
			fo_color (Grammar& g)
				: m_g(g)
			{
			}
			void operator () (size_t color) const
			{
				m_g.m_stack.back()->AddParam(new FColor(color));
			}
		};

		struct fo_keyword {
			Grammar& m_g;
			fo_keyword (Grammar& g)
				: m_g(g)
			{
			}
			void operator () (string::const_iterator begin, string::const_iterator end) const
			{
				m_g.m_stack.back()->AddParam(new FKeyword(string(begin, end)));
			}
		};

		struct fo_open_expr {
			Grammar& m_g;
			fo_open_expr (Grammar& g)
				: m_g(g)
			{
			}
			void operator () (char ) const
			{
				FExpr* pFExpr = new FExpr();
				if (!m_g.m_stack.empty())
					m_g.m_stack.back()->AddParam(pFExpr);
				m_g.m_stack.push_back(pFExpr);
			}
		};

		struct fo_close_expr {
			Grammar& m_g;
			fo_close_expr (Grammar& g)
				: m_g(g)
			{
			}
			void operator () (char ) const
			{
				FExpr* pFExpr = m_g.m_stack.back();
				m_g.m_stack.pop_back();
				if (m_g.m_stack.empty())
					m_g.m_result = pFExpr;
			}
		};

		struct fo_function_name {
			Grammar& m_g;
			fo_function_name (Grammar& g)
				: m_g(g)
			{
			}
			void operator () (string::const_iterator begin, string::const_iterator end) const
			{
				m_g.m_stack.back()->SetFunction(string(begin, end));
			}
		};

	        template <typename ScannerT> struct definition {

			uint_parser<unsigned, 16, 1, 6> color_p;

			definition (const Grammar& self)
			{
				constant	= regex_p("[+-]?[0-9]+\\.[0-9]+")					[EXEC(fo_double)]	// double
						| regex_p("([0-9]+\\-)?[0-9]+\\-[0-9]+\\-[0-9]+\\:[0-9]+(\\:[0-9]+)?")	[EXEC(fo_datetime)]	// [YYYY-]MM-DD-hh:mm[:ss]
						| regex_p("([0-9]+\\-)?[0-9]+\\-[0-9]+")				[EXEC(fo_date)]		// [YYYY-]MM-DD
						| regex_p("[0-9]+\\:[0-9]+(\\:[0-9]+)?")				[EXEC(fo_time)]		// hh:mm[:ss]
						| regex_p("\"(?:[^\"]|(?:\"\"))*\"")					[EXEC(fo_string)]	// "..."
						| regex_p("<[^>]*>")							[EXEC(fo_formatspec)]	// "<...>"
						| int_p									[EXEC(fo_int)]		// 1
						| ch_p('#') >> color_p							[EXEC(fo_color)]	// #012ABC
						| regex_p("[a-z_][a-z_-]*")						[EXEC(fo_keyword)]
						;

				parameter	= function
						| constant
						;

				function_name	= regex_p("[A-Za-z_][A-Za-z0-9_-]*")					[EXEC(fo_function_name)]
						| str_p("+")								[EXEC(fo_function_name)]
						| str_p("-")								[EXEC(fo_function_name)]
						| str_p("*")								[EXEC(fo_function_name)]
						| str_p("/")								[EXEC(fo_function_name)]
						| str_p("==")								[EXEC(fo_function_name)]
						| str_p("!=")								[EXEC(fo_function_name)]
						| str_p("<=")								[EXEC(fo_function_name)]
						| str_p(">=")								[EXEC(fo_function_name)]
						| str_p("<")								[EXEC(fo_function_name)]
						| str_p(">")								[EXEC(fo_function_name)]
						;

				function	= ch_p('(')
															[EXEC(fo_open_expr)]
						  >> function_name >> *parameter >> ch_p(')')
															[EXEC(fo_close_expr)]
						;
			}

			rule<ScannerT>	function, function_name, parameter, constant;

			const rule<ScannerT>& start () const
			{
				return function;
			}
		};
	};

} // namespace Parser

FTree* OutputImpl::Parse (const string& input) throw (Xception)
{
	string i(input);
	Parser::Grammar g;
	parse_info<string::iterator> pi = parse(i.begin(), i.end(), g, space_p);

	if (!pi.full)
	{
		if (!g.m_stack.empty())
		{
			delete g.m_stack.front();
			g.m_stack.clear();
		}

		string left(i.begin(), pi.stop);
		if (left.length() > 35)
		{
			left.erase(0, left.length() - 35);
			left.insert(0, "... ");
		}

		string right(pi.stop, i.end());
		if (right.length() > 35)
		{
			right.erase(35);
			right += " ...";
		}

		if (right.empty())
			throw Xception("unexpected end of input");
		else
		{
			stringstream ss;
			ss << "syntax error" << endl;
			ss << left << endl;
			if (left.length() > 5)
				ss << string(left.length() - 5, ' ');
			ss << "---> " << right;
			throw Xception(ss.str());
		}
	}

	return g.m_result;
}
