// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;

#include "../libpdrx/datatypes.h"
#include "../libpdrx/config.h"
#include "db.h"
#include "in_impl.h"

//=== Input (abstract base class) ==========================================
Input::~Input ()
{
}

//=== InputFactory =========================================================
InputFactory::InputFactory ()
	: m_inputs()
	, m_initialized(false)
{
}

InputFactory::~InputFactory ()
{
	foreach (Input* pInput, m_inputs)
	{
		delete pInput;
	}
}

const InputFactory::Inputs& InputFactory::GetInputs (const Config& config) throw (Xception)
{
	if (!m_initialized)
	{
		m_initialized = true;

		if (!config.HasOption("none"))
		{
			// evaluate the inputs line in the configuration file
			const string& inputs = config.GetStringOption("inputs");
			if (!inputs.empty())
			{
				char_separator<char> sep(",");
				tokenizer<char_separator<char> > tok(inputs, sep);
				foreach (string input_name, tok)
				{
					trim(input_name);

					const string& input_type = config.GetStringOption(input_name + ".type");
					if (input_type.empty())
						throw Xception(format("input %s has no type information") % input_name);

					if (input_type == "pop3")
					{
						m_inputs.push_back(new Pop3MailClient(input_name));
						continue;
					}
#ifdef USE_ETPAN
					if (input_type == "imap")
					{
						m_inputs.push_back(new ImapMailClient(input_name));
						continue;
					}
#endif
					if (input_type == "txt")
					{
						m_inputs.push_back(new TXTFile(input_name));
						continue;
					}

					if (input_type == "csv")
					{
						m_inputs.push_back(new CSVFile(input_name));
						continue;
					}

					if (input_type == "xml")
					{
						m_inputs.push_back(new XMLFile(input_name));
						continue;
					}

					throw Xception(format("input %s has unknown type: %s") % input_name % input_type);
				}
			}
		}

		// we always check the command line
		m_inputs.push_back(new CmdLineExpr());

		// evaluate command line parameters
		if (config.HasOption("txt"))
			m_inputs.push_back(new TXTFile(""));

		if (config.HasOption("csv"))
			m_inputs.push_back(new CSVFile(""));

		if (config.HasOption("xml"))
			m_inputs.push_back(new XMLFile(""));
	}

	return m_inputs;
}

Input* InputFactory::GetInputInteractive () throw (Xception)
{
	return new InputInteractive("interactive");
}
