// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;

#include "../libpdrx/datatypes.h"
#include "../libpdrx/config.h"
#include "../libpdrx/encoding.h"
#include "diagram_impl.h"

using namespace LibBoard;

#define	COLOR(rgb)	Color(((rgb >> 16) & 0x0FF), ((rgb >> 8) & 0x0FF), (rgb & 0x0FF))

#define	LINE_THICKNESS	0.25

//=== DiagramSVG ===========================================================
DiagramSVG::DiagramSVG ()
	: m_canvas()
	, m_shapes()
	, m_width(0.0)
	, m_height(0.0)
	, m_xmin(not_a_date_time)
	, m_xmax(not_a_date_time)
	, m_xunit((XUnit)-1)
	, m_ymin(0.0)
	, m_ymax(0.0)
	, m_yunit(0.0)
{
	m_canvas.setUnit(Board::PT);
}

void DiagramSVG::OpenCanvas (size_t width, size_t height, RGB bgColor)
{
	// - set a size,
	// - fill the background

	m_width = width;
	m_height = height;

	m_canvas.clear(COLOR(bgColor));
}

void DiagramSVG::CloseCanvas ()
{
	m_canvas << m_shapes;
}

void DiagramSVG::DrawAxes (const ptime& x_min, const ptime& x_max, XUnit x_unit, double y_min, double y_max, double y_unit, RGB color)
{
	m_xmin = x_min;
	m_xmax = x_max;
	m_xunit = x_unit;
	m_ymin = y_min;
	m_ymax = y_max;
	m_yunit = y_unit;

	// approximate character dimensions
	//
	// Note: libboard doesn't compute real dimensions for text, if we
	// put text on a shape and ask the shape about the size of the text
	// it returns (0,0), so we must guess the dimensions, mysterious:
	// it seems that we get good results if width and heightare both
	// 5.0 units
	//
	const double char_width = 5.0, char_height = 5.0;

	// Step 1: x-axis
	time_duration increment(not_a_date_time);
	NormalizeX(m_xmin, increment);

	const double x_0 = CalcX(m_xmin);
	m_shapes << Line(x_0, 0.0, CalcX(m_xmax + increment), 0.0, COLOR(color), LINE_THICKNESS);

	const double dash_height = m_height / 100.0; // one percent of m_height
	if (m_xunit == months)
	{
		// the problem with months is that they differ in their
		// lengths, so we must rather increment the month in the
		// date of the first day than adding just a fixed length as
		// in all the other cases below
		for (ptime t(m_xmin); t <= m_xmax; )
		{
			double x = CalcX(t);
			m_shapes << Line(x, 0.0, x, - dash_height, COLOR(color), LINE_THICKNESS);
			string s(lexical_cast<string, ptime>(t), 0, 7);
			Text text(x + (char_height / 2.0), - dash_height - (s.length() * char_width), s, Fonts::Helvetica, 8.0, COLOR(color));
			text.rotate(1.5707963); // 90°
			m_shapes << text;
			const date& d = t.date();
			greg_month m = d.month();
			t = ptime((m == 12) ? date(d.year() + 1, 1, 1) : date(d.year(), m + 1, 1), time_duration(0, 0, 0, 0));
		}
	}
	else
	{
		for (ptime t(m_xmin); t <= m_xmax; t += increment)
		{
			double x = CalcX(t);
			m_shapes << Line(x, 0.0, x, - dash_height, COLOR(color), LINE_THICKNESS);
			string s(lexical_cast<string, ptime>(t));
			switch (m_xunit)
			{
				case years:	s.erase(4); break;
				case days:	s.erase(10); break;
				case hours:	s.erase(13); break;
				case minutes:	s.erase(16); break;
				default:	break;
			}
			Text text(x + (char_height / 2.0), - dash_height - (s.length() * char_width), s, Fonts::Helvetica, 8.0, COLOR(color));
			text.rotate(1.5707963); // 90°
			m_shapes << text;
		}
	}

	// Step 2: y-axis, much easier because of simple values
	//
	// we must draw the y-axis some pixels longer on the upper end to
	// avoid the top most number to be clipped, the shape will not
	// know about the fact that the top most number has a text height
	// that extends the axis
	//
	m_shapes << Line(x_0, CalcY(m_ymin), x_0, CalcY(m_ymax) + 5.0, COLOR(color), LINE_THICKNESS);

	const double dash_width = m_width / 100.0; // one percent of m_width

	string yf; // format string for the numbers on the y-axis
	{
		stringstream ss;
		ss << m_yunit;
		string s(ss.str());
		string::size_type pos = s.find('.');
		if (pos != string::npos)
		{
			s.erase(0, pos + 1);
			yf = (format("%%1.%df") % s.length()).str();
		}
		else
			yf = "%1.0f";
	}

	for (double t = m_ymin; t <= m_ymax; t += m_yunit)
	{
		double y = CalcY(t);
		m_shapes << Line(x_0 - dash_width, y, x_0, y, COLOR(color), LINE_THICKNESS);
		string s((format(yf) % t).str());
		m_shapes << Text(x_0 - dash_width - (s.length() * char_width), y - (char_height / 2.0), s, Fonts::Helvetica, 8.0, COLOR(color));
	}
}

	static string fold (const string& timestamp, Diagram::FoldInterval fi)
	{
		string result(timestamp);
		switch (fi)
		{
			case Diagram::year:	result.erase(0,  5); result.erase(2); break;
			case Diagram::month:	result.erase(0,  8); result.erase(2); break;
			case Diagram::day:	result.erase(0, 11); result.erase(2); break;
			case Diagram::hour:	result.erase(0, 14); result.erase(2); break;
			case Diagram::minute:	result.erase(0, 17); break;
		}
		return result;
	}

void DiagramSVG::DrawAxes (Diagram::FoldInterval fi, double y_min, double y_max, double y_unit, RGB color)
{
	// for folding we must handle dates in the year 9999, this is a bit
	// special but indeed no problem, we just need an according axis
	switch (fi)
	{
		case year:
		{
			m_xmin = ptime(date(9999,  Jan, 1), time_duration(0, 0, 0));
			m_xmax = ptime(date(10000, Jan, 1), time_duration(0, 0, 0));
			m_xunit = months;
			break;
		}
		case month:
		{
			m_xmin = ptime(date(9999, Jan,  1), time_duration(0, 0, 0));
			m_xmax = ptime(date(9999, Feb,  1), time_duration(0, 0, 0));
			m_xunit = days;
			break;
		}
		case day:
		{
			m_xmin = ptime(date(9999, Jan,  1), time_duration(0, 0, 0));
			m_xmax = ptime(date(9999, Jan,  2), time_duration(0, 0, 0));
			m_xunit = hours;
			break;
		}
		case hour:
		{
			m_xmin = ptime(date(9999, Jan,  1), time_duration(0, 0, 0));
			m_xmax = ptime(date(9999, Jan,  1), time_duration(1, 0, 0));
			m_xunit = minutes;
			break;
		}
		case minute:
		{
			m_xmin = ptime(date(9999, Jan,  1), time_duration(0, 0, 0));
			m_xmax = ptime(date(9999, Jan,  1), time_duration(0, 1, 0));
			m_xunit = seconds;
			break;
		}
	}

	m_ymin = y_min;
	m_ymax = y_max;
	m_yunit = y_unit;

	// approximate character dimensions
	//
	// Note: libboard doesn't compute real dimensions for text, if we
	// put text on a shape and ask the shape about the size of the text
	// it returns (0,0), so we must guess the dimensions, mysterious:
	// it seems that we get good results if width and heightare both
	// 5.0 units
	//
	const double char_width = 5.0, char_height = 5.0;

	// Step 1: x-axis
	time_duration increment(not_a_date_time);
	NormalizeX(m_xmin, increment);

	const double x_0 = CalcX(m_xmin);
	m_shapes << Line(x_0, 0.0, CalcX(m_xmax + increment), 0.0, COLOR(color), LINE_THICKNESS);

	const double dash_height = m_height / 100.0; // one percent of m_height
	if (m_xunit == months)
	{
		// the problem with months is that they differ in their
		// lengths, so we must rather increment the month in the
		// date of the first day than adding just a fixed length as
		// in all the other cases below
		for (ptime t(m_xmin); t <= m_xmax; )
		{
			double x = CalcX(t);
			m_shapes << Line(x, 0.0, x, - dash_height, COLOR(color), LINE_THICKNESS);
			const string& s = fold(lexical_cast<string, ptime>(t), fi);
			Text text(x + (char_height / 2.0), - dash_height - (s.length() * char_width), s, Fonts::Helvetica, 8.0, COLOR(color));
			text.rotate(1.5707963); // 90°
			m_shapes << text;
			const date& d = t.date();
			greg_month m = d.month();
			t = ptime((m == 12) ? date(d.year() + 1, 1, 1) : date(d.year(), m + 1, 1), time_duration(0, 0, 0, 0));
		}
	}
	else
	{
		for (ptime t(m_xmin); t <= m_xmax; t += increment)
		{
			double x = CalcX(t);
			m_shapes << Line(x, 0.0, x, - dash_height, COLOR(color), LINE_THICKNESS);
			const string& s = fold(lexical_cast<string, ptime>(t), fi);
			Text text(x + (char_height / 2.0), - dash_height - (s.length() * char_width), s, Fonts::Helvetica, 8.0, COLOR(color));
			text.rotate(1.5707963); // 90°
			m_shapes << text;
		}
	}

	// Step 2: y-axis, much easier because of simple values
	//
	// we must draw the y-axis some pixels longer on the upper end to
	// avoid the top most number to be clipped, the shape will not
	// know about the fact that the top most number has a text height
	// that extends the axis
	//
	m_shapes << Line(x_0, CalcY(m_ymin), x_0, CalcY(m_ymax) + 5.0, COLOR(color), LINE_THICKNESS);

	const double dash_width = m_width / 100.0; // one percent of m_width

	string yf; // format string for the numbers on the y-axis
	{
		stringstream ss;
		ss << m_yunit;
		string s(ss.str());
		string::size_type pos = s.find('.');
		if (pos != string::npos)
		{
			s.erase(0, pos + 1);
			yf = (format("%%1.%df") % s.length()).str();
		}
		else
			yf = "%1.0f";
	}

	for (double t = m_ymin; t <= m_ymax; t += m_yunit)
	{
		double y = CalcY(t);
		m_shapes << Line(x_0 - dash_width, y, x_0, y, COLOR(color), LINE_THICKNESS);
		string s((format(yf) % t).str());
		m_shapes << Text(x_0 - dash_width - (s.length() * char_width), y - (char_height / 2.0), s, Fonts::Helvetica, 8.0, COLOR(color));
	}
}

void DiagramSVG::DrawCurve (const Selection& s, RGB color, LineStyle style, double thickness, bool continuous, int bar1, int bar2)
{
	const size_t* pDimensions = s.shape(); // [0] rows, [1] columns

	if (pDimensions[0] == 1 && any_cast<string>(s[0][0]).empty())
		throw Xception("selection contains just one single value which is the result of an aggregation, you must ensure the selection to have multiple values to draw a curve");

	struct Dot {
		ptime	m_timestamp;
		any	m_value;
		Dot ()
			: m_timestamp(not_a_date_time)
			, m_value()
		{
		}
		Dot (const string& timestamp, const any& value)
			: m_timestamp(Diagram::CompletePtimeFromString(timestamp))
			, m_value(value)
		{
		}
		operator bool () const
		{
			return !m_value.empty();
		}
	};

	if (thickness == 0.0)
		thickness = LINE_THICKNESS;

	Dot last;
	for (size_t row = 0; row < pDimensions[0]; row++)
	{
		Dot current(any_cast<string>(s[row][0]), s[row][1]);

		switch (style)
		{
			case bars:
			{
				if (current)
				{
					if (bar1 != 0 && bar2 != 0 && bar1 <= bar2)
					{
						ptime timestamp_left(current.m_timestamp);
						time_duration increment(not_a_date_time);
						NormalizeX(timestamp_left, increment);
						ptime timestamp_right = timestamp_left + increment;
						double x = CalcX(timestamp_left);
						double x_width = (CalcX(timestamp_right) - x) / double(bar2);
						x += x_width * double(bar1 - 1);
						double y = CalcY(any_cast<double>(current.m_value));
						m_shapes << LibBoard::Rectangle(x, y, x_width, y - CalcY(m_ymin), COLOR(color), COLOR(color), 0);
					}
					else
					{
						double x = CalcX(current.m_timestamp);
						m_shapes << Line(x, CalcY(m_ymin), x, CalcY(any_cast<double>(current.m_value)), COLOR(color), (float)thickness);
					}
				}
				break;
			}
			case symbol_plus:
			{
				if (current)
				{
					double x = CalcX(current.m_timestamp);
					double y = CalcY(any_cast<double>(current.m_value));
					const double extent = m_width / 100.0; // one percent of m_width
					m_shapes << Line(x - extent, y, x + extent, y, COLOR(color), (float)thickness);
					m_shapes << Line(x, y - extent, x, y + extent, COLOR(color), (float)thickness);
				}
				break;
			}
			case symbol_vbar:
			{
				if (current)
				{
					double x = CalcX(current.m_timestamp);
					double y = CalcY(any_cast<double>(current.m_value));
					const double extent = m_width / 100.0; // one percent of m_width
					m_shapes << Line(x, y - extent, x, y + extent, COLOR(color), (float)thickness);
				}
				break;
			}
			case symbol_hbar:
			{
				if (current)
				{
					double x = CalcX(current.m_timestamp);
					double y = CalcY(any_cast<double>(current.m_value));
					const double extent = m_width / 100.0; // one percent of m_width
					m_shapes << Line(x - extent, y, x + extent, y, COLOR(color), (float)thickness);
				}
				break;
			}
			case symbol_cross:
			{
				if (current)
				{
					double x = CalcX(current.m_timestamp);
					double y = CalcY(any_cast<double>(current.m_value));
					const double extent = m_width / 100.0; // one percent of m_width
					m_shapes << Line(x - extent, y - extent, x + extent, y + extent, COLOR(color), (float)thickness);
					m_shapes << Line(x - extent, y + extent, x + extent, y - extent, COLOR(color), (float)thickness);
				}
				break;
			}
			case symbol_circle:
			{
				if (current)
				{
					double x = CalcX(current.m_timestamp);
					double y = CalcY(any_cast<double>(current.m_value));
					const double extent = m_width / 100.0; // one percent of m_width
					m_shapes << Circle(x, y, extent, COLOR(color), Color(false), (float)thickness);
				}
				break;
			}
			default: // zick-zack
			{
				if (last && current)
				{
					m_shapes << Line(
						CalcX(last.m_timestamp), CalcY(any_cast<double>(last.m_value)),
						CalcX(current.m_timestamp), CalcY(any_cast<double>(current.m_value)),
						COLOR(color), (float)thickness
					);
				}
				break;
			}
		}

		if (current || !continuous)
			last = current;
	}
}

void DiagramSVG::DrawVLine (const ptime& timestamp, RGB color, double thickness)
{
	double x = CalcX(timestamp);
	m_shapes << Line(x, CalcY(m_ymin), x, CalcY(m_ymax), COLOR(color), (float)((thickness == 0.0) ? LINE_THICKNESS : thickness));
}

void DiagramSVG::DrawHLine (double value, RGB color, double thickness)
{
	double y = CalcY(value);

	// in x-direction m_xmax marks the last number on the x-axis but
	// the axis is an x-unit longer, normally a day or an hour, so we
	// need to add one x-unit to get the full length of a hline
	ptime dummy(m_xmin);
	time_duration increment(not_a_date_time);
	NormalizeX(dummy, increment);

	m_shapes << Line(CalcX(m_xmin), y, CalcX(m_xmax + increment), y, COLOR(color), (float)((thickness == 0.0) ? LINE_THICKNESS : thickness));
}

void DiagramSVG::SaveAs (const string& filename) const
{
	m_canvas.saveSVG(filename.c_str(), Board::BoundingBox);
}

double DiagramSVG::CalcX (const ptime& x) const
{
	double a = 0.0;
	{
		const time_duration& d = x - m_xmin;
		a = (d.hours() * 60 * 60) + (d.minutes() * 60) + d.seconds();
	}
	double b = 0.0;
	{
		const time_duration& d = m_xmax - m_xmin;
		b = (d.hours() * 60 * 60) + (d.minutes() * 60) + d.seconds();
	}
	return (a / b) * m_width;
}

double DiagramSVG::CalcY (double y) const
{
	double a = y - m_ymin;
	double b = m_ymax - m_ymin;
	return (a / b) * m_height;
}

void DiagramSVG::NormalizeX (ptime& x, time_duration& increment) const
{
	// this method adjusts an odd x-timestamp to the beginning of a
	// x-unit, mostly a day or an hour, and determines the x-increment
	// to get the next x-unit

	switch (m_xunit)
	{
		case years:
		{
			x = ptime(date(x.date().year(), 1, 1), time_duration(0, 0, 0, 0));
			increment = posix_time::hours(365 * 24);
			break;
		}
		case months:
		{
			x = ptime(date(x.date().year(), x.date().month(), 1), time_duration(0, 0, 0, 0));
			increment = posix_time::hours(30 * 24);
			break;
		}
		case days:
		{
			x = ptime(x.date(), time_duration(0, 0, 0, 0));
			increment = posix_time::hours(24);
			break;
		}
		case hours:
		{
			x = ptime(x.date(), time_duration(x.time_of_day().hours(), 0, 0, 0));
			increment = posix_time::hours(1);
			break;
		}
		case minutes:
		{
			x = ptime(x.date(), time_duration(x.time_of_day().hours(), x.time_of_day().minutes(), 0, 0));
			increment = posix_time::minutes(1);
			break;
		}
		default:
		{
			x = ptime(x.date(), time_duration(x.time_of_day().hours(), x.time_of_day().minutes(), x.time_of_day().seconds(), 0));
			increment = posix_time::seconds(1);
			break;
		}
	}
}
