// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#include "../libpdrx/common.h"

using namespace std;
using namespace boost;
using namespace boost::posix_time;
using namespace boost::gregorian;
using namespace boost::program_options;
using namespace boost::filesystem;

#include "../libpdrx/datatypes.h"
#include "../libpdrx/config.h"
#include "db.h"
#include "in_impl.h"

//=== InputImpl (abstract base class) ======================================
InputImpl::InputImpl (const string& option_key)
	: m_option_key(option_key)
{
}

void InputImpl::Parse (const string& expr, const ptime& timestamp, bool verbose, Database::CollectionElements& elements) const throw (Xception)
{
	string line(expr);

	static const regex rx_comment("[;#]\\s*(.+)\\s*$");					// #... ;...
	static const regex rx_time("([0-9]{1,2})[:]([0-9]{1,2})(?:[:]([0-9]{1,2}))?");		// 15:27[:59]
	static const regex rx_date("([0-9]{1,4})-([0-9]{1,2})-([0-9]{1,2})");			// 2009-02-27
	static const regex rx_ratio("([0-9]+(?:\\.[0-9]+)?)([/%])([0-9]+(?:\\.[0-9]+)?)");	// double[/%]double
	static const regex rx_numeric("([0-9]+(?:\\.[0-9]+)?)("RX_COLLECTION_NAME")?");		// double A-Z a-z _ * + ! ? ^   $ & [ ] { } = ~

	smatch mr;

	// try to find a comment
	any comment;
	if (regex_search(line, mr, rx_comment, boost::match_not_dot_newline))
	{
		comment = string(mr[1]);
		line.erase(mr.position(), mr.length());
	}

	// if the expression contains date and/or time components, adapt the
	// timestamp variable
	ptime t(timestamp);
	if (regex_search(line, mr, rx_time, boost::match_not_dot_newline))
	{
		// time found
		{
			int hour, min, sec;
			hour = lexical_cast<int>(mr[1]);
			min = lexical_cast<int>(mr[2]);
			sec = (mr[3].length() > 0) ? lexical_cast<int>(mr[3]) : 0;
			t -= t.time_of_day();
			t += time_duration(hour, min, sec, 0);
			line.erase(mr.position(), mr.length());
		}

		if (regex_search(line, mr, rx_date, boost::match_not_dot_newline))
		{
			// date found
			int year, month, day;
			year = lexical_cast<int>(mr[1]);
			month = lexical_cast<int>(mr[2]);
			day = lexical_cast<int>(mr[3]);
			t = ptime(date(year, month, day), t.time_of_day());
			line.erase(mr.position(), mr.length());
		}
	}

	if (verbose)
		cout << (format("    [%s] %s") % lexical_cast<string,ptime>(t) % expr).str() << endl;

	// now build a data vector for a database insert
	if (!comment.empty())
		elements.push_back(Database::CollectionElement("#", t, comment));

	while (regex_search(line, mr, rx_ratio, boost::match_not_dot_newline))
	{
		elements.push_back(Database::CollectionElement(mr[2], t, Ratio(lexical_cast<double>(mr[1]), lexical_cast<double>(mr[3]))));
		line.erase(mr.position(), mr.length());
	}

	while (regex_search(line, mr, rx_numeric, boost::match_not_dot_newline))
	{
		string name(mr[2]);
		if (name.empty())
			name = '*';
		elements.push_back(Database::CollectionElement(name, t, lexical_cast<double>(mr[1])));
		line.erase(mr.position(), mr.length());
	}

	// last check
	trim(line);
	if (!line.empty())
		throw Xception(format("expression contains unrecognized characters: %s") % line);
}

//=== FileInputImpl (base class) ===========================================
FileInputImpl::FileInputImpl (const string& option_key, const string& vector_key)
	: InputImpl(option_key)
	, m_vector_key(vector_key)
{
}

void FileInputImpl::Do (const Config& config, Database& database) const throw (Xception)
{
	// get configuration data
	vector<string> filenames;
	bool keep = false;
	{
		if (m_option_key.empty())
		{
			// get all filenames from command line
			filenames = config.GetVectorOption(m_vector_key);
			keep = true;
		}
		else
		{
			// get the configured filename from config file
			const string& f = config.GetStringOption(m_option_key + ".filename");
			if (f.empty())
				throw Xception(format("missing specification in configuration file: %s.filename") % m_option_key);

			// check if there are wild cards, if so iterate
			// over the files and fill the filenames vector
			if (f.find('*') != string::npos || f.find('?') != string::npos)
			{
				string filemask(path(f).stem());
				{
					string extension(path(f).extension());
					if (extension != ".")
						filemask += extension;
					replace_all(filemask, ".", "\\.");
					replace_all(filemask, "*", ".*");
					replace_all(filemask, "?", ".");
				}
				const path& dir = path(f).parent_path();
				if (is_directory(dir))
				{
					regex rx(filemask);
					for (directory_iterator I(dir); I != directory_iterator(); I++)
					{
						const path& p = (*I).path();
						if (regex_match(p.filename(), rx) && is_regular_file((*I).status()))
							filenames.push_back(p.string());
					}
				}
				else
					throw Xception(format("invalid path specification in configuration file: %s.filename") % m_option_key);
			}
			else
				filenames.push_back(f);

			keep = config.GetBoolOption(m_option_key + ".keep");
		}
	}

	// process file by file
	string msgs;
	foreach (const string& filename, filenames)
	{
		try
		{
			// collect collection elements
			Database::CollectionElements elements;
			{
				ifstream ifs(filename.c_str(), ios::in);
				if (!ifs.good())
					throw Xception("file not found");

				bool verbose = config.GetBoolOption("verbose");
				if (verbose)
					cout << "parsing " << m_vector_key << " file " << filename << endl;

				ProcessFile(config, database, ifs, elements);
			}

			// add the contents of the file in a single
			// transaction to the database
			database.AddCollectionElements(elements);

			// cleanup
			if (!keep)
				remove(filename.c_str());
		}
		catch (const Xception& e)
		{
			if (!msgs.empty())
				msgs += '\n';
			msgs += e.Message() + ", skipping file " + filename;
		}
	}

	if (!msgs.empty())
		throw Xception(msgs);
}
