// This file is part of the pdr/pdx project.
// Copyright (C) 2010 Torsten Mueller, Bern, Switzerland
//
// This program is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as
// published by the Free Software Foundation, either version 2 of the
// License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.

#ifndef __OUT_FTREE_H
#define __OUT_FTREE_H

#include "out_func.h"

class Config;
class Database;
class Diagram;

//=== FContext classes (helper) ============================================
//
// This class defines a context for the evaluation of functions. We need
// references to the Config and the Database objects as minimum content but
// this class can be derived and can have more data elements. Further the
// Evaluate-Methods of the FTree classes have a none const FContext
// parameter, so the context can also be used for inter function
// communication but - hey, this is makes the functions depend on each
// other and on their calling order.
//
class FContext {

	public:

	const Config&	m_config;
	const Database& m_database;

		FContext (const Config& config, const Database& database);
	virtual ~FContext ();
};

class FDiagramContext: public FContext {

	public:

	Diagram&	m_diagram;

		FDiagramContext (const Config& config, const Database& database, Diagram& diagram);
};

//=== FTree classes ========================================================
//
// These classes describe the parser's output. The "F" means "functional".
// The parse function returns a tree of functions and their parameters. The
// parameters are themselves functions and/or constants.
//

//--- FTree (abstract base class) ------------------------------------------
class FTree {

	public:

		FTree ();
	virtual	~FTree ();

	// return the expression's return type
	virtual const type_info* Type () const throw (Xception) =0;

	// dump the expression's structure
	virtual	void Dump (ostream& os, int indent) const throw (Xception) =0;

	// evaluate the expression completely and return a result
	virtual any Evaluate (FContext& context) const throw (Xception) =0;
};

//--- FOneLiner (abstract base class for constants and keywords) -----------
class FOneLiner: public FTree {

	public:

		FOneLiner ();
};

//--- FConst (abstract base class template for constants) ------------------
template <class T> class FConst: public FOneLiner {

	protected:

	T		m_value;

	public:

		FConst (const T& value)
			: FOneLiner()
			, m_value(value)
		{
		}

	virtual const type_info* Type () const throw (Xception)
	{
		return &typeid(m_value);
	}

	virtual any Evaluate (FContext& ) const throw (Xception)
	{
		return any(m_value);
	}
};

//--- the classes for constants --------------------------------------------
class FDouble: public FConst<double>
{
	public:

		FDouble (double value);

	virtual	void Dump (ostream& os, int indent) const throw (Xception);
};

class FDateTime: public FConst<ptime>
{
	public:

		FDateTime (const ptime& value);

	virtual	void Dump (ostream& os, int indent) const throw (Xception);
};

class FDate: public FConst<date>
{
	public:

		FDate (const date& value);

	virtual	void Dump (ostream& os, int indent) const throw (Xception);
};

class FTime: public FConst<time_duration>
{
	public:

		FTime (const time_duration& value);

	virtual	void Dump (ostream& os, int indent) const throw (Xception);
};

class FString: public FConst<string>
{
	public:

		FString (const string& value);

	virtual	void Dump (ostream& os, int indent) const throw (Xception);
};

	// this type is just a string, it is used internally to separate
	// string constants from format constants even in evaluated form
	class fstring: public string {
		public:
		fstring (const string& f)
			: string(f)
		{
		}
	};

class FFormatSpec: public FConst<fstring>
{
	public:

		FFormatSpec (const fstring& value);

	virtual	void Dump (ostream& os, int indent) const throw (Xception);
};

class FInt: public FConst<int>
{
	public:

		FInt (int value);

	virtual	void Dump (ostream& os, int indent) const throw (Xception);
};

class FColor: public FConst<size_t>
{
	public:

		FColor (size_t value);

	virtual	void Dump (ostream& os, int indent) const throw (Xception);
};

//--- FKeyword -------------------------------------------------------------

	// this type is just a string, it is used internally to separate
	// string constants from keywords even in evaluated form
	class kstring: public string {
		public:
		kstring (const string& k)
			: string(k)
		{
		}
	};

class FKeyword: public FOneLiner {

	protected:

	kstring		m_name;

	public:

		FKeyword (const string& name);

	virtual const type_info* Type () const throw (Xception);
	virtual	void Dump (ostream& os, int indent) const throw (Xception);
	virtual any Evaluate (FContext& context) const throw (Xception);
};

//--- FExpr ----------------------------------------------------------------
class FExpr: public FTree {

	public:

	typedef	vector<FTree*> Params;

	protected:

	string		m_func;
	Params		m_params;

	mutable const FImpl* m_pFImpl;

	public:

	static FImplFactory fImplFactory;

		FExpr ();
	virtual	~FExpr ();

	virtual const type_info* Type () const throw (Xception);
	virtual	void Dump (ostream& os, int indent) const throw (Xception);
	virtual any Evaluate (FContext& context) const throw (Xception);

		void SetFunction (const string& func);
		void AddParam (FTree* pFTree);
};

//=== helper functions =====================================================
ostream& operator << (ostream& os, const FTree& fTree) throw (Xception);
ostream& operator << (ostream& os, const any& a) throw (Xception);

#endif
