/*  
  Copyright (C) 2004 Sarod Yatawatta <sarod@users.sf.net>  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 $Id: hash.c,v 1.19 2004/08/21 21:42:05 sarod Exp $ 
*/


#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <math.h>
#include "types.h"


/* cutoff frequency and propagation constant used 
  in waveguide problems */
MY_DOUBLE global_wave_k0, global_wave_beta;

/* data structure for hash table */
typedef struct htbl_ {
 int positions;  /* no. of positions in table */
 void *vacated;

 /* hash functions used */
 int (*h1) (const void *key);
 int (*h2) (const void *key);
 /* function for comparison */
 int (*match) (const void *key1, const void *key2);
 /* function for deletion */
 void (*destroy) (void *data);

 /* actual elements filled */
 int size;
 /* element array */
 void **table;

} htbl;


/* function interface */
/* intialize hash table */
static int
htbl_init(htbl *tbl, int size, int (*h1)(const void *key),
          int (*h2) (const void *key), 
          int (*match)(const void *key1, const void *key2),
          void (*destroy)(void *data));

/* destoroy hash table */
static void
htbl_destroy(htbl* tbl);

/* insert data */
static int
htbl_insert(htbl *tbl, const void *data);

/* remove data */
/* returns null if not found. returns pointer to data record if found and removed*/
/* query record must be passed using **data */
static void *
htbl_remove(htbl *tbl, void **data);

/* lookup data */
/* returns null if not found. returns pointer to data record if found */
/* query record must be passed using **data */
static void *
htbl_lookup(const htbl *tbl, void *data);

/* returns -1 if not found. returns hash if edge found */
/* query record must be passed using **data */
static MY_INT
htbl_lookup_hash(const htbl *tbl, void *data);


/* macro for getting size of hash table */
#define htbl_size(tbl) ((tbl)->size);


/* sentinel to keep empty slots of */
/* hash table pointed */
static void *vacated;


/* intialize hash table of size 'size' */
static int
htbl_init(htbl *tbl, int size , int (*h1)(const void *key),
          int (*h2) (const void *key), 
          int (*match)(const void *key1, const void *key2),
          void (*destroy)(void *data))
{

 int i;

 /* allocate space for hash table */
 if ( ((tbl)->table = (void **)malloc(size*sizeof(void*))) == NULL )
       return(-1); /* no memory */

 /* now initialize table slots */
 (tbl)->positions = size;

 for ( i = 0; i < tbl->positions; i++ ) {
    tbl->table[i] = NULL;
 }
 /* set the vacated member to the sentinel memory address */
 tbl->vacated = &vacated;

 /* encapsulate the functions */
 /* hash functions */
 tbl->h1 = h1;
 tbl->h2 = h2;
  
 /* comparison */
 tbl->match = match;

 /* deletion */
 tbl->destroy = destroy;

 /* set current size to 0 */
 tbl->size = 0;

 return(0);

}


/* destroy hash table */
static void
htbl_destroy(htbl *tbl)
{

 int i;

 if ( tbl->destroy != NULL ) {
  /* calll used defined function */
  for ( i = 0; i < tbl->positions; i++ ) {
     if ( tbl->table[i] != NULL && tbl->table[i] != vacated ) {
        tbl->destroy(tbl->table[i]);
     }
  }
 }
 /* free the storage aloocated for hash tabel */
 free(tbl->table);

 /* clear the structure */
 memset(tbl,0,sizeof(htbl));  
 
 return;

}


/* insert value */
/* no error returns zero. not zero means error */
static int
htbl_insert(htbl *tbl, const void *data) {
 
 void *temp;
 unsigned int position, i;
 int val1,val2;


 /* chek to see if the table is full */
 if ( tbl->size == tbl->positions )
   return(-1);

 /* do nothing if the data is already in the table */

 temp = (void*)data;

if ( htbl_lookup(tbl,&temp) != 0 )
  return(1);  /* data already in */

 /* double hashing to get the key */
  val1=tbl->h1(data); val2=tbl->h2(data);
 for ( i = 0; i < tbl->positions; i++ ) {
   /* mod value */
   /*position = (tbl->h1(data) + (i*tbl->h2(data)))%tbl->positions; */
   position = (val1 + (i*val2))%tbl->positions;  
   /*position = (val1%tbl->positions + i*(val2%tbl->positions))%tbl->positions;  */
#ifdef DEBUG
	 printf("htbl_insert: pos %d\n",position);
#endif
  
   if (tbl->table[position] == NULL 
       || tbl->table[position] == tbl->vacated) {
      /* insert data here */
			/*if ( ( tbl->table[position] =(void*)malloc(sizeof(*data)))==0 ) {
				fprintf(stderr,"%s: %d: no free memory",__FILE__,__LINE__);
				exit(1);
		  }
      memcpy(tbl->table[position],data,sizeof(*data)); */
			tbl->table[position] = (void*)data;
      tbl->size++;
#ifdef DEBUG
	 printf("htbl_insert: insert at pos %d\n",position);
#endif
 
      return(0);
   }
 }

 /* if we reach here, hashing is not done properly */
 
 return(-1);

}


/* removing an item */
static void *
htbl_remove(htbl *tbl, void **data) 
{

 unsigned int position, i;
 void *temp;
   int val1,val2;
 /* use double hashing to get the key */

  val1=tbl->h1(*data); val2=tbl->h2(*data);
 for ( i =0; i < tbl->positions; i++) {
   /* mod value */
   /*position = (tbl->h1(*data) + (i*tbl->h2(*data)))%tbl->positions; */
   
   position = (val1 + (i*val2))%tbl->positions;  
   /*position = (val1%tbl->positions + i*(val2%tbl->positions))%tbl->positions; 
		*/
   if (tbl->table[position] == NULL) {
     /* no data to delete */
     return(0);
   } else if ( tbl->table[position] == tbl->vacated )  {
     /* search beyond vacated positions */
     continue;
   } else if ( tbl->match(tbl->table[position],*data) ) {
     /* pass back the data from the table */
     temp= tbl->table[position];
     tbl->table[position] = tbl->vacated;
     tbl->size--;
     return(temp);
   }

  }

  /* if we reach here, data not found */
  return(0);

}
 

/* table lookup */
static void *
htbl_lookup(const htbl *tbl, void *data) 
{

  unsigned int position,i;

   int val1,val2;
#ifdef DEBUG
	 printf("htbl_lookup:looking\n");
#endif

  /* use double hashing to get the key */ 
  val1=tbl->h1(data); val2=tbl->h2(data);
  for ( i = 0; i < tbl->positions; i++ ) {
   /* mod value */
  /* position = (tbl->h1(*data) + (i*tbl->h2(*data)))%tbl->positions; */
   position = (val1 + (i*val2))%tbl->positions;  
  /* position = (val1%tbl->positions + i*(val2%tbl->positions))%tbl->positions; 
	 */
#ifdef DEBUG
	 printf("htbl_lookup: pos %d\n",position);
#endif
   if (tbl->table[position] == NULL) {
      /* data not found */
#ifdef DEBUG
	 printf("htbl_lookup: not found\n");
#endif

      return(0);
   } else if ( tbl->match(tbl->table[position], data) ) { 
      /* pass back the data from the table */
      /**data = tbl->table[position]; */
      /*memcpy(*data,tbl->table[position],sizeof(*tbl->table[position]));  */
#ifdef DEBUG
	 printf("htbl_lookup: found matching key at %d\n",position);
#endif

    return((void*)tbl->table[position]);
   }
  
  }
#ifdef DEBUG
	 printf("htbl_lookup: not found error\n");
#endif

 
  /* if we reach here, data not found */
  return(0);

} 

/* table lookup hash */
static MY_INT
htbl_lookup_hash(const htbl *tbl, void *data) 
{

  unsigned int position,i;

   int val1,val2;
  /* use double hashing to get the key */ 
  val1=tbl->h1(data); val2=tbl->h2(data);
  for ( i = 0; i < tbl->positions; i++ ) {
   /* mod value */
  /* position = (tbl->h1(*data) + (i*tbl->h2(*data)))%tbl->positions; */
   position = (val1 + (i*val2))%tbl->positions;  
  /* position = (val1%tbl->positions + i*(val2%tbl->positions))%tbl->positions; 
	 */
#ifdef DEBUG
	 printf("htbl_lookup: pos %d\n",position);
#endif
   if (tbl->table[position] == NULL) {
      /* data not found */
#ifdef DEBUG
	 printf("htbl_lookup: not found\n");
#endif

      return(0);
   } else if ( tbl->match(tbl->table[position], data) ) { 
      /* pass back the data from the table */
      /**data = tbl->table[position]; */
      /*memcpy(*data,tbl->table[position],sizeof(*tbl->table[position]));  */
#ifdef DEBUG
	 printf("htbl_lookup: found matching key at %d\n",position);
#endif

    return(position);
   }
  
  }
#ifdef DEBUG
	 printf("htbl_lookup: not found error\n");
#endif

 
  /* if we reach here, data not found */
  return(-1);

}

typedef struct hash_edge_{
		MY_INT p0; /* point 0*/
		MY_INT p1; /* point 1*/
    MY_INT n; /* unique number */
} hash_edge;


/* define functions to use */
/* 32 bit integer hash function 1 */
/* by Thomas Wang */
static int 
h1(const void *key) 
{
 hash_edge *e;
 unsigned int k;

 e=(hash_edge*)key;
 k=(int)e->p0;
#ifdef DEBUG
 printf("hash 1: key value=%d\n",k);
#endif

 k+=~(k<<15);
 k^=(k>>10);
 k+=(k<<3);
 k^=(k>>6);
 k+=~(k<<11);
 k^=(k>>16);
#ifdef DEBUG
 printf("hash 1: value=%d\n",k);
#endif
 return( k);
}

/* second hash function is the same */
static int 
h2(const void *key) 
{
 hash_edge *e;
 unsigned int k;

 e=(hash_edge*)key;
 k=(int)e->p1;
#ifdef DEBUG
 printf("hash 2: key value=%d\n",k);
#endif

 k+=~(k<<15);
 k^=(k>>10);
 k+=(k<<3);
 k^=(k>>6);
 k+=~(k<<11);
 k^=(k>>16);
#ifdef DEBUG
 printf("hash 2: value=%d\n",k);
#endif
 return( k);
}

/* 64 bit hash function */
/*long longhash1(long key)
{
 key += ~(key << 32);
 key ^= (key >>> 22);
 key += ~(key << 13);
 key ^= (key >>> 8);
 key += (key << 3);
 key ^= (key >>> 15);
 key += ~(key << 27);
 key ^= (key >>> 31);
 return key;
} */
/* comparison function */
int
match(const void *key1, const void *key2)
{
 hash_edge *e1,*e2; 
 e1=(hash_edge*)key1;
 e2=(hash_edge*)key2;

 return((e1->p0==e2->p0) && (e1->p1==e2->p1)); 

}

/* freeing function */
void
destroy(void *data)
{
  if (data)
    free(data);
}


/* if edge (p0-p1) is on a dirichlet boundary, return 1.
 * else return 0 */
static MY_INT
is_this_edge_on_a_dirichlet_boundary(p0,p1)
{
  /* find the ratio using x=(x1+lambda. x2)/(1+lambda) */
	/* if x is on (x1-x2), lambda >0 , equal to y */
	/* try to do robust computations */
 int i;
	double xp0,xp1,xn0,xn1,yp0,yp1,yn0,yn1;
	xp0=Mx(p0);
	yp0=My(p0);
	xp1=Mx(p1);
	yp1=My(p1);

#ifdef DEBUG
	printf("is_this_edge_on_a_boundary:consider points(%d,%d)\n",p0,p1);
#endif
	for (i=0; i< nedges; i++) {
    xn0=Mx(edge_array[i].p1);
    yn0=My(edge_array[i].p1);
    xn1=Mx(edge_array[i].p2);
    yn1=My(edge_array[i].p2);
#ifdef DEBUG
	printf("is_this_edge_on_a_boundary: edge %d\n",i);
#endif
		if ( IS_ZERO((xn0-xp0)*(yp0-yn1)-(yn0-yp0)*(xp0-xn1))
			 /* ratio is equal magnitude, check sign */
			&& ( ((yn0-yp0)*(yp0-yn1)>0) /* if true, equal and +ve lambda. stop */
				|| (IS_ZERO((yn0-yp0)*(yp0-yn1)) 
							&& (xn0-xp0)*(xp0-xn1) >= 0))){
					/* if we are here, point p0 is on line */
					/* do the same check for point p1 */
#ifdef DEBUG
	printf("is_this_edge_on_a_boundary: point %d is on line %d\n",p0,i);
#endif
         if ( IS_ZERO((xn0-xp1)*(yp1-yn1)-(yn0-yp1)*(xp1-xn1))
			 /* ratio is equal magnitude, check sign */
			&& ( ((yn0-yp1)*(yp1-yn1)>0) /* if true, equal and +ve lambda. stop */
				|| (IS_ZERO((yn0-yp1)*(yp1-yn1)) 
							&& (xn0-xp1)*(xp1-xn1) >= 0))){
          if ( edge_array[i].type == DR ) { 
					 return(1);
          } else {
           return(0);
          }
				}
		 }

  }

	return(0); /* not found */

}




static MY_INT
get_edge_number(MY_INT p0,MY_INT p1, htbl H)
{
  hash_edge *ee,*gg;
  
  if ( (ee=(hash_edge*)malloc(sizeof(hash_edge)))==0) {
   fprintf(stderr,"%s: %d: no free memory\n",__FILE__,__LINE__);
	 exit(1);
	}
  if ( p0 > p1 ) {
	 ee->p0=p1;
	 ee->p1=p0;
  } else {
   ee->p0=p0;
	 ee->p1=p1;
  }
  gg=(hash_edge*)htbl_lookup(&H, (void *)ee);
  free(ee);
  if (gg) {
   return(gg->n);
  } else {
   printf("get_edge_number: something SERIOUSLY wrong! (%d,%d)\n",p0,p1);
   return(-1);
  }
}

/* solve for cutoff frequency given propagation constant */
static void
build_equations_from_hash_table_given_beta(htbl H, MY_DOUBLE  **S, MY_DOUBLE **T,
  MY_INT *renumber, MY_INT Ntotal, MY_INT Nedges, MY_DOUBLE beta)
{
 /* 
  S, T - Ntotal by Ntotal matrices
  Nedges - number of edges, the rest points 
  renumber - map points to matrix location
  edges are mapper using hash table H
*/
 triangle *t;
 MY_INT i,j,m,n;
 MY_DOUBLE x[3],y[3],a[3],b[3],c[3],x_bar,y_bar,x_var,y_var;
 MY_DOUBLE A[3],B[3],C[3],D[3],L[3],Ar,k;
 MY_DOUBLE S_tt[3][3],S_tz[3][3],S_zt[3][3],S_zz[3][3],T_tt[3][3],T_zz[3][3];
 MY_DOUBLE I1[3][3],I2[3][3],I3[3][3],I4[3][3],I5[3][3];
 MY_DOUBLE epsilon,sumI,mu;
 MY_INT e[3],mult[3];

 DAG_traverse_list_reset(&dt);
 t=DAG_traverse_prune_list(&dt);
 while(t) {
  if (t && t->status==LIVE) {
        epsilon= bound.poly_array[t->boundary].epsilon;
        mu= bound.poly_array[t->boundary].mu;
		   /* get three edges */
      /*              p0
			                  /\
					 	  				 /  \
				    edge 1    /    \  edge 3
                     /      \
					          /________\
                  p1 edge 2  p2
	   */

					/* get coords */
					/* convert to global coords */
					x[0]=Mx(t->p0)*g_xscale-g_xoff;
					y[0]=My(t->p0)*g_yscale-g_yoff;
					x[1]=Mx(t->p1)*g_xscale-g_xoff;
					y[1]=My(t->p1)*g_yscale-g_yoff;
					x[2]=Mx(t->p2)*g_xscale-g_xoff;
					y[2]=My(t->p2)*g_yscale-g_yoff;
 
					x_bar=(x[1]+x[2]+x[0])*ONE_OVER_THREE;
					y_bar=(y[1]+y[2]+y[0])*ONE_OVER_THREE;
					x_var=(x[1]*x[1]+x[2]*x[2]+x[0]*x[0]);
					y_var=(y[1]*y[1]+y[2]*y[2]+y[0]*y[0]);

					a[0]=x[1]*y[2]-x[2]*y[1];
					a[1]=x[2]*y[0]-x[0]*y[2];
					a[2]=x[0]*y[1]-x[1]*y[0];
					b[0]=y[1]-y[2];
					b[1]=y[2]-y[0];
					b[2]=y[0]-y[1];
					c[0]=x[2]-x[1];
					c[1]=x[0]-x[2];
					c[2]=x[1]-x[0];

     A[0]=a[0]*b[1]-a[1]*b[0];
     A[1]=a[1]*b[2]-a[2]*b[1];
     A[2]=a[2]*b[0]-a[0]*b[2];
					B[0]=c[0]*b[1]-c[1]*b[0];
					B[1]=c[1]*b[2]-c[2]*b[1];
					B[2]=c[2]*b[0]-c[0]*b[2];
					C[0]=a[0]*c[1]-a[1]*c[0];
					C[1]=a[1]*c[2]-a[2]*c[1];
					C[2]=a[2]*c[0]-a[0]*c[2];
					D[0]=-B[0];
					D[1]=-B[1];
					D[2]=-B[2];
     /* edge lengths */
					L[0]=sqrt(ABS(c[2]*c[2]+b[2]*b[2]));
					L[1]=sqrt(ABS(c[0]*c[0]+b[0]*b[0]));
					L[2]=sqrt(ABS(c[1]*c[1]+b[1]*b[1]));
				 /* area */
					Ar=0.5*(a[1]+a[2]+a[0]);

     for (m=0; m<3; m++) {
       for (n=0; n<3; n++) {
					   I1[m][n]=A[m]*A[n]+C[m]*C[n];
					   I2[m][n]=(C[m]*D[n]+C[n]*D[m])*x_bar;
					   I3[m][n]=(A[m]*B[n]+A[n]*B[m])*y_bar;
					   I4[m][n]=ONE_OVER_TWELVE*B[m]*B[n]*(y_var+9*y_bar*y_bar);
					   I5[m][n]=ONE_OVER_TWELVE*D[m]*D[n]*(x_var+9*x_bar*x_bar);
       }
     }


 
					k=1/(16*Ar*Ar*Ar);
     for (m=0; m<3; m++) {
       for (n=0; n<3; n++) {
         sumI=I1[m][n]+I2[m][n]+I3[m][n]+I4[m][n]+I5[m][n];
         S_tt[m][n]=k*L[m]*L[n]*(D[m]*D[n]+sumI)/(mu);
         T_tt[m][n]=k*epsilon*L[m]*L[n]*sumI;
       }
     }

     k=beta*beta/(mu*8*Ar*Ar);
     for (m=0; m<3; m++) {
        for (n=0; n<3; n++) {
           S_tz[m][n]=k*L[m]*(b[n]*(A[m]+B[m]*y_bar)+c[n]*(C[m]+D[m]*x_bar));
           S_zt[n][m]=k*L[m]*(b[n]*(A[m]+B[m]*y_bar)+c[n]*(C[m]+D[m]*x_bar));
        }
     }
    
    k=beta*beta/(mu*4*Ar);
    for (m=0; m<3; m++) {
      for (n=0; n<3; n++) {
        S_zz[m][n]=k*(b[m]*b[n]+c[m]*c[n]);
      }
    }

    k=beta*beta*epsilon*Ar*ONE_OVER_TWELVE;
    for (m=0; m<3; m++) {
      for (n=0; n<3; n++) {
        T_zz[m][n]=k;
      }
    }
    T_zz[0][0]=T_zz[1][1]=T_zz[2][2]=2*k; /* 1/6 */

     /* hashes for the edges */
				e[0]=get_edge_number(t->p0,t->p1,H);
				e[1]=get_edge_number(t->p1,t->p2,H);
				e[2]=get_edge_number(t->p2,t->p0,H);
   
/*
   || Stt   Stz || || Ttt    0||
   || Szt   Szz || ||  0   Tzz||
         S             T
   Sx = lambda Tx 
*/ 
   /* consistencay with sign change */
   mult[0]=mult[1]=mult[2]=1;
   if (t->p0>t->p1) {
     mult[0]=-1;
   }
   if (t->p1>t->p2) {
    mult[1]=-1;
   }
   if (t->p2>t->p0) {
    mult[2]=-1;
   }
   /* signs changes 
      Stt, and Ttt - both rows and columns
      Stz  rows 
      Szt  columns 
   */
   for (i=0; i<3; i++) {
      for (j=0; j<3; j++) {
         if ( mult[i]*mult[j] < 0 ) {
            S_tt[i][j]=-S_tt[i][j]; T_tt[i][j]=-T_tt[i][j];
         }
      }
      if ( mult[i] < 0 ) {
        for (j=0; j<3; j++) {
          S_tz[i][j]=-S_tz[i][j];
          S_zt[j][i]=-S_zt[j][i];
        }
     }
  }

		/* assemble global matrix */
		/* if any of the edges are Dirichlet, we do not include them in the 
		global matrix */
#ifdef DEBUG
    printf("build_equations: triangle %d, (%d,%d,%d)->(%d,%d,%d)/%d, edges (%d,%d,%d)\n",t->n,t->p0,t->p1,t->p2,renumber[t->p0],renumber[t->p1],renumber[t->p2], Nedges,e[0],e[1],e[2]);
    for (i=0;i<3;i++) {
     printf("||");
      printf(MDF" "MDF" "MDF" "MDF" "MDF" "MDF" ",S_tt[i][0],S_tt[i][1],S_tt[i][2],S_tz[i][0],S_tz[i][1],S_tz[i][2]);
     printf("|| ||");
      printf(MDF" "MDF" "MDF" "MDF" "MDF" "MDF" ",T_tt[i][0],T_tt[i][1],T_tt[i][2],0.0,0.0,0.0);
     printf("||\n");
    }
    for (i=0;i<3;i++) {
     printf("||");
      printf(MDF" "MDF" "MDF" "MDF" "MDF" "MDF" ",S_zt[i][0],S_zt[i][1],S_zt[i][2],S_zz[i][0],S_zz[i][1],S_zz[i][2]);
     printf("|| ||");
      printf(MDF" "MDF" "MDF" "MDF" "MDF" "MDF" ",0.0,0.0,0.0,T_zz[i][0],T_zz[i][1],T_zz[i][2]);
     printf("||\n");
    }
#endif

  /* assembling global matrix */
					for (i=0; i<3; i++) {
					 if ( e[i] != -1 ) {
					    for (j=0;j<3;j++) {
							 if (e[j] != -1 ) {
							   T[e[i]][e[j]]+=(T_tt[i][j]);
							   S[e[i]][e[j]]+=(S_tt[i][j]);
							 }
						 }
					}
				}
        i=renumber[t->p0];
        if (i>=Nedges) { /* fixed points have i == 0 */
          for (j=0; j<3; j++) {
           if ( e[j]!=-1) { S[i][e[j]]+=S_zt[0][j]; }
           if ( e[j]!=-1) { S[e[j]][i]+=S_tz[j][0]; }
          }
        }
        i=renumber[t->p1];
        if (i>=Nedges) { /* fixed points have i == 0 */
          for (j=0; j<3; j++) {
           if ( e[j]!=-1) { S[i][e[j]]+=S_zt[1][j]; }
           if ( e[j]!=-1) { S[e[j]][i]+=S_tz[j][1]; }
          }
        }
        i=renumber[t->p2];
        if (i>=Nedges) { /* fixed points have i == 0 */
          for (j=0; j<3; j++) {
           if ( e[j]!=-1) { S[i][e[j]]+=S_zt[2][j]; }
           if ( e[j]!=-1) { S[e[j]][i]+=S_tz[j][2]; }
          }
        }
     /* user e array to store point numbers temporarily */
      e[0]=renumber[t->p0];e[1]=renumber[t->p1];e[2]=renumber[t->p2];
      for (i=0; i<3; i++) {
       if ( e[i] >=Nedges ) {
        for (j=0;j<3; j++) {
         if ( e[j] >=Nedges ) {
             S[e[i]][e[j]]+=S_zz[i][j];
             T[e[i]][e[j]]+=T_zz[i][j];
         }
        }
       }
     }

	  }

   t=DAG_traverse_prune_list(&dt);
  }
}

/* solve for propagation constant given frequency */
static void
build_equations_from_hash_table_given_freq(htbl H, MY_DOUBLE  **S, MY_DOUBLE **T,
  MY_INT *renumber, MY_INT Ntotal, MY_INT Nedges, MY_DOUBLE k0)
{
 triangle *t;
 MY_INT i,j,m,n;
 MY_DOUBLE x[3],y[3],a[3],b[3],c[3],x_bar,y_bar,x_var,y_var;
 MY_DOUBLE A[3],B[3],C[3],D[3],L[3],Ar,k;
 MY_DOUBLE S_tt[3][3],T_tz[3][3],T_zt[3][3],T_zz[3][3],T_tt[3][3],S_zz[3][3];
 MY_DOUBLE I1[3][3],I2[3][3],I3[3][3],I4[3][3],I5[3][3];
 MY_DOUBLE epsilon,sumI,mu;
 MY_INT e[3],mult[3];


 DAG_traverse_list_reset(&dt);
 t=DAG_traverse_prune_list(&dt);
 while(t) {
  if (t && t->status==LIVE) {
        epsilon = bound.poly_array[t->boundary].epsilon;
        mu= bound.poly_array[t->boundary].mu;
		   /* get three edges */
       /*               p0
			                  /\
					             /  \
			     edge 1     /    \  edge 3
                     /      \
					          /________\
                   p1 edge 2  p2
					*/

					/* get coords */
					/* convert to global coords */
					x[0]=Mx(t->p0)*g_xscale-g_xoff;
					y[0]=My(t->p0)*g_yscale-g_yoff;
					x[1]=Mx(t->p1)*g_xscale-g_xoff;
					y[1]=My(t->p1)*g_yscale-g_yoff;
					x[2]=Mx(t->p2)*g_xscale-g_xoff;
					y[2]=My(t->p2)*g_yscale-g_yoff;
 
					x_bar=(x[1]+x[2]+x[0])*ONE_OVER_THREE;
					y_bar=(y[1]+y[2]+y[0])*ONE_OVER_THREE;
					x_var=(x[1]*x[1]+x[2]*x[2]+x[0]*x[0]);
					y_var=(y[1]*y[1]+y[2]*y[2]+y[0]*y[0]);

					a[0]=x[1]*y[2]-x[2]*y[1];
					a[1]=x[2]*y[0]-x[0]*y[2];
					a[2]=x[0]*y[1]-x[1]*y[0];
					b[0]=y[1]-y[2];
					b[1]=y[2]-y[0];
					b[2]=y[0]-y[1];
					c[0]=x[2]-x[1];
					c[1]=x[0]-x[2];
					c[2]=x[1]-x[0];

     A[0]=a[0]*b[1]-a[1]*b[0];
     A[1]=a[1]*b[2]-a[2]*b[1];
     A[2]=a[2]*b[0]-a[0]*b[2];
					B[0]=c[0]*b[1]-c[1]*b[0];
					B[1]=c[1]*b[2]-c[2]*b[1];
					B[2]=c[2]*b[0]-c[0]*b[2];
					C[0]=a[0]*c[1]-a[1]*c[0];
					C[1]=a[1]*c[2]-a[2]*c[1];
					C[2]=a[2]*c[0]-a[0]*c[2];
					D[0]=-B[0];
					D[1]=-B[1];
					D[2]=-B[2];
     /* edge lengths */
					L[0]=sqrt(ABS(c[2]*c[2]+b[2]*b[2]));
					L[1]=sqrt(ABS(c[0]*c[0]+b[0]*b[0]));
					L[2]=sqrt(ABS(c[1]*c[1]+b[1]*b[1]));
				 /* area */
					Ar=0.5*(a[1]+a[2]+a[0]);

     for (m=0; m<3; m++) {
       for (n=0; n<3; n++) {
					   I1[m][n]=A[m]*A[n]+C[m]*C[n];
					   I2[m][n]=(C[m]*D[n]+C[n]*D[m])*x_bar;
					   I3[m][n]=(A[m]*B[n]+A[n]*B[m])*y_bar;
					   I4[m][n]=ONE_OVER_TWELVE*B[m]*B[n]*(y_var+9*y_bar*y_bar);
					   I5[m][n]=ONE_OVER_TWELVE*D[m]*D[n]*(x_var+9*x_bar*x_bar);
       }
     }


 
					k=1/(16*Ar*Ar*Ar);
     for (m=0; m<3; m++) {
       for (n=0; n<3; n++) {
         sumI=I1[m][n]+I2[m][n]+I3[m][n]+I4[m][n]+I5[m][n];
         S_tt[m][n]=k*L[m]*L[n]*(D[m]*D[n]-k0*k0*epsilon*sumI)/(mu);
         T_tt[m][n]=k*epsilon*L[m]*L[n]*sumI;
       }
     }

     k=1/(mu*8*Ar*Ar);
     for (m=0; m<3; m++) {
        for (n=0; n<3; n++) {
           T_tz[m][n]=k*L[m]*(b[n]*(A[m]+B[m]*y_bar)+c[n]*(C[m]+D[m]*x_bar));
           T_zt[n][m]=k*L[m]*(b[n]*(A[m]+B[m]*y_bar)+c[n]*(C[m]+D[m]*x_bar));
        }
     }
    
    k=1/(mu*4*Ar);
    for (m=0; m<3; m++) {
      for (n=0; n<3; n++) {
        S_zz[m][n]=k*(b[m]*b[n]+c[m]*c[n]);
      }
    }

    k=k0*k0*epsilon*Ar*ONE_OVER_TWELVE;
    for (m=0; m<3; m++) {
      for (n=0; n<3; n++) {
        T_zz[m][n]=k;
      }
    }
    T_zz[0][0]=T_zz[1][1]=T_zz[2][2]=2*k; /* 1/6 */
    for (m=0; m<3; m++) {
      for (n=0; n<3; n++) {
        T_zz[m][n]+=S_zz[m][n];
      }
    }




     /* hashes for the edges */
					e[0]=get_edge_number(t->p0,t->p1,H);
					e[1]=get_edge_number(t->p1,t->p2,H);
					e[2]=get_edge_number(t->p2,t->p0,H);
    
   /* consistencay with sign change */
   mult[0]=mult[1]=mult[2]=1;
   if (t->p0>t->p1) {
     mult[0]=-1;
   }
   if (t->p1>t->p2) {
    mult[1]=-1;
   }
   if (t->p2>t->p0) {
    mult[2]=-1;
   }
   for (i=0; i<3; i++) {
      for (j=0; j<3; j++) {
         if ( mult[i]*mult[j] < 0 ) {
            S_tt[i][j]=-S_tt[i][j]; T_tt[i][j]=-T_tt[i][j];
         }
      }
      if ( mult[i] < 0 ) {
        for (j=0; j<3; j++) {
          T_tz[i][j]=-T_tz[i][j];
          T_zt[j][i]=-T_zt[j][i];
        }
     }
  }
					/* assemble global matrix */
					/* if any of the edges are Dirichlet, we do not include them in the 
					   global matrix */
#ifdef DEBUG
    printf("build_equations: triangle %d, (%d,%d,%d)->(%d,%d,%d)/%d, edges (%d,%d,%d)\n",t->n,t->p0,t->p1,t->p2,renumber[t->p0],renumber[t->p1],renumber[t->p2], Nedges,e[0],e[1],e[2]);
    for (i=0;i<3;i++) {
     printf("||");
      printf(MDF" "MDF" "MDF" "MDF" "MDF" "MDF" ",S_tt[i][0],S_tt[i][1],S_tt[i][2],0.0,0.0,0.0);
     printf("|| ||");
      printf(MDF" "MDF" "MDF" "MDF" "MDF" "MDF" ",T_tt[i][0],T_tt[i][1],T_tt[i][2],T_tz[i][2],T_tz[i][2],T_tz[i][2]);
     printf("||\n");
    }
    for (i=0;i<3;i++) {
     printf("||");
      printf(MDF" "MDF" "MDF" "MDF" "MDF" "MDF" ",0.0,0.0,0.0,0.0,0.0,0.0);
     printf("|| ||");
      printf(MDF" "MDF" "MDF" "MDF" "MDF" "MDF" ",T_zt[i][0],T_zt[i][1],T_zt[i][2],T_zz[i][2],T_zz[i][2],T_zz[i][2]);
     printf("||\n");
    }
#endif

  /* assembling global matrix */
					for (i=0; i<3; i++) {
					 if ( e[i] != -1 ) {
					    for (j=0;j<3;j++) {
							 if (e[j] != -1 ) {
							   S[e[i]][e[j]]+=(S_tt[i][j]);
							   T[e[i]][e[j]]+=(T_tt[i][j]);
							 }
						 }
					}
				}
        i=renumber[t->p0];
        if (i>=Nedges) { /* fixed points have i == 0 */
          for (j=0; j<3; j++) {
           if ( e[j]!=-1) { T[i][e[j]]+=T_tz[0][j]; }
           if ( e[j]!=-1) { T[e[j]][i]+=T_zt[j][0]; }
          }
        }
        i=renumber[t->p1];
        if (i>=Nedges) { /* fixed points have i == 0 */
          for (j=0; j<3; j++) {
           if ( e[j]!=-1) { T[i][e[j]]+=T_tz[1][j]; }
           if ( e[j]!=-1) { T[e[j]][i]+=T_zt[j][1]; }
          }
        }
        i=renumber[t->p2];
        if (i>=Nedges) { /* fixed points have i == 0 */
          for (j=0; j<3; j++) {
           if ( e[j]!=-1) { T[i][e[j]]+=T_tz[2][j]; }
           if ( e[j]!=-1) { T[e[j]][i]+=T_zt[j][2]; }
          }
        }
     /* user e array to store point numbers temporarily */
      e[0]=renumber[t->p0];e[1]=renumber[t->p1];e[2]=renumber[t->p2];
      for (i=0; i<3; i++) {
       if ( e[i] >=Nedges ) {
        for (j=0;j<3; j++) {
         if ( e[j] >=Nedges ) {
             T[e[i]][e[j]]+=T_zz[i][j];
         }
        }
       }
     }


   }

   t=DAG_traverse_prune_list(&dt);
  }
}


/* solve for cutoff frequency */
static void
build_equations_from_hash_table(htbl H, MY_DOUBLE  **S, MY_DOUBLE **T,
  MY_INT *renumber, MY_INT Ntotal, MY_INT Nedges)
{
 /* 
  S, T - Ntotal by Ntotal matrices
  Nedges - number of edges, the rest points 
  renumber - map points to matrix location
  edges are mapper using hash table H
*/
 triangle *t;
 MY_INT i,j,m,n;
 MY_DOUBLE x[3],y[3],a[3],b[3],c[3],x_bar,y_bar,x_var,y_var;
 MY_DOUBLE A[3],B[3],C[3],D[3],L[3],Ar,k;
 MY_DOUBLE S_tt[3][3],S_zz[3][3],T_tt[3][3],T_zz[3][3];
 MY_DOUBLE I1[3][3],I2[3][3],I3[3][3],I4[3][3],I5[3][3];
 MY_DOUBLE epsilon,sumI,mu;
 MY_INT e[3],mult[3];

 DAG_traverse_list_reset(&dt);
 t=DAG_traverse_prune_list(&dt);
 while(t) {
  if (t && t->status==LIVE) {
        epsilon= bound.poly_array[t->boundary].epsilon;
        mu= bound.poly_array[t->boundary].mu;
		   /* get three edges */
      /*              p0
			                  /\
					 	  				 /  \
				    edge 1    /    \  edge 3
                     /      \
					          /________\
                  p1 edge 2  p2
	   */

					/* get coords */
					/* convert to global coords */
					x[0]=Mx(t->p0)*g_xscale-g_xoff;
					y[0]=My(t->p0)*g_yscale-g_yoff;
					x[1]=Mx(t->p1)*g_xscale-g_xoff;
					y[1]=My(t->p1)*g_yscale-g_yoff;
					x[2]=Mx(t->p2)*g_xscale-g_xoff;
					y[2]=My(t->p2)*g_yscale-g_yoff;
 
					x_bar=(x[1]+x[2]+x[0])*ONE_OVER_THREE;
					y_bar=(y[1]+y[2]+y[0])*ONE_OVER_THREE;
					x_var=(x[1]*x[1]+x[2]*x[2]+x[0]*x[0]);
					y_var=(y[1]*y[1]+y[2]*y[2]+y[0]*y[0]);

					a[0]=x[1]*y[2]-x[2]*y[1];
					a[1]=x[2]*y[0]-x[0]*y[2];
					a[2]=x[0]*y[1]-x[1]*y[0];
					b[0]=y[1]-y[2];
					b[1]=y[2]-y[0];
					b[2]=y[0]-y[1];
					c[0]=x[2]-x[1];
					c[1]=x[0]-x[2];
					c[2]=x[1]-x[0];

     A[0]=a[0]*b[1]-a[1]*b[0];
     A[1]=a[1]*b[2]-a[2]*b[1];
     A[2]=a[2]*b[0]-a[0]*b[2];
					B[0]=c[0]*b[1]-c[1]*b[0];
					B[1]=c[1]*b[2]-c[2]*b[1];
					B[2]=c[2]*b[0]-c[0]*b[2];
					C[0]=a[0]*c[1]-a[1]*c[0];
					C[1]=a[1]*c[2]-a[2]*c[1];
					C[2]=a[2]*c[0]-a[0]*c[2];
					D[0]=-B[0];
					D[1]=-B[1];
					D[2]=-B[2];
     /* edge lengths */
					L[0]=sqrt(ABS(c[2]*c[2]+b[2]*b[2]));
					L[1]=sqrt(ABS(c[0]*c[0]+b[0]*b[0]));
					L[2]=sqrt(ABS(c[1]*c[1]+b[1]*b[1]));
				 /* area */
					Ar=0.5*(a[1]+a[2]+a[0]);

     for (m=0; m<3; m++) {
       for (n=0; n<3; n++) {
					   I1[m][n]=A[m]*A[n]+C[m]*C[n];
					   I2[m][n]=(C[m]*D[n]+C[n]*D[m])*x_bar;
					   I3[m][n]=(A[m]*B[n]+A[n]*B[m])*y_bar;
					   I4[m][n]=ONE_OVER_TWELVE*B[m]*B[n]*(y_var+9*y_bar*y_bar);
					   I5[m][n]=ONE_OVER_TWELVE*D[m]*D[n]*(x_var+9*x_bar*x_bar);
       }
     }


 
					k=1/(16*Ar*Ar*Ar);
     for (m=0; m<3; m++) {
       for (n=0; n<3; n++) {
         sumI=I1[m][n]+I2[m][n]+I3[m][n]+I4[m][n]+I5[m][n];
         S_tt[m][n]=4*k*L[m]*L[n]*(D[m]*D[n])/(mu);
         T_tt[m][n]=k*epsilon*L[m]*L[n]*sumI;
       }
     }
    
    k=1/(mu*4*Ar);
    for (m=0; m<3; m++) {
      for (n=0; n<3; n++) {
        S_zz[m][n]=k*(b[m]*b[n]+c[m]*c[n]);
      }
    }

    k=epsilon*Ar*ONE_OVER_TWELVE;
    for (m=0; m<3; m++) {
      for (n=0; n<3; n++) {
        T_zz[m][n]=k;
      }
    }
    T_zz[0][0]=T_zz[1][1]=T_zz[2][2]=2*k; /* 1/6 */

     /* hashes for the edges */
				e[0]=get_edge_number(t->p0,t->p1,H);
				e[1]=get_edge_number(t->p1,t->p2,H);
				e[2]=get_edge_number(t->p2,t->p0,H);
   
/*
   || Stt   0   || || Ttt    0||
   || 0     Szz || ||  0   Tzz||
         S             T
   Sx = lambda Tx 
*/ 
   /* consistencay with sign change */
   mult[0]=mult[1]=mult[2]=1;
   if (t->p0>t->p1) {
     mult[0]=-1;
   }
   if (t->p1>t->p2) {
    mult[1]=-1;
   }
   if (t->p2>t->p0) {
    mult[2]=-1;
   }
   /* signs changes 
      Stt, and Ttt - both rows and columns
      Stz  rows 
      Szt  columns 
   */
   for (i=0; i<3; i++) {
      for (j=0; j<3; j++) {
         if ( mult[i]*mult[j] < 0 ) {
            S_tt[i][j]=-S_tt[i][j]; T_tt[i][j]=-T_tt[i][j];
         }
      }
  }

		/* assemble global matrix */
		/* if any of the edges are Dirichlet, we do not include them in the 
		global matrix */
#ifdef DEBUG
    printf("build_equations: triangle %d, (%d,%d,%d)->(%d,%d,%d)/%d, edges (%d,%d,%d)\n",t->n,t->p0,t->p1,t->p2,renumber[t->p0],renumber[t->p1],renumber[t->p2], Nedges,e[0],e[1],e[2]);
    for (i=0;i<3;i++) {
     printf("||");
      printf(MDF" "MDF" "MDF" "MDF" "MDF" "MDF" ",S_tt[i][0],S_tt[i][1],S_tt[i][2],0.0,0.0,0.0);
     printf("|| ||");
      printf(MDF" "MDF" "MDF" "MDF" "MDF" "MDF" ",T_tt[i][0],T_tt[i][1],T_tt[i][2],0.0,0.0,0.0);
     printf("||\n");
    }
    for (i=0;i<3;i++) {
     printf("||");
      printf(MDF" "MDF" "MDF" "MDF" "MDF" "MDF" ",0.0,0.0,0.0,S_zz[i][0],S_zz[i][1],S_zz[i][2]);
     printf("|| ||");
      printf(MDF" "MDF" "MDF" "MDF" "MDF" "MDF" ",0.0,0.0,0.0,T_zz[i][0],T_zz[i][1],T_zz[i][2]);
     printf("||\n");
    }
#endif

  /* assembling global matrix */
					for (i=0; i<3; i++) {
					 if ( e[i] != -1 ) {
					    for (j=0;j<3;j++) {
							 if (e[j] != -1 ) {
							   T[e[i]][e[j]]+=(T_tt[i][j]);
							   S[e[i]][e[j]]+=(S_tt[i][j]);
							 }
						 }
					}
				}

     /* user e array to store point numbers temporarily */
      e[0]=renumber[t->p0];e[1]=renumber[t->p1];e[2]=renumber[t->p2];
      for (i=0; i<3; i++) {
       if ( e[i] >=Nedges ) {
        for (j=0;j<3; j++) {
         if ( e[j] >=Nedges ) {
             S[e[i]][e[j]]+=S_zz[i][j];
             T[e[i]][e[j]]+=T_zz[i][j];
         }
        }
       }
     }

	  }

   t=DAG_traverse_prune_list(&dt);
  }
}

int 
buildup_hash_table_and_equation(void)
{

 triangle *t;
 htbl H;
 hash_edge *ee;
#ifdef DEBUG
 hash_edge *gg;
#endif
 MY_INT i,j,Ni,temp;
 MY_DOUBLE **S, **T;
 MY_INT *r_hash=0;
 MY_DOUBLE **v;
 MY_INT *renumber, *re_renumber;
 MY_DOUBLE *max_value;
 MY_INT NN;

 MY_DOUBLE x[3],y[3],a[3],b[3],c[3],et[2],Ar;
 MY_DOUBLE A[3],B[3],C[3],D[3],L[3];
 MY_INT e1,e2,e3;

 /* make degree of freedom=1 */
 degree_of_freedom=requested_degree_of_freedom;
 /* find  size needed for hash table */ 
 i=0;
 DAG_traverse_list_reset(&dt);
 t=DAG_traverse_prune_list(&dt);
 while (t) {
  if ( t && t->status==LIVE ) {
   i++;
  }
 t=DAG_traverse_prune_list(&dt);
 }
#ifdef DEBUG
printf("buildup_hash: creating table size %d\n",i);
#endif
 if (i<300) { /* small mesh, more edges */
  htbl_init(&H, 6*i, &h1, &h2, &match, &destroy);
 } else {
  htbl_init(&H, 2*i, &h1, &h2, &match, &destroy);
 }

 /* create reverse mapping array for edges */
 if (i<100) { /* small mesh, more edges */
  if((r_hash= (MY_INT*) calloc(6*i, sizeof(MY_INT)))==0){
   fprintf(stderr, "%s: %d: no free memory\n", __FILE__,__LINE__);
   exit(1);
  }
 } else {
 if((r_hash= (MY_INT*) calloc(2*i, sizeof(MY_INT)))==0){
   fprintf(stderr, "%s: %d: no free memory\n", __FILE__,__LINE__);
   exit(1);
  }
 }

/* create renumbering array for points */
/* keep point numbers also in edge number array. 
 we first go through the edges and number them.
 next we run another triangle loop to number the points.
 We alwasy remember last edge number. So everthyng after
 that is a point. */
/* example if e is an edge, the location of matrix is given 
 be r_hash[e] also we know e <= Ni : Ni is max edge number
 if p is a point, location in matrix is given by
  r_hash[Ni+p] */
 i=0; 
 /* insert data */
 DAG_traverse_list_reset(&dt);
 t=DAG_traverse_prune_list(&dt);
 while (t) {
  if ( t && t->status==LIVE ) {
  if ( (ee=(hash_edge*)malloc(sizeof(hash_edge)))==0) {
   fprintf(stderr,"%s: %d: no free memory\n",__FILE__,__LINE__);
	 exit(1);
	}
	ee->p0=t->p0;
	ee->p1=t->p1;
  /* always p0 <= p1 */
	if ( ee->p0 > ee->p1 ) {
			temp=ee->p0;
			ee->p0=ee->p1;
			ee->p1=temp;
	}
#ifdef DEBUG
		printf("**build: tring to insert (%d,%d) into table\n",ee->p0,ee->p1);
#endif

  if ( (!htbl_lookup(&H,(void *)ee)) &&(!htbl_insert(&H, (const void *)ee)) ) {
#ifdef DEBUG
		printf("sucessfully inserted (%d,%d) into table\n",ee->p0,ee->p1);
#endif
    /* check whether this edge is on a Dirichlet boundary */
    if ( !is_this_edge_on_a_dirichlet_boundary(ee->p0,ee->p1) ) {
      /* give a unique contiguous number */
      ee->n=i;
      r_hash[i++]=htbl_lookup_hash(&H, (void *)ee);
    } else {
#ifdef DEBUG
      printf("buildup_hash: edge on a Dirichlet boundary\n");
#endif
      ee->n=-1;
    }
    
  } else {
#ifdef DEBUG
		printf("failed inserting (%d,%d) into table ",ee->p0,ee->p1);
		/* lookup */
   if ( (gg=(hash_edge*)htbl_lookup(&H, (void *)ee)) ) {
		printf("it exists as %d(%d,%d)->%d.\n", htbl_lookup_hash(&H, (void *)ee),gg->p0,gg->p1,gg->n);
   }
#endif
		free(ee);
  }
  if ( (ee=(hash_edge*)malloc(sizeof(hash_edge)))==0) {
   fprintf(stderr,"%s: %d: no free memory\n",__FILE__,__LINE__);
	 exit(1);
	}
  ee->p0=t->p1;
	ee->p1=t->p2;
  /* always p0 <= p1 */
	if ( ee->p0 > ee->p1 ) {
			temp=ee->p0;
			ee->p0=ee->p1;
			ee->p1=temp;
	}
#ifdef DEBUG
		printf("**build: tring to insert (%d,%d) into table\n",ee->p0,ee->p1);
#endif

  if ( (!htbl_lookup(&H,(void *)ee)) && (!htbl_insert(&H, (const void *)ee)) ) {
#ifdef DEBUG
		printf("sucessfully inserted (%d,%d) into table\n",ee->p0,ee->p1);
#endif
    if ( !is_this_edge_on_a_dirichlet_boundary(ee->p0,ee->p1) ) {
   /* give a unique contiguous number */
      ee->n=i;
      r_hash[i++]=htbl_lookup_hash(&H, (void *)ee);
    } else {
#ifdef DEBUG
      printf("buildup_hash: edge on a Dirichlet boundary\n");
#endif
      ee->n=-1;
    }
  } else {
#ifdef DEBUG
		printf("failed inserting (%d,%d) into table ",ee->p0,ee->p1);
		/* lookup */
   if ( (gg=(hash_edge*)htbl_lookup(&H, (void *)ee)) ) {
		printf("it exists as %d(%d,%d)->%d.\n", htbl_lookup_hash(&H, (void *)ee),gg->p0,gg->p1,gg->n);
   }
#endif
		free(ee);
  }
  if ( (ee=(hash_edge*)malloc(sizeof(hash_edge)))==0) {
   fprintf(stderr,"%s: %d: no free memory\n",__FILE__,__LINE__);
	 exit(1);
	}
  ee->p0=t->p2;
	ee->p1=t->p0;
  /* always p0 <= p1 */
	if ( ee->p0 > ee->p1 ) {
			temp=ee->p0;
			ee->p0=ee->p1;
			ee->p1=temp;
	}
#ifdef DEBUG
		printf("**build: tring to insert (%d,%d) into table\n",ee->p0,ee->p1);
#endif

  if ( (!htbl_lookup(&H,(void *)ee)) && (!htbl_insert(&H, (const void *)ee)) ) {
#ifdef DEBUG
		printf("sucessfully inserted (%d,%d) into table\n",ee->p0,ee->p1);
#endif
    if ( !is_this_edge_on_a_dirichlet_boundary(ee->p0,ee->p1) ) {
    /* give a unique contiguous number */
      ee->n=i;
      r_hash[i++]=htbl_lookup_hash(&H, (void *)ee);
    } else {
#ifdef DEBUG
      printf("buildup_hash: edge on a Dirichlet boundary\n");
#endif
      ee->n=-1;
    }

  } else {
#ifdef DEBUG
		printf("failed inserting (%d,%d) into table ",ee->p0,ee->p1);
		/* lookup */
   if ( (gg=(hash_edge*)htbl_lookup(&H, (void *)ee)) ) {
		printf("it exists as %d(%d,%d)->%d.\n", htbl_lookup_hash(&H, (void *)ee),gg->p0,gg->p1,gg->n);
   }
#endif
		free(ee);
  }
 }

 t=DAG_traverse_prune_list(&dt);
 }
#ifdef DEBUG
 printf("buildup_hash %d of %d edges, hash table filled %lf\n",i,H.positions,H.size/(double)H.positions);
#endif
 /* remember no of edges */
 Ni=i;
 /* resize reverse mapping array */
 if((r_hash= (MY_INT*) realloc((void*)r_hash,(size_t)Ni*sizeof(MY_INT)))==0){
   fprintf(stderr, "%s: %d: no free memory\n", __FILE__,__LINE__);
   exit(1);
 }

/******************/
 /* setup point to matrix mapping array */
 if ((renumber = (MY_INT *)calloc(M.count,sizeof(MY_INT )))==0){
   fprintf(stderr, "%s: %d: no free memory\n", __FILE__,__LINE__);
   exit(1);
 }
 if((re_renumber = (MY_INT *)calloc(M.count,sizeof(MY_INT)))==0){
   fprintf(stderr, "%s: %d: no free memory\n", __FILE__,__LINE__);
   exit(1);
 }

  /* need to solve for only points referred by triangles */
  /* SO -- */
  DAG_traverse_list_reset(&dt);
  t=DAG_traverse_prune_list(&dt);
  while ( t ) {
    if ( t && t->status==LIVE ) {
      /* mark the points in this triangle */
      /* use re_renumber[] array for this */
      re_renumber[t->p0] = 1;
      re_renumber[t->p1] = 1;
      re_renumber[t->p2] = 1;

    }
    t=DAG_traverse_prune_list(&dt);
  }

  /* now give numbers to variable points*/
  /* start numbering from last edge */
  j = Ni;
  for ( i = 0; i < M.count; i++ )
    if (  Mval(i) == VAR  && re_renumber[i] == 1) {
      renumber[i] = j++;
#ifdef DEBUG
            printf("buildup_hash: renumbering variable point %d as %d\n",i,renumber[i]);
#endif
    }
  free(re_renumber);
/******************/
 /* update total size of matrix */
NN=j; 
/* Ni edges and NN-Ni points */
#ifdef DEBUG
  for (i=0; i< Ni; i++) {
   printf("buildup_hash: [%d]->%d ",i,r_hash[i]);
  if ( (r_hash[i]>=0) && (r_hash[i]<=H.positions) ) {
  gg=(hash_edge*)H.table[r_hash[i]];
  printf(" edge (%d,%d)\n",gg->p0,gg->p1);
  } else {
  printf(" no valid edge\n");
  }
  }
#endif
 /* set up matrices */
 if((S= (MY_DOUBLE**) calloc(NN, sizeof(MY_DOUBLE*)))==0){
   fprintf(stderr, "%s: %d: no free memory\n", __FILE__,__LINE__);
   exit(1);
 }
 if((T= (MY_DOUBLE**) calloc(NN, sizeof(MY_DOUBLE*)))==0){
   fprintf(stderr, "%s: %d: no free memory\n", __FILE__,__LINE__);
   exit(1);
 }
	/* STEP 2: SET UP THE COLUMNS. */
	for (j = 0; j < NN; ++j) {
			  /* full length columns - matrix not symmetric  */
	 if((S[j] = ( MY_DOUBLE * ) calloc( (size_t)NN, sizeof(MY_DOUBLE)))==0){
    fprintf(stderr, "%s: %d: no free memory\n", __FILE__,__LINE__);
    exit(1);
   }
   if((T[j] = ( MY_DOUBLE * ) calloc( (size_t)NN, sizeof(MY_DOUBLE)))==0){
    fprintf(stderr, "%s: %d: no free memory\n", __FILE__,__LINE__);
    exit(1);
   }
 }


 /* depending on equation type, we formulate a different problem */ 
  if (solve_equation == HELMHOLTZ_INHOMO ) {
   build_equations_from_hash_table(H, S, T, renumber, NN, Ni);
  } else if (solve_equation == HELMHOLTZ_FREQ) { 
   build_equations_from_hash_table_given_freq(H, S, T, renumber, NN, Ni, global_wave_k0); 
  } else if (solve_equation == HELMHOLTZ_BETA) {
   build_equations_from_hash_table_given_beta(H, S, T, renumber, NN, Ni, global_wave_beta); 
  } 

#ifdef DEBUG
 printf("====================\n");
 printf("S=\n");
 for (i=0;i<NN;i++) {
   printf("|");
   for (j=0;j<NN;j++) {
    if(S[i][j]) {printf(" "MDF,S[i][j]);} else {printf(" 0");}
   }
   printf("|\n");
 }
 printf("====================\n");
 printf("T=\n");
 for (i=0;i<NN;i++) {
   printf("|");
   for (j=0;j<NN;j++) {
    if(T[i][j]) {printf(" "MDF,T[i][j]);} else {printf(" 0");}
   }
   printf("|\n");
 }
#endif

	/* structure to store all eigenvectors - matrix */
  if((v=(MY_DOUBLE **)calloc(NN, sizeof(MY_DOUBLE*)))==0){
   fprintf(stderr, "%s: %d: no free memory\n", __FILE__,__LINE__);
   exit(1);
 }
 for (i = 0; i <NN; i++)
	  if((v[i] = ( MY_DOUBLE * ) calloc( (size_t) degree_of_freedom, sizeof(MY_DOUBLE)))==0){
   fprintf(stderr, "%s: %d: no free memory\n", __FILE__,__LINE__);
   exit(1);
 }

 /* now solve eigenvalue problem (size Ni by Ni ) 
   S. x=k^2 T.x 
  or (S-lambda T)x= 0 
  call 
  solve_helmholtz(MY_DOUBLE **P,MY_DOUBLE **T, MY_DOUBLE **x,  MY_INT  NUN, MY_INT n_to_find)
 */

#ifdef USE_LAPACK
 solve_generalized_eig_lapack(S,T,v,eig_array, NN,degree_of_freedom);
#endif

 /* reallocate memory for z array and copy all eigenvectors */
 /* we need 2 positions for x,y components per degree of freedom */
 /* and one more for the  z component */
 for ( i = 0; i < M.count; i++ ) {
  if (((M.Narray[i])->z=(MY_DOUBLE *)realloc((void*)(M.Narray[i])->z,(size_t)3*degree_of_freedom*sizeof(MY_DOUBLE)))==0) {
     fprintf(stderr,"%s: %d: no free memory",__FILE__,__LINE__);
   exit(2);
 }
 }

  /* array to remember max values of potentials */
  if ((max_value=(MY_DOUBLE *)malloc((size_t)degree_of_freedom*sizeof(MY_DOUBLE)))==0) {
     fprintf(stderr,"%s: %d: no free memory",__FILE__,__LINE__);
   exit(2);
 }
 for (i=0; i< degree_of_freedom; i++) {
    max_value[i]=-10000; /* low value */
 }

 /* update nodal values from the calculated edge values */
 DAG_traverse_list_reset(&dt);
 t=DAG_traverse_prune_list(&dt);
 while(t) {
  if (t && t->status==LIVE) {
					x[0]=Mx(t->p0)*g_xscale-g_xoff;
					y[0]=My(t->p0)*g_yscale-g_yoff;
					x[1]=Mx(t->p1)*g_xscale-g_xoff;
					y[1]=My(t->p1)*g_yscale-g_yoff;
					x[2]=Mx(t->p2)*g_xscale-g_xoff;
					y[2]=My(t->p2)*g_yscale-g_yoff;
 
					a[0]=x[1]*y[2]-x[2]*y[1];
					a[1]=x[2]*y[0]-x[0]*y[2];
					a[2]=x[0]*y[1]-x[1]*y[0];
					b[0]=y[1]-y[2];
					b[1]=y[2]-y[0];
					b[2]=y[0]-y[1];
					c[0]=x[2]-x[1];
					c[1]=x[0]-x[2];
					c[2]=x[1]-x[0];

     A[0]=a[0]*b[1]-a[1]*b[0];
     A[1]=a[1]*b[2]-a[2]*b[1];
     A[2]=a[2]*b[0]-a[0]*b[2];
					B[0]=c[0]*b[1]-c[1]*b[0];
					B[1]=c[1]*b[2]-c[2]*b[1];
					B[2]=c[2]*b[0]-c[0]*b[2];
					C[0]=a[0]*c[1]-a[1]*c[0];
					C[1]=a[1]*c[2]-a[2]*c[1];
					C[2]=a[2]*c[0]-a[0]*c[2];
					D[0]=-B[0];
					D[1]=-B[1];
					D[2]=-B[2];
     /* edge lengths */
					L[0]=sqrt(ABS(c[2]*c[2]+b[2]*b[2]));
					L[1]=sqrt(ABS(c[0]*c[0]+b[0]*b[0]));
					L[2]=sqrt(ABS(c[1]*c[1]+b[1]*b[1]));
				 /* area */
					Ar=0.5*(a[1]+a[2]+a[0]);

#ifdef DEBUG
     printf("build_hash: adding tria "MIF" X:("MDF","MDF","MDF")Y:("MDF","MDF","MDF") ",t->n,x[0],x[1],x[2],y[0],y[1],y[2]);
#endif
      /* normalize edges: make L -> L/(4A^2) */
      Ar=1/(4*Ar*Ar);
      L[0]*=Ar;
      L[1]*=Ar;
      L[2]*=Ar;
  
     /* hashes for the edges */
					e1=get_edge_number(t->p0,t->p1,H);
					e2=get_edge_number(t->p1,t->p2,H);
					e3=get_edge_number(t->p2,t->p0,H);
    
     for (i=0;i<degree_of_freedom;i++) {
     if (e1!=-1) {
      et[0]=v[e1][i];
#ifdef DEBUG
     printf("<=="MDF","MDF"==>",et[0],v[e1][i]);
#endif
     } else {
      et[0]=0.0;
     }
     if (e2!=-1) {
      et[1]=v[e2][i];
#ifdef DEBUG
     printf("<=="MDF","MDF"==>",et[1],v[e2][i]);
#endif

     } else {
      et[1]=0.0;
     }
     if (e3!=-1) {
      et[2]=v[e3][i];
#ifdef DEBUG
     printf("<=="MDF","MDF"==>",et[2],v[e3][i]);
#endif

     } else {
      et[2]=0.0;
     }

#ifdef DEBUG
     printf("edges %d(%d,%d), %d(%d,%d), %d(%d,%d),t1="MDF" t2="MDF" t3="MDF" ",e1,t->p0,t->p1,e2,t->p1,t->p2,e3,t->p2,t->p0,et[0],et[1],et[2]);
#endif
     Mzz(t->p0,3*i)=0.0;
     Mzz(t->p0,3*i+1)=0.0;
     Mzz(t->p1,3*i)=0.0;
     Mzz(t->p1,3*i+1)=0.0;
     Mzz(t->p2,3*i)=0.0;
     Mzz(t->p2,3*i+1)=0.0;
     for (j=0; j<3; j++) {
       Mzz(t->p0,3*i)+=et[j]*L[j]*(A[j]+B[j]*y[0]);
       Mzz(t->p0,3*i+1)+=et[j]*L[j]*(C[j]+D[j]*x[0]);
       Mzz(t->p1,3*i)+=et[j]*L[j]*(A[j]+B[j]*y[1]);
       Mzz(t->p1,3*i+1)+=et[j]*L[j]*(C[j]+D[j]*x[1]);
       Mzz(t->p2,3*i)+=et[j]*L[j]*(A[j]+B[j]*y[2]);
       Mzz(t->p2,3*i+1)+=et[j]*L[j]*(C[j]+D[j]*x[2]);
     }

     /* z component */
     Mzz(t->p0,3*i+2)=v[renumber[t->p0]][i];
     Mzz(t->p1,3*i+2)=v[renumber[t->p1]][i];
     Mzz(t->p2,3*i+2)=v[renumber[t->p2]][i];

     /* find min, max value for plotting */
     if ( ABS(Mzz(t->p0,3*i)) > max_value[i] ) max_value[i]=ABS(Mzz(t->p0,3*i));
     if ( ABS(Mzz(t->p0,3*i+1)) > max_value[i] ) max_value[i]=ABS(Mzz(t->p0,3*i+1));
     if ( ABS(Mzz(t->p0,3*i+2)) > max_value[i] ) max_value[i]=ABS(Mzz(t->p0,3*i+2));
     if ( ABS(Mzz(t->p1,3*i)) > max_value[i] ) max_value[i]=ABS(Mzz(t->p1,3*i));
     if ( ABS(Mzz(t->p1,3*i+1)) > max_value[i] ) max_value[i]=ABS(Mzz(t->p1,3*i+1));
     if ( ABS(Mzz(t->p1,3*i+2)) > max_value[i] ) max_value[i]=ABS(Mzz(t->p1,3*i+2));
     if ( ABS(Mzz(t->p2,3*i)) > max_value[i] ) max_value[i]=ABS(Mzz(t->p2,3*i));
     if ( ABS(Mzz(t->p2,3*i+1)) > max_value[i] ) max_value[i]=ABS(Mzz(t->p2,3*i+1));
     if ( ABS(Mzz(t->p2,3*i+2)) > max_value[i] ) max_value[i]=ABS(Mzz(t->p2,3*i+2));


    

#ifdef DEBUG
    printf("Z:("MDF","MDF","MDF")("MDF","MDF","MDF")("MDF","MDF","MDF")\n",Mzz(t->p0,3*i),Mzz(t->p1,3*i),Mzz(t->p2,3*i),Mzz(t->p0,3*i+1),Mzz(t->p1,3*i+1),Mzz(t->p2,3*i+1),Mzz(t->p0,3*i+2),Mzz(t->p1,3*i+2),Mzz(t->p2,3*i+2));
#endif
    }

   }
   t=DAG_traverse_prune_list(&dt);
  }


  /* normalize all potentials by dividing their max values */
  for (i=0; i<M.count; i++) {
   for (j=0; j<degree_of_freedom; j++) {
    if ( max_value[j]>TOL) {
    Mzz(i,3*j) /=max_value[j];
    Mzz(i,3*j+1) /=max_value[j];
    Mzz(i,3*j+2) /=max_value[j];
    }
   }
  }
   g_maxpot = 1.0;
   g_minpot = -1.0;



#ifdef DEBUG 
     printf("potentials max=%lf min=%lf\n",g_maxpot,g_minpot);
#endif
 /**********************************************************/
 /* destroy hash table */
 htbl_destroy(&H);

 
	for (j = 0; j < NN; j++) {
   free(S[j]);
   free(T[j]);
   free(v[j]);
  }
  free(S);
  free(T);
  free(v);
 free(r_hash);
 free(renumber);
 free(max_value);
 return(1);

}
