/*****************************************************************************
 * $Id: threadid.c,v 1.2 2005/08/23 12:30:42 killabyte Exp $
 *
 * This file implements the algorithms used to initialize and use the callback
 * thread stacks.
 *
 * ---------------------------------------------------------------------------
 * pDI-Tools - portable Dynamic Instrumentation Tools
 *   (C) 2004, 2005 Gerardo Garca Pea
 *   Programmed by Gerardo Garca Pea - Inspired on CEPBA DItools
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301
 *   USA
 *
 *****************************************************************************/

#include<config.h>
#include<threadid.h>
#include<pdiconfig.h>
#include<log.h>

/*****************************************************************************
 * static int (*threadIdResolver)(void)
 *
 *   Puntero a funcin (slo visible en este objeto) a una funcin capaz de
 *   decirnos que nmero de thread se est ejecutando.
 *
 *   Esta funcin debe devolver un valor entre 0 y N, donde N es el nmero de
 *   threads que estn corriendo (vivos) actualmente en la aplicacin.
 *
 *****************************************************************************/

static int (*threadIdResolver)(void);

/*****************************************************************************
 * void **_pdi_callback_stacks
 *
 *   Puntero a una tabla de punteros. Cada entrada de la tabla apunta a una
 *   pila de tamao PDICFG.cb_stack_size o tiene el valor NULL.
 *   Esta variable slo se usa si el parmetro PDICFG.max_threads es mayor que
 *   cero.
 *
 * void *pdi_callback_stack;
 *   Puntero a la pila para los callbacks. Slo se usa este puntero cuando
 *   'max_threads' es 0, ya que entonces se considera que no hay threads y por
 *   lo tanto slo se usa una pila para los callbacks.
 *
 *****************************************************************************/

void **_pdi_callback_stacks;
void *_pdi_callback_stack;

/*****************************************************************************
 * int (*_pdi_ebe_getThreadIdResolver(void))(void)
 * void _pdi_ebe_setThreadIdResolver(int (*thresolver)(void))
 *
 * Descripcin:
 *   Permiten consultar y cambiar el "thread id resolver" actualmente
 *   configurado.
 *
 * Parmetros:
 *   thresolver - puntero al nuevo "thread id resolver"
 *
 * Devuelve:
 *   En el caso de _pdi_ebe_setThreadIdResolver() no se devuelve nada.
 *   _pdi_ebe_getThreadIdResolver devuelve un puntero a la funcin actualmente
 *   configurada.
 *
 *****************************************************************************/

int (*_pdi_ebe_getThreadIdResolver(void))(void)
{
  return threadIdResolver;
}

void _pdi_ebe_setThreadIdResolver(int (*thresolver)(void))
{
  threadIdResolver = thresolver;
}

/*****************************************************************************
 * int _pdi_ebe_initGetThreadId(void)
 *
 * Descripcin:
 *   Inicializa las estructuras de datos de este fichero.
 *
 * Parmetros:
 *   - ninguno -
 *
 * Devuelve:
 *   0 en caso de xito, otro valor en caso de error.
 *
 *****************************************************************************/

int _pdi_ebe_initGetThreadId(void)
{
  int i;

  if(PDICFG.max_threads > 0)
  {
    /* Reservamos memoria para la tabla _pdi_callback_stacks */
    if((_pdi_callback_stacks = malloc(PDICFG.max_threads * sizeof(void *))) == NULL)
    {
      _pdi_error(THIS, "Cannot alloc memory for the pointer table to the stack threads.");
      return -1;
    }
    memset(_pdi_callback_stacks, 0, PDICFG.max_threads * sizeof(void *));

    /* Inicializamos los primeros PDICFG.num_threads threads */
    for(i = 0; i < PDICFG.num_threads; i++)
      if((_pdi_callback_stacks[i] = malloc(PDICFG.cb_stack_size)) == NULL)
      {
        _pdi_error(THIS, "Cannot alloc memory for %d callback thread stacks.", PDICFG.num_threads);
        for(i--; i >= 0; i--)
        {
          free(_pdi_callback_stacks[i]);
          _pdi_callback_stacks[i] = NULL;
        }
        free(_pdi_callback_stacks);
        _pdi_callback_stacks = NULL;
        return -1;
      }
  } else
    if((_pdi_callback_stack = malloc(PDICFG.cb_stack_size)) == NULL)
    {
      _pdi_error(THIS, "Cannot alloc memory for the callback stack.");
      return -1;
    }

  return 0;
}

/*****************************************************************************
 * void _pdi_ebe_finiGetThreadId(void)
 *
 * Descripcin:
 *   Termina y libera la memoria usada por las estructuras de este fichero.
 *
 * Parmetros:
 *   - ninguno -
 *
 * Devuelve:
 *   Nada.
 *
 *****************************************************************************/

void _pdi_ebe_finiGetThreadId(void)
{
  int i;

  /* Inicializamos los primeros PDICFG.num_threads threads */
  if(PDICFG.max_threads > 0)
  {
    for(i = 0; i < PDICFG.num_threads; i++)
      if(_pdi_callback_stacks[i])
      {
        free(_pdi_callback_stacks[i]);
        _pdi_callback_stacks[i] = NULL;
      }
  } else {
    free(_pdi_callback_stack);
    _pdi_callback_stack = NULL;
  }
}

/*****************************************************************************
 * int _pdi_ebe_getThreadCallbackStack(int *p_thread_id)
 *
 * Descripcin:
 *   Esta es la funcin encargada de llamar a nuestro "thread id resolver" y ya
 *   de paso controla que no se haya excedido el nmero mximo de threads y
 *   reserva ms memoria segn convenga.
 *
 * Parmetros:
 *   p_thread_id - [salida] si no es NULL, se pone en el entero al que apunta
 *                 el identificador del thread actual.
 *
 * Devuelve:
 *   Un puntero al top de la pila del thread en ejecucin.
 *   Si hubiese un error abortara directamente la ejecucin.
 *
 *****************************************************************************/

void *_pdi_ebe_getThreadCallbackStack(int *p_thread_id)
{
  int thread_id;

  /* Si no hay threads, devolvemos de inmediato la pila _pdi_callback_stack */
  if(!PDICFG.max_threads)
    return &_pdi_callback_stack;

  /* Primero de todo: averiguamos cual es el thread actual */
  if(!threadIdResolver)
  {
    _pdi_error(THIS, "A 'thread id resolver' has NOT defined.");
    thread_id = 0;
  } else
    thread_id = threadIdResolver();

  if(p_thread_id)
    *p_thread_id = thread_id;

  /* Si se ha superado la cota mxima de threads, debemos morir */
  if(thread_id >= PDICFG.max_threads)
  {
    _pdi_error(THIS, "Maximum thread limit has reached (fixed on %d threads).",
               PDICFG.max_threads);
    _pdi_error(THIS, "Cannot recover from this error. Aborting.");
    abort();
  }

  /* Si estamos en modo de ejecucin con una cota mxima de threads, y el
   * thread_id es mayor al nmero de pilas que tenemos, entonces tenemos que
   * morir */
  if(!_pdi_callback_stacks[thread_id]
  && (_pdi_callback_stacks[thread_id] = malloc(PDICFG.cb_stack_size)) == NULL)
  {
    _pdi_error(THIS, "Cannot alloc a new stack for thread %d.", thread_id);
    _pdi_error(THIS, "Cannot recover from this error. Aborting.");
    abort();
  }

  return _pdi_callback_stacks + thread_id;
}

