/*****************************************************************************
 * $Id: sl-rel.c,v 1.1 2005/08/28 17:34:36 killabyte Exp $
 *
 * This file implements some function to search REL/A structures.
 *
 * ---------------------------------------------------------------------------
 * pDI-Tools - portable Dynamic Instrumentation Tools
 *   (C) 2004, 2005 Gerardo Garca Pea
 *   Programmed by Gerardo Garca Pea - Inspired on CEPBA DItools
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301
 *   USA
 *
 *****************************************************************************/

#include<config.h>
#include<stdlib.h>
#include<log.h>
#include<solaris/elfops.h>

/*****************************************************************************
 * static SOLARIS_PLTREL *searchJmpRel(PDI_SOLARIS_ELFOBJ *obj,
 *                                     unsigned int symndx)
 *
 * Description:
 *   This function search in section DT_JMPREL of object 'obj' the REL/A
 *   structure of the symbol with index 'symndx' in the symbol table, if it
 *   exists of course.
 *
 *   searchJmpRel() makes a linear search in DT_JMPREL.
 *
 * Parameters:
 *   obj    - object where we are looking for the relocation structure
 *   symndx - index of the searched symbol in symbol table
 *
 * Returns:
 *   A pointer to the REL/A structure or NULL if it doesn't exist.
 *
 *****************************************************************************/

static SOLARIS_PLTREL *searchJmpRel(PDI_SOLARIS_ELFOBJ *obj, unsigned int symndx)
{
  ElfW(Addr) pos;

  for(pos = obj->jmpRelTable;
      pos < obj->jmpRelTable + obj->jmpRelTableSize;
      pos += SOLARIS_PLTREL_SZ)
    if(ELFW_R_SYM(((SOLARIS_PLTREL *) pos)->r_info) == symndx)
      return (SOLARIS_PLTREL *) pos;

  return NULL;
}

/*****************************************************************************
 * SOLARIS_PLTREL *_pdi_solaris_getFunctionRel(PDI_SOLARIS_ELFOBJ *obj,
 *                                             unsigned int symndx)
 *
 * Description:
 *   This function search in object 'obj' the REL/A structure of the symbol
 *   with index 'symndx' in the symbol table, if it exists of course.
 *
 * Parameters:
 *   obj    - target object
 *   symndx - index of the searched symbol in symbol table
 *
 * Returns:
 *   A pointer to the REL/A structure or NULL if it doesn't exist.
 *
 *****************************************************************************/

SOLARIS_PLTREL *_pdi_solaris_getFunctionRel(PDI_SOLARIS_ELFOBJ *obj,
                                            unsigned int symndx)
{
  SOLARIS_PLTREL *rel;

  /* make a linear search in section DT_JMPREL */
  rel = searchJmpRel(obj, symndx);

  /* if it have not found this function exits quietly returning NULL */
  if(!rel)
    return NULL;

  /* Check that is is a R_SPARC_JMP_SLOT relocation structure */
  if(ELFW(R_TYPE)(rel->r_info) != R_SPARC_JMP_SLOT)
  {
    _pdi_debug(THIS, "Symbol '%s' does not have a R_SPARC_JMP_SLOT relocation structure.",
               PDI_GET_SYMBOL_NAME(obj, symndx));
    return NULL;
  }

  return rel;
}

