/*****************************************************************************
 * $Id: sl-redefine.c,v 1.1 2005/08/28 17:05:14 killabyte Exp $
 *
 * This file implements '_pdi_solaris_redefine()' which allows to redefine a
 * function to another function.
 *
 * ---------------------------------------------------------------------------
 * pDI-Tools - portable Dynamic Instrumentation Tools
 *   (C) 2004, 2005 Gerardo Garca Pea
 *   Programmed by Gerardo Garca Pea - Inspired on CEPBA DItools
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301
 *   USA
 *
 *****************************************************************************/

#include<config.h>
#include<pdiconfig.h>
#include<ebeif.h>
#include<log.h>
#include<solaris/elfops.h>

/*****************************************************************************
 * int _pdi_solaris_redefine(PDI_ELFOBJ *eo, PDI_INTERCEPT *i)
 *
 * Description:
 *   Redefines function 'i->symbol' in objecty 'eo'. Object 'eo' must contain
 *   the implementation of this function, and not only a reference. Redefining
 *   this function all objects that use it will be relinked to our newer
 *   definition.
 *
 * Parameters:
 *   eo - object which contains the function to redefine
 *   i  - Parameters of the new redefinition
 *
 * Returns:
 *   This function returns 0 if succesful, otherwise it returns a negative
 *   value.
 *
 *****************************************************************************/

int _pdi_solaris_redefine(PDI_ELFOBJ *eo, PDI_INTERCEPT *i)
{
  unsigned int symndx;
  ElfW(Sym) *symptr;
  SOLARIS_PLTREL *rel;
  PDI_ELFOBJ *ito; /* the iterating object */

  /* Search the symbol in symbol table */
  if((symptr = _pdi_solaris_getSymbolTableEntry(eo->o_arch,
                                                i->i_symbol,
                                                &symndx)) == NULL)
  {
    _pdi_error(THIS, "Cannot find function '%s::%s()'.",
               _pdi_ebe_getObjectName(eo), i->i_symbol);
    return -1;
  }

  /* If this object contains really a definition of this function, the value */
  /* of symptr->st_shndx will not be zero. If it is zero the user must be    */
  /* profoundly idiot, so we must notify that to him :(                      */
  if(symptr->st_shndx == SHN_UNDEF)
  {
    _pdi_error(THIS, "Object '%s' does not contain definition of '%s'.",
               _pdi_ebe_getObjectName(eo), i->i_symbol);
    return -1;
  }

  /* Now will proceed to make the redefinition. It consist on alter symptr  */
  /* pointer so it resolve to our definition.                               */
  /* NOTE 1: We must reajust this pointer to be relative to base address of */
  /*         the object we are working on. It is the reason of the strange  */
  /*         addition with lm->l_addr.                                      */
  /* NOTE 2: We save also the original address (viciously)!                */
  if(mprotect(
       (void *) ((unsigned long) symptr & (unsigned long) ~(PAGESIZE - 1)),
       PAGESIZE, PROT_READ | PROT_WRITE))
  {
    _pdi_error(THIS, "Cannot unprotect symbol info of '%s::%s()'.",
               _pdi_ebe_getObjectName(eo), i->i_symbol);
    return -1;
  }

  i->i_arch->u.redefinition.symbol_st_value =
    (void *) (symptr->st_value + eo->o_arch->lm->l_addr);
  symptr->st_value = ((ElfW(Addr)) i->i_wrapper) - eo->o_arch->lm->l_addr;

  /* Now comes the best part: iterate on all objects to apply or new def.   */
  /* Reason: it is possible that some object have called this function so   */
  /* it has a true pointer to it in PLTGOT.                                 */
  for(ito = _pdi_objlist; ito; ito = ito->o_next)
  {
    /* Search symbol on symbol table of this object       */
    /* (if this symbol is used in this object, of course) */
    if((symptr = _pdi_solaris_getSymbolTableEntry(ito->o_arch,
                                                i->i_symbol,
                                                &symndx)) == NULL)
      continue; /* no esta! En ese caso no hacemos nada... */
    
    /* Search its relocation */
    if((rel = _pdi_solaris_getFunctionRel(ito->o_arch, symndx)) == NULL)
      continue;

    /* Execute relink                                                   */
    /* NOTE: Don't save pointer because we will use the true pointer to */
    /*       the function when uninstalling this interposition.         */
    /* NOTE 2: If 'donttouch_backends' or 'donttouch_pdi' are on, then  */
    /*         we'll be good and relink to the original function.       */
    _pdi_solaris_alterPlt(ito, rel,
                          (PDICFG.donttouch_backends && ito->o_isBackend) ||
                          (PDICFG.donttouch_pdi && ito->o_alias && !strcmp(ito->o_alias, PDI_ALIAS_PDI))
                            ? i->i_arch->u.redefinition.symbol_st_value
                            : i->i_wrapper,
                          NULL);
  }

  return 0;
}

/*****************************************************************************
 * int _pdi_solaris_deactivateRedefinition(PDI_INTERCEPT *i, PDI_ELFOBJ *target)
 *
 * Description:
 *   Deactivate redefinition 'i' in object 'target'.
 *
 * Parameters:
 *   i      - redefinition to deactivate
 *   target - object where we want to deactivate this redefenition
 *
 * Returns:
 *   0 if all ok, -1 if error
 *
 *****************************************************************************/

int _pdi_solaris_deactivateRedefinition(PDI_INTERCEPT *i, PDI_ELFOBJ *target)
{
  unsigned int symndx;
  ElfW(Sym) *symptr;
  SOLARIS_PLTREL *rel;
  PDI_INTERCEPT *iti;

  /* Search for a relink (or callback) on this symbol. If it */
  /* exists we don't have to do anything here...             */
  for(iti = target->o_rules;
      iti &&
      iti->i_type != PDI_IT_CALLBACK &&
      strcmp(iti->i_symbol, i->i_symbol);
      iti = iti->i_next)
    ;
  if(iti)
    return 0;

  /* Search symbol in symbol table (if this symbol is used, of course) */
  if((symptr = _pdi_solaris_getSymbolTableEntry(target->o_arch, i->i_symbol, &symndx)) == NULL)
    return 0; /* doesnt exist! Byeeeeee!! */
  
  /* Search relocation of this symbol */
  if((rel = _pdi_solaris_getFunctionRel(target->o_arch, symndx)) == NULL)
    return 0;

  /* Deactivate redefinition using true definition of symbol */
  _pdi_solaris_alterPlt(target, rel, i->i_arch->u.redefinition.symbol_st_value, NULL);

  return 0;
}

/*****************************************************************************
 * int _pdi_solaris_undoRedefine(PDI_ELFOBJ *eo, PDI_INTERCEPT *i)
 *
 * Description:
 *   Uninstall redefinition 'i' which is installed on object 'eo'.
 *
 * Parameters:
 *   eo - object affected by 'i'
 *   i  - interposition to uninstall
 *
 * Returns:
 *   0 if ok, -1 otherwise.
 *
 *****************************************************************************/

int _pdi_solaris_undoRedefine(PDI_ELFOBJ *eo, PDI_INTERCEPT *i)
{
  unsigned int symndx;
  ElfW(Sym) *symptr;
  PDI_ELFOBJ *ito;

  /* In the same way when we redefined this function, search its info in */
  /* symbol table to undo the changes.                                   */
  if((symptr = _pdi_solaris_getSymbolTableEntry(eo->o_arch,
                                                i->i_symbol,
                                                &symndx)) == NULL)
  {
    _pdi_error(THIS, "Cannot find function '%s::%s()'.",
               _pdi_ebe_getObjectName(eo), i->i_symbol);
    return -1;
  }

  /* Now we'll proceed to undo symbol redefinition. I know it is not */
  /* necessary to use mprotect(2) this time too, but it is like the  */
  /* word 'colosal' (see Kill Bill 2)                                */
  if(mprotect(
       (void *) ((unsigned long) symptr & (unsigned long) ~(PAGESIZE - 1)),
       PAGESIZE, PROT_READ | PROT_WRITE))
  {
    _pdi_error(THIS, "Cannot unprotect symbol info of '%s::%s()'.",
               _pdi_ebe_getObjectName(eo), i->i_symbol);
    return -1;
  }

  symptr->st_value = ((ElfW(Addr)) i->i_arch->u.redefinition.symbol_st_value)
                   - eo->o_arch->lm->l_addr;

  /* Iterate on all objects to undo relinks derived from this redefinition */
  for(ito = _pdi_objlist; ito; ito = ito->o_next)
    _pdi_solaris_deactivateRedefinition(i, ito);

  return 0;
}

