/*****************************************************************************
 * $Id: sl-debug.c,v 1.1 2005/08/26 09:08:35 killabyte Exp $
 *
 * Some tests used in debug mode to check the consistency of ELF structures in
 * memory.
 *
 * ---------------------------------------------------------------------------
 * pDI-Tools - portable Dynamic Instrumentation Tools
 *   (C) 2004, 2005 Gerardo Garca Pea
 *   Programmed by Gerardo Garca Pea - Inspired on CEPBA DItools
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301
 *   USA
 *
 *****************************************************************************/

#include<config.h>
#include<ebeif.h>
#include<log.h>

/*****************************************************************************
 * All known sections names (well... almost all) for this platform
 *
 *   Used to dump ELF structures in a human readable way. I really wish that
 *   macros used to build the list don't give too much problems on the
 *   different C compilers of this universe...
 *
 *****************************************************************************/

typedef struct {
  int     d_tag;
  char    *name;
  char    *desc;
} DT_NAMES;
  
#define BDTDESC(x,y)            { x, #x, y }
#define BDTDESC_END             { 0, NULL, NULL }

static DT_NAMES dt_names[] = {
  BDTDESC(DT_NULL,            "Marks end of dynamic section"),
  BDTDESC(DT_NEEDED,          "Name of needed library"),
  BDTDESC(DT_PLTRELSZ,        "Size in bytes of PLT relocs"),
  BDTDESC(DT_PLTGOT,          "Processor defined value"),
  BDTDESC(DT_HASH,            "Address of symbol hash table"),
  BDTDESC(DT_STRTAB,          "Address of string table"),
  BDTDESC(DT_SYMTAB,          "Address of symbol table"),
  BDTDESC(DT_RELA,            "Address of Rela relocs"),
  BDTDESC(DT_RELASZ,          "Total size of Rela relocs"),
  BDTDESC(DT_RELAENT,         "Size of one Rela reloc"),
  BDTDESC(DT_STRSZ,           "Size of string table"),
  BDTDESC(DT_SYMENT,          "Size of one symbol table entry"),
  BDTDESC(DT_INIT,            "Address of init function"),
  BDTDESC(DT_FINI,            "Address of termination function"),
  BDTDESC(DT_SONAME,          "Name of shared object"),
  BDTDESC(DT_RPATH,           "Library search path (deprecated)"),
  BDTDESC(DT_SYMBOLIC,        "Start symbol search here"),
  BDTDESC(DT_REL,             "Address of Rel relocs"),
  BDTDESC(DT_RELSZ,           "Total size of Rel relocs"),
  BDTDESC(DT_RELENT,          "Size of one Rel reloc"),
  BDTDESC(DT_PLTREL,          "Type of reloc in PLT"),
  BDTDESC(DT_DEBUG,           "For debugging; unspecified"),
  BDTDESC(DT_TEXTREL,         "Reloc might modify .text"),
  BDTDESC(DT_JMPREL,          "Address of PLT relocs"),
  BDTDESC(DT_BIND_NOW,        "Process relocations of object"),
  BDTDESC(DT_INIT_ARRAY,      "Array with addresses of init fct"),
  BDTDESC(DT_FINI_ARRAY,      "Array with addresses of fini fct"),
  BDTDESC(DT_INIT_ARRAYSZ,    "Size in bytes of DT_INIT_ARRAY"),
  BDTDESC(DT_FINI_ARRAYSZ,    "Size in bytes of DT_FINI_ARRAY"),
  BDTDESC(DT_RUNPATH,         "Library search path"),
  BDTDESC(DT_FLAGS,           "Flags for the object being loaded"),
  BDTDESC(DT_ENCODING,        "Start of encoded range"),
  BDTDESC(DT_PREINIT_ARRAY,   "Array with addresses of preinit fc"),
  BDTDESC(DT_PREINIT_ARRAYSZ, "size in bytes of DT_PREINIT_ARRAY"),
  BDTDESC(DT_LOOS,            "Start of OS-specific"),
  BDTDESC(DT_HIOS,            "End of OS-specific"),
  BDTDESC(DT_LOPROC,          "Start of processor-specific"),
  BDTDESC(DT_HIPROC,          "End of processor-specific"),
  BDTDESC(DT_VALRNGLO,        NULL),
  BDTDESC(DT_CHECKSUM,        NULL),
  BDTDESC(DT_PLTPADSZ,        NULL),
  BDTDESC(DT_MOVEENT,         NULL),
  BDTDESC(DT_MOVESZ,          NULL),
  BDTDESC(DT_FEATURE_1,       "Feature selection (DTF_*)."),
  BDTDESC(DT_POSFLAG_1,       "Flags for DT_* entries, effecting the following DT_* entry."),
  BDTDESC(DT_SYMINSZ,         "Size of syminfo table (in bytes)"),
  BDTDESC(DT_SYMINENT,        "Entry size of syminfo"),
  BDTDESC(DT_VALRNGHI,        NULL),
  BDTDESC(DT_ADDRRNGLO,       NULL),
  BDTDESC(DT_CONFIG,          "Configuration information."),
  BDTDESC(DT_DEPAUDIT,        "Dependency auditing."),
  BDTDESC(DT_AUDIT,           "Object auditing."),
  BDTDESC(DT_PLTPAD,          "PLT padding."),
  BDTDESC(DT_MOVETAB,         "Move table."),
  BDTDESC(DT_SYMINFO,         "Syminfo table."),
  BDTDESC(DT_ADDRRNGHI,       NULL),
  BDTDESC(DT_VERSYM,          NULL),
  BDTDESC(DT_RELACOUNT,       NULL),
  BDTDESC(DT_RELCOUNT,        NULL),
  BDTDESC(DT_FLAGS_1,         "State flags, see DF_1_* below."),
  BDTDESC(DT_VERDEF,          "Address of version definition table"),
  BDTDESC(DT_VERDEFNUM,       "Number of version definitions"),
  BDTDESC(DT_VERNEED,         "Address of table with needed versions"),
  BDTDESC(DT_VERNEEDNUM,      "Number of needed versions"),
  BDTDESC(DT_AUXILIARY,       "Shared object to load before self"),
  BDTDESC(DT_FILTER,          "Shared object to get values from"),
  BDTDESC_END
};

typedef struct {
  int     val;
  char    *name;
  char    *desc;
} FLAG_DESC;
  
#define FLGDESC(x,y)            { x, #x, y }
#define FLGDESC_END             { 0, NULL, NULL }

static FLAG_DESC flags_1_names[] = {
  FLGDESC(DF_1_NOW,        "set RTLD_NOW for this object"),
  FLGDESC(DF_1_GLOBAL,     "set RTLD_GLOBAL for this object"),
  FLGDESC(DF_1_GROUP,      "set RTLD_GROUP for this object"),
  FLGDESC(DF_1_NODELETE,   "set RTLD_NODELETE for this object"),
  FLGDESC(DF_1_LOADFLTR,   "trigger filtee loading at runtime"),
  FLGDESC(DF_1_INITFIRST,  "set RTLD_INITFIRST for this object"),
  FLGDESC(DF_1_NOOPEN,     "set RTLD_NOOPEN for this object"),
  FLGDESC(DF_1_ORIGIN,     "ORIGIN processing required"),
  FLGDESC(DF_1_DIRECT,     "direct binding enabled"),
  FLGDESC(DF_1_TRANS,      "DF_1_TRANS"),
  FLGDESC(DF_1_INTERPOSE,  "object is an 'interposer'"),
  FLGDESC(DF_1_NODEFLIB,   "ignore default library search path"),
  FLGDESC(DF_1_NODUMP,     "object can't be dldump(3x)'ed"),
  FLGDESC(DF_1_CONFALT,    "configuration alternative created"),
  FLGDESC(DF_1_ENDFILTEE,  "filtee terminates filters search"),
  FLGDESC(DF_1_DISPRELDNE, "disp reloc applied at build time"),
  FLGDESC(DF_1_DISPRELPND, "disp reloc applied at run-time"),
  FLGDESC_END
};

static FLAG_DESC flags_names[] = {
  FLGDESC(DF_ORIGIN,      "ORIGIN processing required"),
  FLGDESC(DF_SYMBOLIC,    "symbolic bindings in effect"),
  FLGDESC(DF_TEXTREL,     "text relocations remain"),
  FLGDESC(DF_BIND_NOW,    "process all relocations"),
  FLGDESC(DF_STATIC_TLS,  "obj. contains static TLS refs"),
  FLGDESC_END
};

static FLAG_DESC feature_1_names[] = {
  FLGDESC(DTF_1_PARINIT, "partially initialization feature"),
  FLGDESC(DTF_1_CONFEXP, "configuration file expected"),
  FLGDESC_END
};

/*****************************************************************************
 * Section check table
 * 
 *   In this table is expressed which sections should exist always and which
 *   are optional. Only these sections will be examined, any other section
 *   found will be ignored. Note that you can add a pointer to a function to
 *   check deeply a section. The order in this table is important.
 *
 *****************************************************************************/

typedef struct _tag_CHECK_DT_SECT {
  int          d_tag;
  int          d_un;
#define D_UN_IGNORED    0
#define D_UN_VAL        1
#define D_UN_PTR        2
#define D_UN_UNSPEC     3
  int          pres_ex;
  int          pres_so;
#define PRES_IGNORED     0
#define PRES_OPTIONAL    1
#define PRES_SEMIMANDA   2
#define PRES_MANDATORY   3
  int          found;
  int          (*func)(struct _tag_CHECK_DT_SECT *sect, struct link_map *lm);
} CHECK_DT_SECT;

static int showFlags(CHECK_DT_SECT *sect, struct link_map *lm);
static int showSOName(CHECK_DT_SECT *sect, struct link_map *lm);
static int checkNeeded(CHECK_DT_SECT *sect, struct link_map *lm);
static int checkRelSections(CHECK_DT_SECT *sect, struct link_map *lm);

/* NOTE: The order in this table is important because */
/* some sections depend on the existence of other!    */
static CHECK_DT_SECT sections[] = {
  { DT_FLAGS,     D_UN_VAL,     PRES_OPTIONAL,  PRES_OPTIONAL,  0, showFlags },
  { DT_FLAGS_1,   D_UN_VAL,     PRES_OPTIONAL,  PRES_OPTIONAL,  0, showFlags },
  { DT_FEATURE_1, D_UN_VAL,     PRES_OPTIONAL,  PRES_OPTIONAL,  0, showFlags },
  { DT_STRTAB,    D_UN_PTR,     PRES_MANDATORY, PRES_MANDATORY, 0, NULL },
  { DT_STRSZ,     D_UN_VAL,     PRES_MANDATORY, PRES_MANDATORY, 0, NULL },
  { DT_SONAME,    D_UN_VAL,     PRES_IGNORED,   PRES_OPTIONAL,  0, showSOName },
  { DT_SYMTAB,    D_UN_PTR,     PRES_MANDATORY, PRES_MANDATORY, 0, NULL },
  { DT_SYMENT,    D_UN_VAL,     PRES_MANDATORY, PRES_MANDATORY, 0, NULL },
  { DT_NEEDED,    D_UN_VAL,     PRES_OPTIONAL,  PRES_OPTIONAL,  0, checkNeeded },
  { DT_RELA,      D_UN_PTR,     PRES_SEMIMANDA, PRES_OPTIONAL,  0, checkRelSections },
  { DT_RELASZ,    D_UN_VAL,     PRES_SEMIMANDA, PRES_OPTIONAL,  0, checkRelSections },
  { DT_RELAENT,   D_UN_VAL,     PRES_SEMIMANDA, PRES_OPTIONAL,  0, checkRelSections },
  { DT_REL,       D_UN_PTR,     PRES_SEMIMANDA, PRES_OPTIONAL,  0, checkRelSections },
  { DT_RELSZ,     D_UN_VAL,     PRES_SEMIMANDA, PRES_OPTIONAL,  0, checkRelSections },
  { DT_RELENT,    D_UN_VAL,     PRES_SEMIMANDA, PRES_OPTIONAL,  0, checkRelSections },
  { DT_PLTREL,    D_UN_VAL,     PRES_OPTIONAL,  PRES_OPTIONAL,  0, checkRelSections },
  { DT_PLTRELSZ,  D_UN_VAL,     PRES_OPTIONAL,  PRES_OPTIONAL,  0, checkRelSections },
  { DT_JMPREL,    D_UN_PTR,     PRES_OPTIONAL,  PRES_OPTIONAL,  0, checkRelSections },
  { DT_PLTGOT,    D_UN_PTR,     PRES_OPTIONAL,  PRES_OPTIONAL,  0, NULL },
  { DT_HASH,      D_UN_PTR,     PRES_MANDATORY, PRES_MANDATORY, 0, NULL },
  { DT_INIT,      D_UN_PTR,     PRES_OPTIONAL,  PRES_OPTIONAL,  0, NULL },
  { DT_FINI,      D_UN_PTR,     PRES_OPTIONAL,  PRES_OPTIONAL,  0, NULL },
  { DT_RPATH,     D_UN_VAL,     PRES_OPTIONAL,  PRES_IGNORED,   0, NULL },
  { DT_SYMBOLIC,  D_UN_IGNORED, PRES_IGNORED,   PRES_OPTIONAL,  0, NULL },
  { DT_DEBUG,     D_UN_PTR,     PRES_OPTIONAL,  PRES_IGNORED,   0, NULL },
  { DT_TEXTREL,   D_UN_IGNORED, PRES_OPTIONAL,  PRES_OPTIONAL,  0, NULL },
  { DT_BIND_NOW,  D_UN_IGNORED, PRES_OPTIONAL,  PRES_OPTIONAL,  0, NULL },
  { DT_NULL,      D_UN_IGNORED, PRES_MANDATORY, PRES_MANDATORY, 1, NULL }
};

static char *getSectionName(ElfW(Sword) d_tag)
{
  DT_NAMES *act = dt_names;
  for(act = dt_names; act->name; act++)
    if(act->d_tag == d_tag)
      return act->name;

  return "-desconocida-";
}

static ElfW(Dyn) *searchDynSection(ElfW(Dyn) *dyn, ElfW(Sword) d_tag)
{
  if(d_tag == DT_NULL)
    return NULL;

  for(; dyn->d_tag != DT_NULL; dyn++)
    if(dyn->d_tag == d_tag)
      return dyn;

  return NULL;
}

static ElfW(Dyn) *nextDynSection(ElfW(Dyn) *dyn, ElfW(Sword) d_tag)
{
  if(d_tag == DT_NULL) return NULL;
  if(dyn->d_tag != DT_NULL) dyn++;

  for(; dyn->d_tag != DT_NULL; dyn++)
    if(dyn->d_tag == d_tag)
      return dyn;

  return NULL;
}

static CHECK_DT_SECT *getSection(ElfW(Sword) d_tag)
{
  CHECK_DT_SECT *sect;
  for(sect = sections; sect->d_tag != DT_NULL; sect++)
    if(sect->d_tag == d_tag)
      return sect;
  return NULL;
}

/* This function requires that DT_STRTAB has been checked */
static int checkNeeded(CHECK_DT_SECT *sect, struct link_map *lm)
{
  ElfW(Dyn) *dyn;
  char *strtab;
  ElfW(Addr) base_addr = lm->l_prev ? lm->l_addr : 0;

  if(sect->found)
  {
    strtab = (char *) searchDynSection(lm->l_ld, DT_STRTAB)->d_un.d_val + base_addr;
    _pdi_debug(__FILE__, NULL, "    + Library dependencies (DT_NEEDED):");

    for(dyn = searchDynSection(lm->l_ld, sect->d_tag);
        dyn;
        dyn = nextDynSection(dyn, sect->d_tag))
    {
      _pdi_debug(__FILE__, NULL, "      - '%s'",
                 strtab + dyn->d_un.d_val);
    }
  }

  return 0;
}

/* Show contents of a flags section (DT_FLAGS, DT_FEATURE_1, ...) */
static int showFlags(CHECK_DT_SECT *sect, struct link_map *lm)
{
  ElfW(Dyn) *dyn;
  FLAG_DESC *fd;

  if(!sect->found)
    return 0;

  switch(sect->d_tag)
  {
    case DT_FLAGS:     fd = flags_names;     break;
    case DT_FLAGS_1:   fd = flags_1_names;   break;
    case DT_FEATURE_1: fd = feature_1_names; break;
    default: return 0;
  }

  dyn = searchDynSection(lm->l_ld, sect->d_tag);
  if(dyn->d_un.d_val)
  {
    _pdi_debug(__FILE__, NULL, "    + Enabled flags in %s:", getSectionName(sect->d_tag));
    for(; fd->val; fd++)
      if(dyn->d_un.d_val & fd->val)
        _pdi_debug(__FILE__, NULL, "         %s (%s)", fd->name, fd->desc);
  } else
    _pdi_debug(__FILE__, NULL, "    + %s has not any enabled flag.", getSectionName(sect->d_tag));

  return 0;
}

static int showSOName(CHECK_DT_SECT *sect, struct link_map *lm)
{
  ElfW(Addr) base_addr = lm->l_prev ? lm->l_addr : 0;

  if(!sect->found)
    return 0;

  _pdi_debug(__FILE__, NULL, "    + DT_SONAME = '%s'",
             (char *) (
                 searchDynSection(lm->l_ld, DT_STRTAB)->d_un.d_ptr
               + base_addr
               + searchDynSection(lm->l_ld, sect->d_tag)->d_un.d_ptr)
             );

  return 0;
}

static void dumpJmpRelSection(struct link_map *lm)
{
  ElfW(Dyn) *DynJmpRel,
            *DynPltRel,
            *DynPltGot,   
            *DynPltRelSz;
  ElfW(Addr) base, size_rel;
  int *ltags, *lcount, i;
  ElfW(Addr) base_addr = lm->l_prev ? lm->l_addr : 0;

  /* Get a pointer to each section */
  if((DynJmpRel = searchDynSection(lm->l_ld, DT_JMPREL)) == NULL
  || (DynPltRel = searchDynSection(lm->l_ld, DT_PLTREL)) == NULL
  || (DynPltGot = searchDynSection(lm->l_ld, DT_PLTGOT)) == NULL
  || (DynPltRelSz = searchDynSection(lm->l_ld, DT_PLTRELSZ)) == NULL)
  {
    _pdi_error(THIS, "One or more important sections are missing "
                     "(DT_JMPREL, DT_PLTREL, DT_PLTGOT o DT_PLTRELSZ).");
    exit(1);
  }

  /* Guess what relocation structure is used here (REL or RELA) */
  size_rel = DynPltRel->d_un.d_val == DT_REL
               ? sizeof(ElfW(Rel))
               : sizeof(ElfW(Rela));

  /* Check that all have a nice sense */
  if((DynPltRelSz->d_un.d_val % size_rel) != 0)
    _pdi_warning(THIS,
                 "The value of DT_PLTRELSZ (%d) should be multiple of sizeof(%s) (%d).",
                 DynPltRelSz->d_un.d_val,
                 DynPltRel->d_un.d_val == DT_REL ? "DT_REL" : "DT_RELA",
                 size_rel);

  /* Alloc memory for the worst case */
  if((ltags  = malloc(sizeof(int) * (DynPltRelSz->d_un.d_val / size_rel))) == NULL
  || (lcount = malloc(sizeof(int) * (DynPltRelSz->d_un.d_val / size_rel))) == NULL)
  {
    _pdi_error(THIS, "No memory for para 'ltags' and 'lcount'.");
    exit(1);
  }
  memset(ltags,  0, sizeof(int) * (DynPltRelSz->d_un.d_val / size_rel));
  memset(lcount, 0, sizeof(int) * (DynPltRelSz->d_un.d_val / size_rel));

  /* Make an inventory */
  for(base = DynJmpRel->d_un.d_ptr + base_addr;
      base < (DynJmpRel->d_un.d_ptr + base_addr + DynPltRelSz->d_un.d_val);
      base += size_rel)
  {
/* List JMPREL's ... it shows the separation between them
 *  if(!base_addr)
 *    _pdi_debug(THIS, "---> JMPREL->r_offset = 0x%lx",
 *               ((ElfW(Rel) *) base)->r_offset + base_addr);
 */

    /* Search the type of this element in table */
    for(i = 0;
        ltags[i] && ltags[i] != ELFW_R_TYPE(((ElfW(Rel) *) base)->r_info);
        i++)
      ;

    /* Register this type if it does not yet exist in the inventory */
    /* and, inconditionally, increment its counter                  */
    ltags[i] = ELFW_R_TYPE(((ElfW(Rel) *) base)->r_info);
    lcount[i] ++;
  }

  /* List inventory contents */
  _pdi_debug(__FILE__, NULL,
             "      - Listing %s different types found in DT_JMPREL:",
             DynPltRel->d_un.d_val == DT_REL ? "DT_REL" : "DT_RELA");
  for(i = 0; ltags[i]; i++)
    _pdi_debug(__FILE__, NULL, "         Type %d found %d times..",
               ltags[i], lcount[i]);

  /* free mem */
  free(ltags);
  free(lcount);
}

static int checkRelSections(CHECK_DT_SECT *sect, struct link_map *lm)
{
  ElfW(Dyn) *dyn;

  dyn = searchDynSection(lm->l_ld, sect->d_tag);
  if(sect->found)
    switch(sect->d_tag)
    {
      case DT_JMPREL:
        _pdi_debug(__FILE__, NULL, "    + Found section DT_JMPREL.");
        _pdi_debug(__FILE__, NULL, "      - d_ptr = 0x%lx", dyn->d_un.d_ptr);
        /* Comprobamos que las secciones requeridas esten */
        if(!getSection(DT_PLTREL)->found)
        {
          _pdi_error(__FILE__, NULL, "      - WARNING: DT_JMPREL requires section DT_PLTREL!");
          return -1;
        }
        if(!getSection(DT_PLTRELSZ)->found)
        {
          _pdi_error(__FILE__, NULL, "      - WARNING: DT_JMPREL requires section DT_PLTRELSZ!");
          return -1;
        }
        /* Listamos los diferentes tipos de entradas en esta seccin */
        dumpJmpRelSection(lm);
        break;
      case DT_PLTREL:
        _pdi_debug(__FILE__, NULL,
                   "    + Found DT_PLTREL. DT_JMPREL uses structures of type '%s'",
                   dyn->d_un.d_val == DT_REL ? "DT_REL" : "DT_RELA");
        if(!getSection(DT_PLTRELSZ)->found)
        {
          _pdi_error(__FILE__, NULL, "      - WARNING: DT_PLTREL requires section DT_PLTRELSZ!");
          return -1;
        }
        break;
      case DT_RELA:
        _pdi_debug(__FILE__, NULL, "    + Found DT_RELA. (0x%lx)", dyn->d_un.d_ptr);
        if(!getSection(DT_RELASZ)->found)
        {
          _pdi_error(__FILE__, NULL, "      - WARNING: DT_RELA requires section DT_RELASZ!");
          return -1;
        }
        if(!getSection(DT_RELAENT)->found)
        {
          _pdi_error(__FILE__, NULL, "      - WARNING: DT_RELA requires section DT_RELAENT!");
          return -1;
        }
        break;
      case DT_REL:
        _pdi_debug(__FILE__, NULL, "    + Found DT_REL. (0x%lx)", dyn->d_un.d_ptr);
        if(!getSection(DT_RELSZ)->found)
        {
          _pdi_error(__FILE__, NULL, "      - WARNING: DT_REL requires section DT_RELSZ!");
          return -1;
        }
        if(!getSection(DT_RELENT)->found)
        {
          _pdi_error(__FILE__, NULL, "      - WARNING: DT_REL requires section DT_RELENT!");
          return -1;
        }
        break;
      case DT_RELASZ:
      case DT_RELAENT:
      case DT_RELSZ:
      case DT_RELENT:
      case DT_PLTRELSZ:
        break;
      default:
        _pdi_error(THIS, "Unknown section %d.", sect->d_tag);
        abort();
    }
  else
    switch(sect->d_tag)
    {
      case DT_RELA:
        if(getSection(DT_REL)->found)
          _pdi_debug(__FILE__, NULL, "      - Section DT_RELA not found, but DT_REL exists. All ok.");
        else {
          _pdi_error(__FILE__, NULL, "      - WARNING: Not found DT_RELA nor DT_REL.");
          return -1;
        }
        break;
      case DT_RELASZ:
        if(!getSection(DT_RELSZ)->found)
        {
          _pdi_error(__FILE__, NULL, "      - WARNING: Not found DT_RELASZ nor DT_RELSZ.");
          return -1;
        }
        break;
      case DT_RELAENT:
        if(!getSection(DT_RELENT)->found)
        {
          _pdi_error(__FILE__, NULL, "      - WARNING: Not found DT_RELAENT nor DT_RELENT.");
          return -1;
        }
        break;
      case DT_REL:
        if(getSection(DT_RELA)->found)
          _pdi_debug(__FILE__, NULL, "      - Section DT_REL not found, but DT_RELA exists. All ok.");
        else {
          _pdi_error(__FILE__, NULL, "      - WARNING: Not found DT_RELA nor DT_REL.");
          return -1;
        }
        break;
      case DT_RELSZ:
        if(!getSection(DT_RELASZ)->found)
        {
          _pdi_error(__FILE__, NULL, "      - WARNING: Not found DT_RELASZ nor DT_RELSZ.");
          return -1;
        }
        break;
      case DT_RELENT:
        if(!getSection(DT_RELAENT)->found)
        {
          _pdi_error(__FILE__, NULL, "      - WARNING: Not found DT_RELAENT nor DT_RELENT.");
          return -1;
        }
        break;
      case DT_PLTREL:
      case DT_PLTRELSZ:
        break;
      default:
        _pdi_error(THIS, "Unknown section %d.", sect->d_tag);
        abort();
    }
  return 0;
}

static void showSectionValues(CHECK_DT_SECT *sect, struct link_map *lm)
{
  ElfW(Dyn) *dyn;

  if(sect->found > 0)
  {
    _pdi_debug(__FILE__, NULL,
               sect->found > 1
                 ? "    + Section '%s' (0x%x), %d times."
                 : "    + Section '%s' (0x%x).",
               getSectionName(sect->d_tag), sect->d_tag,
               sect->found);
    if(sect->d_un != D_UN_IGNORED)
    {
      for(dyn = searchDynSection(lm->l_ld, sect->d_tag);
          dyn != NULL;
          dyn = nextDynSection(dyn, sect->d_tag))
        _pdi_debug(__FILE__, NULL, "      - %s = 0x%lx",
                   sect->d_un == D_UN_VAL ? "d_val" : "d_ptr",
                   sect->d_un == D_UN_VAL ? dyn->d_un.d_val : dyn->d_un.d_ptr);
    }
  }
}

static void resetSections(void)
{
  int i;

  for(i = 0; sections[i].d_tag != DT_NULL; i++)
    sections[i].found = 0;
}

static void countSections(struct link_map *lm)
{
  ElfW(Dyn) *dyn;
  int i;

  for(dyn = lm->l_ld; dyn->d_tag != DT_NULL; dyn++)
  {
    for(i = 0; sections[i].d_tag != DT_NULL && sections[i].d_tag != dyn->d_tag; i++)
      ;
    if(sections[i].d_tag == dyn->d_tag)
      sections[i].found ++;
    else
      _pdi_debug(__FILE__, NULL,
                 "    + Found unknown section '%s' (0x%x).",
                 getSectionName(dyn->d_tag),
                 dyn->d_tag);
  }
}

static void checkSections(struct link_map *lm, int shared_object)
{
  int i, presence;

  for(i = 0; sections[i].d_tag != DT_NULL; i++)
  {
    presence = shared_object
                 ? sections[i].pres_so
                 : sections[i].pres_ex;

    if(presence == PRES_IGNORED)
    {
/* _pdi_debug(THIS, "Ignoring (%d).", sections[i].d_tag); */
      continue;
    }
    if(presence == PRES_MANDATORY && sections[i].found == 0)
    {
      _pdi_debug(__FILE__, NULL,
                 "    + Section '%s' (0x%x) is obligatory and it is NOT present.",
                 getSectionName(sections[i].d_tag),
                 sections[i].d_tag);
      exit(1);
    }
    if(presence == PRES_SEMIMANDA && sections[i].found == 0)
      _pdi_debug(__FILE__, NULL,
                 "    + Section '%s' (0x%x) is obligatory and SHOULD BE present.",
                 getSectionName(sections[i].d_tag),
                 sections[i].d_tag);

    if(sections[i].func != NULL)
    {
      if(sections[i].func(&(sections[i]), lm))
      {
        ElfW(Dyn) *dyn;
        _pdi_debug(__FILE__, NULL,
                   "  ---> ELF STRUCTURE OF THIS DSO DOES NOT SEEM TO BE CORRECT!");
        _pdi_debug(__FILE__, NULL, "      + Quick list of present sections:");
        for(dyn = lm->l_ld; dyn->d_tag != DT_NULL; dyn++)
          _pdi_debug(__FILE__, NULL, "           %s (0x%x)",
                     getSectionName(dyn->d_tag), dyn->d_tag);
        _pdi_debug(__FILE__, NULL, "           [end of list]");
        return;
      }
    } else
      showSectionValues(&(sections[i]), lm);
  }
}

void _pdi_solaris_debug_checkSections(void)
{
  struct link_map *lm;
  int shared_object;

  /* the first element of the list is the executable binary */
  shared_object = 0;

  _pdi_debug(__FILE__, NULL, "Check ELF structures in memory:");
  for(lm = _pdi_solaris_r_debug->r_map; lm; lm = lm->l_next)
  {
    _pdi_debug(__FILE__, NULL,
               "  # Checking %s '%s' (base=0x%lx).",
               strlen(lm->l_name) ? "object" : "executable binary",
               strlen(lm->l_name) ? lm->l_name : PDI_ALIAS_MAIN,
               lm->l_addr);
    resetSections();
    countSections(lm);
    checkSections(lm, shared_object);

    shared_object = 1;
  }
}

