/*****************************************************************************
 * $Id: rules.c,v 1.3 2005/08/23 12:37:50 killabyte Exp $
 *
 * No arch dependant rule managing (PDI_INTERCEPT structures).
 *
 * ---------------------------------------------------------------------------
 * pDI-Tools - portable Dynamic Instrumentation Tools
 *   (C) 2004, 2005 Gerardo Garca Pea
 *   Programmed by Gerardo Garca Pea - Inspired on CEPBA DItools
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301
 *   USA
 *
 *****************************************************************************/

#include<config.h>
#include<ebeif.h>
#include<log.h>
#include<pdiconfig.h>

/*****************************************************************************
 * Private variables of this module
 *
 *   _pdi_redefs
 *     Redefinition list. It allows to iterate quickly on all redefinitions
 *     installed on the instrumented application.
 * 
 *****************************************************************************/

static PDI_INTERCEPT *_pdi_redefs;

/*****************************************************************************
 * static int checkRelink(PDI_ELFOBJ *object, char *func, char *target)
 *
 * Description:
 *   Check relink parameters and there is not collision with another
 *   interposition.
 *
 * Parameters:
 *   object - object where we're going to install on the relink
 *   func   - function we'll relink
 *   target - wrapper name
 *
 * Returns:
 *   0 if all ok, or -1 if error
 *
 *****************************************************************************/

static int checkRelink(PDI_ELFOBJ *object, char *func, char *target)
{
  PDI_INTERCEPT *i;

  /* Check that func and target are not NULL */
  if(!func)
  {
    _pdi_error(THIS, "You cannot use a wildcard in a relink command.");
    return -1;
  }
  if(!target)
  {
    _pdi_error(THIS, "Cannot relink to NULL.");
    return -1;
  }

  /* Check there is no conflict between this relink and */
  /* interpositions currently installed on this object  */
  for(i = object->o_rules; i; i = i->i_next)
  {
    switch(i->i_type)
    {
      case PDI_IT_CALLBACK:
        _pdi_error(THIS, "Function %s::%s() is affected by a callback.",
                   _pdi_ebe_getObjectName(object), func);
        return -1;
      case PDI_IT_RELINK:
        if(!strcmp(func, i->i_symbol))
        {
          _pdi_error(THIS, "Function %s::%s() was relinked.",
                     _pdi_ebe_getObjectName(object), func);
          return -1;
        }
        break;
      case PDI_IT_REDEFINITION:
        break;
      default:
        _pdi_error(THIS,
                   "Inconsistency: I have found an interposition '%s::%s()' of unknown type %d.",
                   _pdi_ebe_getObjectName(object), i->i_symbol, i->i_type);
        return -1;
    }
  }

  /* Now I will search for a redefinition that collides with this relink */
  /* (If found it only warns about it)                                   */
  for(i = _pdi_redefs; i; i = i->i_next)
  {
    if(!strcmp(i->i_symbol, func))
      _pdi_warning(THIS, "Function %s::%s() is affected by a redefinition",
                   _pdi_ebe_getObjectName(object), func);
  }

  return 0;
}

/*****************************************************************************
 * static int checkRedefinition(char *func, char *target)
 *
 * Description:
 *   Check redefinition parameters and if there is not collision with another
 *   interposition.
 *
 * Parameters:
 *   func   - function we'll redefine
 *   target - wrapper name
 *
 * Returns:
 *   0 if all ok, or -1 if error
 *
 *****************************************************************************/

static int checkRedefinition(char *func, char *target)
{
  PDI_INTERCEPT *i;
  PDI_ELFOBJ *o;

  /* Check that symbol is not NULL */
  if(!func)
  {
    _pdi_error(THIS, "You cannot use a wildcard in a redefinition.");
    return -1;
  }
  if(!target)
  {
    _pdi_error(THIS, "Cannot redefine to NULL.");
    return -1;
  }

  /* Check there is no conflict between this redefinition and */
  /* interpositions currently installed on the application    */
  for(o = _pdi_objlist; o; o = o->o_next)
  {
    for(i = o->o_rules; i; i = i->i_next)
    {
      switch(i->i_type)
      {
        case PDI_IT_CALLBACK:
          _pdi_error(THIS, "Function %s() is affected by a callback.", func);
          return -1;
        case PDI_IT_RELINK:
          if(!strcmp(func, i->i_symbol))
          {
            _pdi_error(THIS, "Function %s() was relinked.", func);
            return -1;
          }
          break;
        case PDI_IT_REDEFINITION:
          if(!strcmp(func, i->i_symbol))
          {
            _pdi_error(THIS, "Function %s() was previously redefined.", func);
            return -1;
          }
          break;
        default:
          _pdi_error(THIS,
                     "Inconsistency: I have found an interposition %s() of unknown type %d.",
                     i->i_symbol, i->i_type);
          return -1;
      }
    }
  }

  return 0;
}

/*****************************************************************************
 * static int checkCallback(PDI_ELFOBJ *object, char *func)
 *
 * Description:
 *   Check if it is possible to install this callback
 *
 * Parameters:
 *   object - object where we're going to install a callback
 *   func   - should be NULL
 *
 * Returns:
 *   0 if all ok, or -1 if error
 *
 *****************************************************************************/

static int checkCallback(PDI_ELFOBJ *object, char *func)
{
  PDI_INTERCEPT *i;

  /* 'func' should be NULL */
  if(func)
  {
    _pdi_error(THIS, "You must use a wildcard in a callback.");
    return -1;
  }

  /* Check there is no conflict between this relink and */
  /* interpositions currently installed on this object  */
  for(i = object->o_rules; i; i = i->i_next)
  {
    switch(i->i_type)
    {
      case PDI_IT_CALLBACK:
        _pdi_error(THIS, "Object '%s' is already affected by a callback.",
                   _pdi_ebe_getObjectName(object));
        return -1;
      case PDI_IT_RELINK:
        if(!strcmp(func, i->i_symbol))
        {
          _pdi_error(THIS, "Object '%s' was previously relinked.",
                     _pdi_ebe_getObjectName(object));
          return -1;
        }
        break;
      case PDI_IT_REDEFINITION:
        break;
      default:
        _pdi_error(THIS,
                   "Inconsistency: I have found an interposition '%s::%s()' of unknown type %d.",
                   _pdi_ebe_getObjectName(object), i->i_symbol, i->i_type);
        return -1;
    }
  }

  /* Now I will search for a redefinition that collides with this relink */
  /* (If found it only warns about it)                                   */
  for(i = _pdi_redefs; i; i = i->i_next)
    if(_pdi_arch_symbolIsUsedOrReferenced(object, i->i_symbol) < 0)
    {
      _pdi_warning(THIS, "Object '%s' is affected by a redefinition",
                   _pdi_ebe_getObjectName(object));
      return 0;
    }

  return 0;
}

/*****************************************************************************
 * static
 * PDI_INTERCEPT *getNewInterposition(int type,
 *                                    PDI_ELFOBJ *object, char *symbol,
 *                                    PDI_ELFOBJ *backend, char *wname)
 *
 * Description:
 *   Allocates memory for new interposition information and initializes it.
 *
 * Parameters:
 *   type    - type of interposition (PDI_IT_*)
 *   object  - object where we are going to install this interposition
 *   symbol  - function affected by this interposition (NULL on callback)
 *   backend - backend which will handle this interposition
 *   wname   - function interposed (only NULL in callback)
 *
 * Returns:
 *   A pointer to the new interposition or NULL if error.
 *
 *****************************************************************************/

static PDI_INTERCEPT *getNewInterposition(int type,
                                          PDI_ELFOBJ *object, char *symbol,
                                          PDI_ELFOBJ *backend, char *wname)
{
  PDI_INTERCEPT *i;
  int ret;

  /* CHECK INTERPOSITION PARAMETERS */
  /* check that type is correct */
  switch(type)
  {
    case PDI_IT_RELINK:       ret = checkRelink(object, symbol, wname); break;
    case PDI_IT_REDEFINITION: ret = checkRedefinition(symbol, wname); break;
    case PDI_IT_CALLBACK:     ret = checkCallback(object, symbol); break;
    default:
      _pdi_error(THIS, "Uknown type of interposition (%d).", type);
      return NULL;
  }

  if(ret)
  {
    _pdi_error(THIS, "Cannot install interposition.");
    return NULL;
  }

  /* BUILD INTERPOSITION INFORMATION */
  /* Allocate memory for interposition */
  if((i = malloc(sizeof(PDI_INTERCEPT))) == NULL)
  {
    _pdi_error(THIS, "There is no memory for a PDI_INTERCEPT structure.");
    return NULL;
  }

  if(symbol)
  {
    if((i->i_symbol = malloc(strlen(symbol) + 1)) == NULL)
    {
      free(i);
      _pdi_error(THIS, "There is no memory for a PDI_INTERCEPT structure.");
      return NULL;
    }
    strcpy(i->i_symbol, symbol);
  } else
    i->i_symbol = NULL;

  /* GET A POINTER TO THE INTERPOSED FUNCTION */
  /* Si 'wname' est definido, dependiendo de si 'backend' es un backend o */
  /* un DSO comn pillamos el puntero al wrapper de distintas formas       */
  i->i_wrapper =
    wname
      ? (backend->o_isBackend
          ? _pdi_ebe_getBackendSymbol(backend, wname)
          : _pdi_arch_resolveSymbolInObject(backend, wname))
      : NULL;

  if(wname && !i->i_wrapper)
  {
    if(i->i_symbol) free(i->i_symbol);
    free(i);
    _pdi_error(THIS, "Funcin %s::%s() does not exist.",
               _pdi_ebe_getObjectName(backend), wname);
    return NULL;
  }

  i->i_type    = type;
  i->i_backend = backend;
  
  /* INITIALIZE ARCH-DEPENDANT INFO */
  if(_pdi_arch_newInterposition(object, wname, i))
  {
    _pdi_error(THIS, "An error happened while initializing arch dependant structure part.");
    if(i->i_symbol) free(i->i_symbol);
    free(i);
    return NULL;
  }

  /* ADD THIS INTERPOSITION TO INTERPOSITIONS LIST */
  if(object->o_rules)
    object->o_rules->i_prev = i;
  i->i_next = object->o_rules;
  i->i_prev = NULL;
  object->o_rules = i;

  /* IF IT IS A REDEFINITION, ADD TO THE REDEFINITIONS LIST */
  if(type == PDI_IT_REDEFINITION)
  {
    i->i_rd_prev = NULL;
    i->i_rd_next = _pdi_redefs;
    if(_pdi_redefs) _pdi_redefs->i_rd_prev = i;
    _pdi_redefs = i;
  }

  return i;
}

/*****************************************************************************
 * static void freeInterposition(PDI_ELFOBJ *eo, PDI_INTERCEPT *i)
 *
 * Description::
 *   Free memory used by an interposition
 *
 * Parameters:
 *   i - interposition to destroy
 *
 * Returns:
 *   Nothing.
 *
 *****************************************************************************/

static void freeInterposition(PDI_ELFOBJ *eo, PDI_INTERCEPT *i)
{
  /* Unlink the structure (also from redefinition list if it applies) */
  if(eo->o_rules == i)
    eo->o_rules = i->i_next;
  if(i->i_prev) i->i_prev->i_next = i->i_next;
  if(i->i_next) i->i_next->i_prev = i->i_prev;

  if(i->i_type == PDI_IT_REDEFINITION)
  {
    if(_pdi_redefs == i)
      _pdi_redefs = i->i_rd_next;
    if(i->i_rd_prev) i->i_rd_prev->i_rd_next = i->i_rd_next;
    if(i->i_rd_next) i->i_rd_next->i_rd_prev = i->i_rd_prev;
  }

  /* free arch dependant structure */
  _pdi_arch_freeInterposition(eo, i);

  /* free memory used by this interposition */
  if(i->i_symbol) free(i->i_symbol);
  free(i);
}

/*****************************************************************************
 * PDI_INTERCEPT *_pdi_ebe_searchInterposition(PDI_ELFOBJ *eo, char *func)
 *
 * Description:
 *   Search for an interposition on function 'func' that belongs to the object
 *   'eo'.
 *
 * Parameters:
 *   eo   - object
 *   func - function name (NULL is valid if searching for a callback)
 *
 * Returns:
 *   A pointer to the rule definition or NULL if there isn't a interposition on
 *   this function/object.
 *
 *****************************************************************************/

PDI_INTERCEPT *_pdi_ebe_searchInterposition(PDI_ELFOBJ *eo, char *func)
{
  PDI_INTERCEPT *i;

  for(i = eo->o_rules; i; i = i->i_next)
    if(i->i_type == PDI_IT_CALLBACK || (func && !strcmp(i->i_symbol, func)))
      return i;

  return NULL;
}

/*****************************************************************************
 * static int relink(PDI_ELFOBJ *eo, char *func,
 *                   PDI_ELFOBJ *backend, char *wname)
 *
 * Description:
 *   Relink a function to a wrapper.
 *
 * Parameters:
 *   object  - object on we are going to install this relink
 *   func    - function relinked
 *   backend - backend which will receive this relink
 *   wname   - function interposed
 *
 * Returns:
 *   0 if ok, -1 if error
 *
 *****************************************************************************/

static int relink(PDI_ELFOBJ *object, char *func,
                  PDI_ELFOBJ *backend, char *wname)
{
  PDI_INTERCEPT *i;
  
  /* Check parameters */
  if(!wname)
  {
    _pdi_error(THIS, "Cannot relink calls from '%s' to '%s()' with NULL.",
               _pdi_ebe_getObjectName(object), func);
    return -1;
  }

  /* Build interposition info and install this relink */
  if((i = getNewInterposition(PDI_IT_RELINK,
                              object, func,
                              backend, wname)) == NULL)
  {
    _pdi_error(THIS, "Cannot alloc a new interposition for a relink on '%s'.",
               func);
    return -1;
  }

  if(_pdi_arch_relink(object, i))
  {
    freeInterposition(object, i);
    _pdi_error(THIS, "Cannot relink calls from '%s' to '%s()'.",
               _pdi_ebe_getObjectName(object), func);
    return -1;
  }
  
  return 0;
}

/*****************************************************************************
 * static int undoRelink(PDI_ELFOBJ *eo, PDI_INTERCEPT *i)
 *
 * Description:
 *   Undo relink 'i' on object 'eo'.
 *
 * Parameters:
 *   eo - object currently relinked
 *   i  - relink information
 *
 * Returns:
 *   0 if ok, -1 if error
 *
 *****************************************************************************/

static int undoRelink(PDI_ELFOBJ *eo, PDI_INTERCEPT *i)
{
  return _pdi_arch_undoRelink(eo, i);
}

/*****************************************************************************
 * static int redefine(PDI_ELFOBJ *object, char *func,
 *                     PDI_ELFOBJ *backend, char *redef_func)
 *
 * Description:
 *   Redefine function 'func' implemented in 'object'. A redefinition is like a
 *   relink but it affects all objects. It also changes the pointer to the
 *   function in symbol table so in future 'ld-linux.so' will resolve all
 *   references to our new definition.
 *
 * Parameters:
 *   object     - object which contains the function to redefine
 *   func       - name of this function
 *   backend    - object with the new definition
 *   redef_func - new definition name
 *
 * Returns:
 *   0 if ok, -1 if error
 *
 *****************************************************************************/

static int redefine(PDI_ELFOBJ *object, char *func,
                    PDI_ELFOBJ *backend, char *redef_func)
{
#if PDI_REDEFINES_ENABLED
  PDI_INTERCEPT *r;
  
  if(!redef_func)
  {
    _pdi_error(THIS, "Cannot redefine function '%s::%s()' with NULL.",
               _pdi_ebe_getObjectName(object), func);
    return -1;
  }

  if((r = getNewInterposition(PDI_IT_REDEFINITION,
                              object, func,
                              backend, redef_func)) == NULL)
  {
    _pdi_error(THIS, "Cannot alloc a new interposition to redefine '%s'.",
               func);
    return -1;
  }

  if(_pdi_arch_redefine(object, r))
  {
    freeInterposition(object, r);
    _pdi_error(THIS, "Cannot redefine '%s::%s()'.",
               _pdi_ebe_getObjectName(object), func);
    return -1;
  }
  
  return 0;
#else
  _pdi_error(THIS, "Redefinitions are disabled in this version of pDI-Tools.");
  return -1;
#endif
}

/*****************************************************************************
 * static int undoRedefine(PDI_ELFOBJ *eo, PDI_INTERCEPT *i)
 *
 * Description:
 *   Undo redefinition 'i' installed on 'eo'.
 *
 * Parameters:
 *   eo - object affected by 'i'
 *   i  - redefinition to uninstall
 *
 * Returns:
 *   0 if ok, -1 if error
 *
 *****************************************************************************/

static int undoRedefine(PDI_ELFOBJ *eo, PDI_INTERCEPT *i)
{
#if PDI_REDEFINES_ENABLED
  return _pdi_arch_undoRedefine(eo, i);
#else
  _pdi_error(THIS, "Redefinitions are disabled in this version of pDI-Tools.");
  return -1;
#endif
}

/*****************************************************************************
 * static int callback(PDI_ELFOBJ *object,
 *                     PDI_ELFOBJ *backend,
 *                     char *alt_genwrapper)
 *
 * Description:
 *   Install a callback interpositin on object 'object'.
 *
 * Parameters:
 *   object         - object
 *   backend        - backend which will control this callback
 *   alt_genwrapper - alternative callback handler. Usually this is NULL.
 *
 * Returns:
 *   0 if ok, -1 if error
 *
 *****************************************************************************/

static int callback(PDI_ELFOBJ *object,
                    PDI_ELFOBJ *backend,
                    char *alt_genwrapper)
{
#if PDI_CALLBACKS_ENABLED
  PDI_INTERCEPT *r;
  
  /* Cannot be interpositions on this object */
  if(object->o_rules)
  {
    _pdi_error(THIS,
               "Cannot install a callback on '%s' because there are other "
               "interpositions that would be in conflict with it.",
               _pdi_ebe_getObjectName(object));
    return -1;
  }

  /* If the user has given an alternative "generic wrapper", and the */
  /* configuration doesn't allow that, we give an error message      */
  if(alt_genwrapper && !PDICFG.cb_allow_handler)
  {
    _pdi_error(THIS,
               "Enable flag 'cb_allow_handler' to use an "
               "alternative generic wrapper on '%s'.",
               _pdi_ebe_getObjectName(object));
    return -1;
  }

  /* Check that this backend implements callback handler functions */
  if(!backend->o_beCallbackRequired)
  {
    _pdi_error(THIS,
               "Backend '%s' has not implemented function " PDI_STR_BE_FUNC_CB_REQ "().",
               _pdi_ebe_getObjectName(backend));
    return -1;
  }

  /* Check that at least exists a pre or post function */
  if(!backend->o_bePreEventCallback && !backend->o_bePostEventCallback)
  {
    _pdi_error(THIS,
               "Backend '%s' should have at least one (post or pre) event handler.",
               _pdi_ebe_getObjectName(backend));
    return -1;
  }

  /* Create a new interposition */
  if((r = getNewInterposition(PDI_IT_CALLBACK,
                              object, NULL,
                              backend, alt_genwrapper)) == NULL)
  {
    _pdi_error(THIS, "Cannot alloc memory for a new callback on '%s'.",
               _pdi_ebe_getObjectName(object));
    return -1;
  }

  /* Install callback */
  if(_pdi_arch_callback(object, r))
  {
    _pdi_error(THIS, "Cannot install a callback on '%s'.",
               _pdi_ebe_getObjectName(object));
    return -1;
  }
  
  return 0;
#else
  _pdi_error(THIS, "Callbacks are disabled in this version of pDI-Tools.");
  return -1;
#endif
}

/*****************************************************************************
 * static int undoCallback(PDI_ELFOBJ *eo, PDI_INTERCEPT *i)
 *
 * Description:
 *   Uninstall callback 'i' from object 'eo'.
 *
 * Parameters:
 *   eo - object
 *   i  - interposition
 *
 * Returns:
 *   0 if ok, -1 if error
 *
 *****************************************************************************/

static int undoCallback(PDI_ELFOBJ *eo, PDI_INTERCEPT *i)
{
#if PDI_CALLBACKS_ENABLED
  return _pdi_arch_undoCallback(eo, i);
#else
  _pdi_error(THIS, "Callbacks are disabled in this version of pDI-Tools.");
  return -1;
#endif
}

/*****************************************************************************
 * int _pdi_ebe_installInterposition(int type,
 *                                   PDI_ELFOBJ *object, char *func,
 *                                   PDI_ELFOBJ *backend, void *wrapper_func)
 *
 * Description:
 *   Install a interpostion of type 'type' on object 'object'.
 *
 * Parameters:
 *   type         - interposition type (PDI_IT_*)
 *   object       - object we are working on
 *   symbol       - symbol we are installing an interposition on
 *                  (NULL if we're installing a callback)
 *   backend      - object will receive the interception
 *   wrapper_func - function will receive this interception
 *                  (never NULL but if callback)
 *
 * Returns:
 *   0 if ok, -1 if error
 *
 *****************************************************************************/

int _pdi_ebe_installInterposition(int type,
                                  PDI_ELFOBJ *object, char *func,
                                  PDI_ELFOBJ *backend, char *wname)
{
  /* Check that this interposition is ok with config */
  if((PDICFG.donttouch_backends && object->o_isBackend)
  || (PDICFG.donttouch_pdi && object->o_alias && !strcmp(object->o_alias, PDI_ALIAS_PDI)))
  {
    _pdi_error(THIS, "Cannot install a %s on %s.",
               type == PDI_IT_RELINK         ? "relink"
               : type == PDI_IT_REDEFINITION ? "redefine"
               : type == PDI_IT_CALLBACK     ? "callback"
               : "'unknown type interposition'",
               (PDICFG.donttouch_backends && object->o_isBackend)
                 ? "a backend"
                 : PACKAGE_NAME);
    return -1;
  }

  /* Check that 'backend' is really a backend... */
  if(!backend->o_isBackend && !PDICFG.allow_lib_as_be)
  {
    _pdi_error(THIS, "Objeto '%s' is not a backend.",
               _pdi_ebe_getObjectName(backend));
    return -1;
  }

  switch(type)
  {
    case PDI_IT_RELINK:
      if(relink(object, func, backend, wname))
      {
        _pdi_error(THIS, "Error while installing relink interposition.");
        return -1;
      }
      break;
      
    case PDI_IT_REDEFINITION:
      if(redefine(object, func, backend, wname))
      {
        _pdi_error(THIS, "Error while installing redefinition interposition.");
        return -1;
      }
      break;

    case PDI_IT_CALLBACK:
      if(callback(object, backend, wname))
      {
        _pdi_error(THIS, "Error while installing callback interposition.");
        return -1;
      }
      break;
      
    default:
      _pdi_error(THIS, "Unknowm interposition type '%d'.", type);
      return -1;
  }

  return 0;
}

/*****************************************************************************
 * int _pdi_ebe_uninstallInterposition(PDI_ELFOBJ *eo, PDI_INTERCEPT *i)
 *
 * Description:
 *   Uninstall interposition 'i' from object 'eo'.
 *
 * Parameters:
 *   eo - object where is installed this interposition
 *   i  - interposition to uninstall
 *
 * Returns:
 *   0 if ok, -1 if error
 *
 *****************************************************************************/

int _pdi_ebe_uninstallInterposition(PDI_ELFOBJ *eo, PDI_INTERCEPT *i)
{
  int ret;

  /* deshacemos los efectos de la regla */
  switch(i->i_type)
  {
    case PDI_IT_RELINK:       ret = undoRelink(eo, i);   break;
    case PDI_IT_REDEFINITION: ret = undoRedefine(eo, i); break;
    case PDI_IT_CALLBACK:     ret = undoCallback(eo, i); break;
    default:
      _pdi_error(THIS, "Unknowm interposition type '%d'.", i->i_type);
      return -1;
  }

  /* desencadenamos la regla, la quitamos del objeto y liberamos memoria */
  freeInterposition(eo, i);

  return ret;
}

/*****************************************************************************
 * int _pdi_ebe_uninstallInterpositions(PDI_ELFOBJ *eo)
 *
 * Description:
 *   Uninstall all interposition in object 'eo'.
 *
 * Parameters:
 *   eo - object
 *
 * Returns:
 *   0 if ok, -1 if error
 *
 *****************************************************************************/

int _pdi_ebe_uninstallInterpositions(PDI_ELFOBJ *eo)
{
  while(eo->o_rules)
    if(_pdi_ebe_uninstallInterposition(eo, eo->o_rules))
    {
      _pdi_error(THIS, "Cannot uninstall interposition.");
      return -1;
    }

  return 0;
}

/*****************************************************************************
 * int _pdi_ebe_uninstallBackendInterpositions(PDI_ELFOBJ *backend)
 *
 * Description:
 *   Uninstall all interpositions that references backend 'backend'.
 *
 * Parameters:
 *   backend - backend
 *
 * Returns:
 *   0 if ok, -1 if error
 *
 *****************************************************************************/

int _pdi_ebe_uninstallBackendInterpositions(PDI_ELFOBJ *backend)
{
  PDI_ELFOBJ *eo;
  PDI_INTERCEPT *i, *i_next;

  for(eo = _pdi_objlist; eo; eo = eo->o_next)
    for(i = eo->o_rules; i; i = i_next)
    {
      i_next = i->i_next;
      if(i->i_backend == backend)
        if(_pdi_ebe_uninstallInterposition(eo, i))
        {
          _pdi_error(THIS,
                     "An error happened while uninstalling an interposition related to backend '%s'.",
                     _pdi_ebe_getObjectName(backend));
          return -1;
        }
    }

  return 0;
}

/*****************************************************************************
 * int _pdi_ebe_uninstallAllInterpositions(void)
 *
 * Description:
 *   Uninstall all interpositions on instrumented application.
 *
 * Parameters:
 *   This function doesn't take any parameter
 *
 * Returns:
 *   0 ok, -1 if error
 *
 *****************************************************************************/

int _pdi_ebe_uninstallAllInterpositions(void)
{
  PDI_ELFOBJ *eo;

  for(eo = _pdi_objlist; eo; eo = eo->o_next)
    if(_pdi_ebe_uninstallInterpositions(eo))
    {
      _pdi_error(THIS,
                 "Cannot uninstall all interpositions on object '%s'.",
                 _pdi_ebe_getObjectName(eo));
      return -1;
    }

  return 0;
}

/*****************************************************************************
 * int _pdi_ebe_protectBackend(PDI_ELFOBJ *backend)
 *
 * Description:
 *   Resolve symbols on this object so it will not be affected by any
 *   interposition installed on this application.
 *
 * Parameters:
 *   backend - object to protect
 *
 * Returns:
 *   0 ok, -1 if error
 *
 *****************************************************************************/

int _pdi_ebe_protectBackend(PDI_ELFOBJ *backend)
{
  PDI_INTERCEPT *i;

  for(i = _pdi_redefs; i; i = i->i_rd_next)
    if(_pdi_arch_deactivateRedefinition(i, backend))
    {
      _pdi_error(THIS, "Cannot deactivate redefinition '%s::%s()'.",
                 _pdi_ebe_getObjectName(backend), i->i_symbol);
      return -1;
    }

  return 0;
}

