/*****************************************************************************
 * $Id: pdiconfig.c,v 1.2 2005/08/23 11:39:26 killabyte Exp $
 *
 * This file implements API used to get pDI-Tools configuration from several
 * sources (config files, environment variables, LDAP directories, SQL
 * databases, etc).
 *
 * ---------------------------------------------------------------------------
 * pDI-Tools - portable Dynamic Instrumentation Tools
 *   (C) 2004, 2005 Gerardo Garca Pea
 *   Programmed by Gerardo Garca Pea - Inspired on CEPBA DItools
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301
 *   USA
 *
 *****************************************************************************/

#include<config.h>
#include<ebeif.h>
#include<pdiconfig.h>
#include<log.h>
#include<files.h>

/*****************************************************************************
 * _pdi_configuration
 *
 *   Estructura, de visibilidad global con la configuracin de pDI
 *   Se recomienda acceder a ella mediante el alias PDICFG.
 *
 *****************************************************************************/

PDI_CONFIG _pdi_configuration;

/*****************************************************************************
 * CONFIGPARAM
 *
 *   Estructura que define el aspecto de un parmetro de configuracin de
 *   pDI-Tools.
 *
 *****************************************************************************/

typedef struct _tag_CONFIGPARAM {
  char   *name;
  char   *env;
  int    type;
#define PDI_CFGPARAM_STRING 0
#define PDI_CFGPARAM_INT    1
#define PDI_CFGPARAM_BOOL   2
  void   *pointer;
  int    (*func)(struct _tag_CONFIGPARAM *cp, char *sValue, int iValue);
  int    counter;
} CONFIGPARAM;

/*****************************************************************************
 * pushBackendConfigFile
 *
 *   Funcin que aade a la lista de ficheros de configuracin para cargar un
 *   fichero nuevo. Tiene diferentes comportamientos segn param sea 'runtime'
 *   o 'config'. El parmetro 'runtime' slo se puede declarar una vez,
 *   mientras que 'config' infinitas veces.
 *
 *****************************************************************************/

static int addSearchPath(CONFIGPARAM *param, char *sValue, int iValue);
static int pushBackendConfigFile(CONFIGPARAM *param, char *sValue, int iValue);
static int resetParam(CONFIGPARAM *param, char *sValue, int iValue);
static int setDebugMode(CONFIGPARAM *param, char *sValue, int iValue);

/*****************************************************************************
 * configparams
 *
 *   Parmetros y variables de entorno que reconoce pDI
 *
 *   NOTA: Usamos as macros ENV, CP y MJOIN para hacer el listado ms compacto
 *         y con la esperanza de que as sea ms legible.
 *
 *****************************************************************************/

#define ENV(x)          MJOIN(PDI_ENV_,x)
#define CP(x)           MJOIN(PDI_CFGPARAM_,x)
#define MJOIN(x,y)      x##y

static CONFIGPARAM configparams[] = {
  { "allow_lib_as_be",    NULL,             CP(BOOL),   &(PDICFG.allow_lib_as_be),    NULL },
  { "be_path",            NULL,             CP(STRING), NULL,                         addSearchPath         },
  { "becfg_path",         NULL,             CP(STRING), NULL,                         addSearchPath         },
  { "cb_allow_handler",   NULL,             CP(BOOL),   &(PDICFG.cb_allow_handler),   NULL },
  { "cb_max_stubs",       NULL,             CP(INT),    &(PDICFG.cb_max_stubs),       NULL },
  { "cb_stack_size",      NULL,             CP(INT),    &(PDICFG.cb_stack_size),      NULL },
  { "config",             ENV(CONFIGFILE),  CP(STRING), NULL,                         pushBackendConfigFile },
  { "debug",              ENV(DEBUG),       CP(BOOL),   &(PDICFG.debug),              setDebugMode },
  { "donttouch_backends", NULL,             CP(BOOL),   &(PDICFG.donttouch_backends), NULL },
  { "donttouch_pdi",      NULL,             CP(BOOL),   &(PDICFG.donttouch_pdi),      NULL },
  { "lib_path",           NULL,             CP(STRING), NULL,                         addSearchPath         },
  { "logfile",            ENV(LOGFILE),     CP(STRING), &(PDICFG.log_filename),       NULL },
  { "max_objects",        NULL,             CP(INT),    &(PDICFG.max_objects),        NULL },
  { "max_threads",        NULL,             CP(INT),    &(PDICFG.max_threads),        NULL },
  { "no_check_on_config", NULL,             CP(BOOL),   &(PDICFG.no_check_on_config), NULL },
  { "num_threads",        NULL,             CP(INT),    &(PDICFG.num_threads),        NULL },
  { "runtime",            ENV(RUNTIMEFILE), CP(STRING), NULL,                         pushBackendConfigFile },
  { "verbose",            NULL,             CP(INT),    &(PDICFG.verbose),            NULL },
  { "reset_runtime",      NULL,             0,          NULL,                         resetParam },
  { "reset_config",       NULL,             0,          NULL,                         resetParam },
  { "reset_be_path",      NULL,             0,          NULL,                         resetParam },
  { "reset_becfg_path",   NULL,             0,          NULL,                         resetParam },
  { "reset_lib_path",     NULL,             0,          NULL,                         resetParam },
  { NULL,                 NULL,             0,          NULL,                         NULL }
};

/*****************************************************************************
 * static int insertPath(char ***stack, int *nstack, char *path,
 *                       int before, char sep)
 *
 * Descripcin:
 *   Inserta un path nuevo en la lista '*stack' de '*nstack' entradas.
 *
 * Parmetros:
 *   stack  - pila donde insertar al principio
 *   nstack - altura de la pila
 *   path   - nuevo path
 *   before - si es distinto de 0 se aade al principio de la tabla, si no se
 *            aade al final.
 *   sep    - en caso de que se indiquen varios path en un nico string, aqu
 *            indicamos el carcter separador (generalmente ':'). Si no se
 *            desea esta funcionalidad, se debe asignar '\0'.
 *
 * Devuelve:
 *   0 si el atributo se proces correctamente, -1 en caso de que hubiese algn
 *   error.
 *
 *****************************************************************************/

static int insertPath(char ***stack, int *nstack, char *path,
                      int before, char sep)
{
  int nent, i;
  char *s1, *s2, **tstack;

  /* Contamos cuantas entradas vamos a aadir */
  if(sep)
  {
    s1 = s2 = path;
    for(nent = 0; *s2; s2++)
      if(*s2 == ':')
      {
        if(s1 < s2)
          nent++;
        s1 = s2+1;
      }
    if(s1 < s2)
      nent++;
  } else
    nent = 1;

  if(strlen(path) == 0 || !nent)
  {
    _pdi_warning(THIS, "A void path was passed.");
    return 0;
  }

  /* Reservamos memoria para las nuevas entradas */
  if((tstack = realloc(*stack, sizeof(char *) * (*nstack + nent + 1))) == NULL)
  {
    _pdi_error(THIS, "There is no memory for %d new entries.", nent);
    return -1;
  }
  *stack = tstack;

  /* Ponemos NULL en la ltima entrada */
  (*stack)[*nstack + nent] = NULL;

  /* dependiendo de 'before' movemos las entradas y escogemos */
  /* el valor de inicial del ndice 'i'                       */
  if(before)
  {
    /* Movemos a atrs las entradas antiguas */
    for(i = (*nstack)-1; i >= 0; i--)
      (*stack)[i + nent] = (*stack)[i];
    i = 0;
  } else
    i = *nstack;

  /* Aadimos las nuevas entradas */
  for(s1 = s2 = path; *s2; s2++)
    if(*s2 == sep)
    {
      if(s2 - s1 > 0)
      {
        if(((*stack)[i] = malloc(s2 - s1 + 1)) == NULL)
        {
          _pdi_error(THIS, "There is no memory for a buffer to copu the path '%s'.", s1);
          return -1;
        }
        memcpy((*stack)[i], s1, s2 - s1);
        (*stack)[i][s2 - s1] = '\0';
        i++;
      }
      s1 = s2 + 1;
    }

  if(s1 < s2)
  {
    if(((*stack)[i] = malloc(s2 - s1 + 1)) == NULL)
    {
      _pdi_error(THIS, "There is no mem to copy directory '%s'.", s1);
      return -1;
    }
    memcpy((*stack)[i], s1, s2 - s1);
    (*stack)[i][s2 - s1] = '\0';
  }
  *nstack += nent;

  return 0;
}

/*****************************************************************************
 * static void resetStack(char ***stack, int *nstack, int first)
 *
 * Descripcin:
 *   Elimina todas las entradas de una lista de ficheros.
 *
 * Parmetros:
 *   stack  - puntero al puntero a la lista a resetear
 *   nstack - puntero al entero con la altura de la pila
 *
 * Devuelve:
 *   -nada-
 *
 *****************************************************************************/

static void resetStack(char ***stack, int *nstack, int first)
{
  char **p;

  if(*stack)
  {
    for(p = (*stack + first); *p; p++)
    {
      free(*p);
      *nstack = *nstack - 1;
    }
    free(*stack);
    *stack = NULL;
  }
}

/*****************************************************************************
 * static CONFIGPARAM *searchConfigParam(char *tag)
 *
 * Descripcin:
 *   Busca la informacin del parmetro de nombre 'tag'
 *
 * Parmetros:
 *   tag - nombre del parmetro a buscar
 *
 * Devuelve:
 *   Un puntero a la informacin del parmetro o NULL en caso de que no
 *   existiese.
 *
 *****************************************************************************/

static CONFIGPARAM *searchConfigParam(char *tag)
{
  CONFIGPARAM *cp;

  /* buscamos el parmetro. en caso de fallo salimos con error */
  for(cp = configparams; cp->name && strcmp(cp->name, tag); cp++)
    ;
  return cp->name ? cp : NULL;
}

/*****************************************************************************
 * static char *setConfigParam(CONFIGPARAM *cp, char *value)
 *
 * Descripcin:
 *   Procesa el parmetro 'cp' asignndole el valor 'value'.
 *
 * Parmetros:
 *   cp    - parmetro al que asignamos
 *   value - valor que asignamos.
 *
 * Devuelve:
 *   NULL en caso de xito, o un puntero a la descripcin del error.
 *
 *****************************************************************************/

static char *setConfigParam(CONFIGPARAM *cp, char *value)
{
  char *p;
  int x;

  if(!cp)
    return "unknown parameter";

  switch(cp->type)
  {
    case PDI_CFGPARAM_STRING:
      if(cp->func && (cp->func)(cp, value, 0))
          return "error processing parameter";
      if(cp->pointer)
      {
        /* Si ya hay un valor asignado, liberamos la memoria que ocupa */
        if(*((char **) cp->pointer) != NULL)
          free(*((char **) cp->pointer));

        /* asignamos el nuevo valor */
        if(value)
        {
          /* si 'value' no es NULL, reservamos memoria y copiamos el valor */
          if((*((char **) cp->pointer) = malloc(strlen(value) + 1)) == NULL)
          {
            _pdi_error(THIS, "No memory while loading '%s' contents.",
                       cp->name);
            exit(1);
          }
          strcpy(*((char **) cp->pointer), value);
        } else
          *((char **) cp->pointer) = NULL;
      }
      break;
    case PDI_CFGPARAM_INT:
      /* Comprobamos que sea un nmero */
      p = value;
      if(*p == '-' || *p == '+') p++;
      for(; *p >= '0' && *p <= '9'; p++)
        ;
      if(*p)
        return "a numeric value is expected";

      /* recogemos su valor */
      sscanf(value, "%d", &x);

      if(cp->func && (cp->func)(cp, NULL, x))
          return "error processing parameter";
      if(cp->pointer)
        *((int *) cp->pointer) = x;
      break;
    case PDI_CFGPARAM_BOOL:
      x = !strcasecmp(value, "yes")      || !strcasecmp(value, "si")
       || !strcasecmp(value, "s")       || !strcasecmp(value, "on")
       || !strcasecmp(value, "true")     || !strcasecmp(value, "cierto")
       || !strcasecmp(value, "activado") || !strcasecmp(value, "1");

      if(!x
      && strcasecmp(value, "no")          && strcasecmp(value, "off")
      && strcasecmp(value, "false")       && strcasecmp(value, "falso")
      && strcasecmp(value, "desactivado") && strcasecmp(value, "0"))
        return "'true' or 'false' values are expected";

      if(cp->func && (cp->func)(cp, NULL, x))
          return "error processing parameter";
      if(cp->pointer)
        *((int *) cp->pointer) = x;
      break;

    default:
      _pdi_error(THIS, "Unknown parameter type (%d).", cp->type);
      exit(1);
  }

  cp->counter++;
  
  return NULL;
}

/*****************************************************************************
 * static int setDebugMode(CONFIGPARAM *cp, char *sValue, int iValue)
 *
 * Description:
 *   This function changes the debug flag.
 *
 * Parameters:
 *   cp     - info about parameter (this is a pointer to the entry which
 *            describes the parameter in 'configparams')
 *   sValue - it's value as string
 *   iValue - it's value as integer
 *
 * Returns:
 *   0 if this attrib was correctly processed, -1 otherwise.
 *
 *****************************************************************************/

static int setDebugMode(CONFIGPARAM *cp, char *sValue, int iValue)
{
  if(iValue)
    PDICFG.verbose = 3;

  return 0;
}

/*****************************************************************************
 * static int pushBackendConfigFile(CONFIGPARAM *cp, char *sValue, int iValue)
 *
 * Descripcin:
 *   Funcin llamada por el procesador de parmetros cuando se encuentra un
 *   atributo 'config' o 'runtime'. Su funcin es aadir un nuevo fichero de
 *   configuracin de backend a la lista de ficheros de configuracin.
 *
 * Parmetros:
 *   cp     - informacin del parmetro recogido (la entrada que lo describe en
 *            la tabla 'configparams')
 *   sValue - Su valor en forma de string
 *   iValue - Su valor en forma de nmero entero
 *
 * Devuelve:
 *   0 si el atributo se proces correctamente, -1 en caso de que hubiese algn
 *   error.
 *
 *****************************************************************************/

static int pushBackendConfigFile(CONFIGPARAM *cp, char *sValue, int iValue)
{
  /* Comprobamos, si la directiva es runtime, que no se haya indicado antes */
  if(!strcmp(cp->name, "runtime") && cp->counter > 0)
  {
    _pdi_error(__FILE__, NULL,
               "Parameter 'runtime' is defined too many times.");
    return -1;
  }

  /* si el valor es NULO o demasiado corto, salimos silenciosamente */
  if(!sValue || !strlen(sValue))
  {
    if(!strcmp(cp->name, "runtime"))
      cp->counter--;
    return 0;
  }

  /* Insertamos el nuevo fichero a la lista. Ntese que en el caso */
  /* del parmetro 'runtime' insertamos al principio de todo.      */
  return insertPath(&(PDICFG.beconfig_files),
                    &(PDICFG.n_beconfig_files),
                    sValue,
                    !strcmp(cp->name, "runtime"),
                    '\0');
}

/*****************************************************************************
 * static int addSearchPath(CONFIGPARAM *cp, char *sValue, int iValue)
 *
 * Descripcin:
 *   Funcin encargada de aadir paths de bsqueda nuevos a los atributos
 *   'be_path' y 'becfg_path'. Esta funcin es llamada por el procesador de
 *   parmetros.
 *
 * Parmetros:
 *   cp     - informacin del parmetro recogido (la entrada que lo describe en
 *            la tabla 'configparams')
 *   sValue - Su valor en forma de string
 *   iValue - Su valor en forma de nmero entero
 *
 * Devuelve:
 *   0 si el atributo se proces correctamente, -1 en caso de que hubiese algn
 *   error.
 *
 *****************************************************************************/

static int addSearchPath(CONFIGPARAM *cp, char *sValue, int iValue)
{
  char ***stack;
  int *nstack;

  /* Averiguamos en donde tenemos que aadir el path */
  if(!strcmp(cp->name, "be_path"))
  {
    stack = &(PDICFG.be_path);
    nstack = &(PDICFG.n_be_path);
  } else
  if(!strcmp(cp->name, "becfg_path"))
  {
    stack = &(PDICFG.becfg_path);
    nstack = &(PDICFG.n_becfg_path);
  } else
  if(!strcmp(cp->name, "lib_path"))
  {
    stack = &(PDICFG.lib_path);
    nstack = &(PDICFG.n_lib_path);
    if(strcmp(sValue, "%LD_LIBRARY_PATH%"))
      if((sValue = getenv("LD_LIBRARY_PATH")) == NULL)
      {
        _pdi_warning(THIS, "Environment variable LD_LIBRARY_PATH is not defined.");
        return 0;
      }
  } else {
    _pdi_error(THIS, "I don't know how to add a path to parameter '%s'.", cp->name);
    return -1;
  }

  /* Insertamos el nuevo path */
  return insertPath(stack, nstack, sValue, 1, ':');
}

/*****************************************************************************
 * static int resetParam(CONFIGPARAM *cp, char *sValue, int iValue)
 *
 * Descripcin:
 *   Funcin llamada por el procesador de atributos que permite ejecutar el
 *   reset que implican los atributos/comando "reset_be_path",
 *   "reset_becfg_path", "reset_config" y "reset_runtime".
 *
 * Parmetros:
 *   cp     - informacin del parmetro recogido (la entrada que lo describe en
 *            la tabla 'configparams')
 *   sValue - Su valor en forma de string
 *   iValue - Su valor en forma de nmero entero
 *
 * Devuelve:
 *   0 si el atributo se proces correctamente, -1 en caso de que hubiese algn
 *   error.
 *
 *****************************************************************************/

static int resetParam(CONFIGPARAM *cp, char *sValue, int iValue)
{
  CONFIGPARAM *runtime = searchConfigParam("runtime");

  if(!strcmp(cp->name, "reset_be_path"))
    resetStack(&(PDICFG.be_path), &(PDICFG.n_be_path), 0);
  else
  if(!strcmp(cp->name, "reset_becfg_path"))
    resetStack(&(PDICFG.becfg_path), &(PDICFG.n_becfg_path), 0);
  if(!strcmp(cp->name, "reset_lib_path"))
    resetStack(&(PDICFG.lib_path), &(PDICFG.n_lib_path), 0);
  else
  if(!strcmp(cp->name, "reset_config"))
  {
    resetStack(&(PDICFG.beconfig_files),
               &(PDICFG.n_beconfig_files),
               runtime->counter > 0 ? 1 : 0);
  } else
  if(!strcmp(cp->name, "reset_runtime"))
  {
    char **p;

    if(runtime->counter == 0)
    {
      _pdi_warning(THIS, "Parameter 'runtime' is not defined yet.");
      return 0;
    }
    runtime->counter = 0;
    free(PDICFG.beconfig_files[0]);
    for(p = PDICFG.beconfig_files; *p; p++)
      p[0] = p[1];
    PDICFG.n_beconfig_files--;
    if(!PDICFG.n_beconfig_files)
      resetStack(&(PDICFG.beconfig_files), &(PDICFG.n_beconfig_files), 0);
  } else {
    _pdi_error(THIS, "I don't know how to reset paths in parameter '%s'.", cp->name);
    return -1;
  }

  return 0;
}

/*****************************************************************************
 * static char *findConfigurationFile(void)
 *
 * Descripcin:
 *   Busca el primer archivo de configuracin a procesar. Estudia las variables
 *   de entorno y recorre una serie de directorios predefinidos en busca del
 *   archivo de configuracin inicial.
 *
 * Parmetros:
 *   -ninguno-
 *
 * Devuelve:
 *   Un puntero al nombre del fichero en caso de xito o NULL en caso de que
 *   no se encontrara ningn fichero de configuracin.
 *
 *****************************************************************************/

static char *findConfigurationFile(void)
{
  char *configPath[] = {
    ".",                  /* El directorio actual es una buena propuesta    */
    "~/etc",              /* algunos carcas pueden tener un /etc en su home */
    "~/etc/pdi",
    "~/etc/pDI",
    "~/etc/pdi-tools",
    "~/etc/pDI-Tools",
    PDI_PATH_ETC,         /* durante la compilacin se indica un lugar...   */
    "/etc",               /* algn da pDI vendr de serie en las distros   */
    "/etc/pdi",
    "/etc/pDI",
    "/etc/pdi-tools",
    "/etc/pDI-Tools",
    NULL
  };
  char fname[PATH_MAX], *t;
  char *ret;

  /* Comprobamos si est la variable de entorno 'DI_CFG_FILE' */
  if((t = getenv(PDI_ENV_PDICFGFILE)) != NULL)
  {
    /* si esta definida, pero vacia, simplemente salimos devolviendo NULL. */
    /* En caso contrario, nos copiamos el valor de la variable de entorno  */
    if(strlen(t) == 0 || !_pdi_fileIsRegular(t))
      t = NULL;
  } else
    if(_pdi_findFile(configPath, PDI_CFG_DEF_FNAME, fname, 1))
      t = NULL;
    else
      t = fname;

  if(t)
  {
    if((ret = malloc(strlen(t)+1)) == NULL)
    {
      _pdi_error(THIS, "Cannot alloc memory for path '%s'.", t);
      exit(1);
    }

    strcpy(ret, t);
  } else
    ret = NULL;

  return ret;
}

/*****************************************************************************
 * static int loadConfigurationFile(char *fname, char *sect)
 *
 * Descripcin:
 *   Procesa la seccin 'sect' del fichero de configuracin 'fname'. Si se
 *   encuentra una directiva 'include' procesa recursivamente el fichero de
 *   configuracin que se le indique.
 *
 * Parmetros:
 *   fname - fichero a procesar
 *   sect  - seccin a procesar
 *
 * Devuelve:
 *   0 en caso de xito, -1 en caso de error.
 *
 *****************************************************************************/

static int loadConfigurationFile(char *fname, char *sect)
{
  char lineBuffer[80 * 25], *a, *b, *h;
  int lineno         = 0,
      ignore_section = 0,
      fin            = 0,
      c,
      error = 0;
  FILE *f;

  if(!sect || !strcmp(sect, "global"))
  {
    ignore_section = 0;
    sect = "global";
  } else
    ignore_section = 1;

  _pdi_debug(THIS, "Processing '%s:%s' configuration.", fname, sect);
  if((f = fopen(fname, "rb")) == NULL)
  {
    _pdi_error(THIS, "Cannot open config file '%s'.", fname);
    return -1;
  }
  
  while(!fin)
  {
    /* leemos una lnea */
    if(!fgets(lineBuffer, sizeof(lineBuffer), f))
    {
      fin = 1;
      continue;
    }
    lineno++;

    if(lineBuffer[strlen(lineBuffer) - 1] != '\n')
    {
      /* Nos saltamos el resto de la lnea (o el fichero) */
      while((c = fgetc(f)) != '\n' && c != EOF)
        ;
      if(c == '\n')
      {
        _pdi_error(__FILE__, NULL,
                   "%s:%d:Line ignored because it is too long.",
                   fname, lineno);
        error++;
        continue;
      } else
        fin = 1; /* indicamos que ya no habr ms fichero */
    } else
      lineBuffer[strlen(lineBuffer) - 1] = '\0';

    /* quitamos los espacios de los extremos */
    for(a = lineBuffer; *a && (*a == ' ' || *a == '\t'); a++)
      ;
    if(a != lineBuffer)
    {
      for(b = lineBuffer; *a; *b++ = *a++)
        ;
      *b = '\0';
    }
    for(a = lineBuffer + strlen(lineBuffer) - 1;
        a > lineBuffer && (*a == ' ' || *a == '\t');
        *a-- = '\0')
      ;
    
    /* nos saltamos comentarios y lneas vacias */
    if(lineBuffer[0] == '\0' || lineBuffer[0] == '#')
      continue;

    /* si es una seccin, recogemos su nombre y segn */
    /* cual sea activamos el flag 'ignore_section'    */
    if(lineBuffer[0] == '[')
    {
      if(lineBuffer[strlen(lineBuffer)-1] == ']')
      {
        lineBuffer[strlen(lineBuffer)-1] = '\0';
        ignore_section = strcmp(lineBuffer + 1, sect);
      } else {
        _pdi_error(__FILE__, NULL, "%s:%d:Malformed section name.",
                   fname, lineno);
        error++;
      }
      continue;
    }

    /* comprobamos que tenga una estructura vlida */
    for(a = lineBuffer; *a && *a != ' ' && *a != '\t' && *a != '='; a++)
      ;
    b = a;
    for(; *a && (*a == ' ' || *a == '\t'); a++)
      ;
    if(!*a || *a == '=')
    {
      if(*a++)
      {
        *b = '\0';
        for(; *a && (*a == ' ' || *a == '\t'); a++)
          ;

        if(*a == '"')
        {
          /* Parseamos la cadena y aplicamos los escapes */
          for(h = b = ++a; *h && *h != '"'; h++, b++)
          {
            if(h[0] == '\\')
            {
              if(!h[1])
              {
                _pdi_error(__FILE__, NULL,
                           "%s:%d:Escape mark at the end of a string.",
                           fname, lineno);
                error++;
              }
              if(h[1] == '"' || h[1] == '\\')
                h++;
            }
            *b = *h;
          }
          if(!*h++)
          {
            _pdi_error(__FILE__, NULL, "%s:%d:Expected quotes.",
                       fname, lineno);
            error++;
            continue;
          }
          while(*h && (*h == ' ' || *h == '\t'))
            h++;
          if(*h)
            _pdi_warning(__FILE__, NULL,
                         "%s:%d:Trailing characters at the end of line.",
                         fname, lineno);
          *b = '\0';
        }
      } else
        a = NULL;
      if(!ignore_section)
        if((b = setConfigParam(searchConfigParam(lineBuffer), a && *a ? a : NULL)) != NULL)
        {
          _pdi_error(__FILE__, NULL, "%s:%d:Parameter '%s': %s.",
                     fname, lineno, lineBuffer, b);
          error++;
        }
    } else {
      *b = '\0';

      if(*a == '"')
      {
        /* Parseamos la cadena y aplicamos los escapes */
        for(h = b = ++a; *h && *h != '"'; h++, b++)
        {
          if(h[0] == '\\')
          {
            if(!h[1])
            {
              _pdi_error(__FILE__, NULL,
                         "%s:%d:Escape character at end of string.",
                         fname, lineno);
              error++;
            }
            if(h[1] == '"' || h[1] == '\\')
              h++;
          }
          *b = *h;
        }
        if(!*h++)
        {
          _pdi_error(__FILE__, NULL, "%s:%d:Quotes expected.",
                     fname, lineno);
          error++;
          continue;
        }
        while(*h && (*h == ' ' || *h == '\t'))
          h++;
        if(*h)
          _pdi_warning(__FILE__, NULL, "%s:%d:Trailing characters at end of string.",
                       fname, lineno);
        *b = '\0';
      }

      if(!strcasecmp(lineBuffer, "include"))
      {
        if(!ignore_section)
        {
          for(b = a; *b && *b != ':'; b++)
            ;
          if(*b == ':')
            *b++ = '\0';
          else
            b = NULL;

          if(b && !strcasecmp(b, "%PLATFORM%"))
            b = PLATFORM_ID;
          if(!strlen(a))
            a = fname;

          if(_pdi_fileIsRegular(a))
          {
            _pdi_debug(THIS, "%s:%d:Including '%s%s%s'.",
                       fname, lineno, a, b ? ":" : "", b ? b : "");

            if(loadConfigurationFile(a, b))
            {
              _pdi_error(__FILE__, NULL,
                         "%s:%d:An error happened while including '%s%s%s'.",
                         fname, lineno, a, b ? ":" : "", b ? b : "");
              error++;
            }
          } else {
            _pdi_warning(__FILE__, NULL, "%s:%d:Cannot include file '%s'.",
                         fname, lineno, a, b);
          }
        }
      } else
      if(!strcasecmp(lineBuffer, "log"))
      {
        if(!ignore_section)
          _pdi_log(__FILE__, NULL, "%s:%d:%s.", fname, lineno, a);
      } else
      if(!strcasecmp(lineBuffer, "warning"))
      {
        if(!ignore_section)
          _pdi_warning(__FILE__, NULL, "%s:%d:%s.", fname, lineno, a);
      } else
      if(!strcasecmp(lineBuffer, "error"))
      {
        if(!ignore_section)
        {
          _pdi_error(__FILE__, NULL, "%s:%d:%s.", fname, lineno, a);
          error++;
        }
      } else {
        _pdi_error(__FILE__, NULL, "%s:%d:Unknown command '%s'.",
                   fname, lineno, lineBuffer);
        error++;
      }
    }
  }

  fclose(f);

  return error;
}

/*****************************************************************************
 * static void loadEnvVars(void)
 *
 * Descripcin:
 *   Procesa algunas variables de entorno.
 *
 * Parmetros:
 *   -ninguno-
 *
 * Devuelve:
 *   -nada-
 *
 *****************************************************************************/

static void loadEnvVars(void)
{
  CONFIGPARAM *cp;
  char *v;
  
  /* Reseguimos la lista de parmetros */
  for(cp = configparams; cp->name; cp++)
    if(cp->env && (v = getenv(cp->env)) != NULL)
      setConfigParam(cp, strlen(v) ? v : NULL);

  /* La variable DI_FEEDBACK se mantiene por compatibilidad. Su */
  /* existencia equivale a poner el parmetro verbose al mximo */
  if(getenv(PDI_ENV_FEEDBACK))
    PDICFG.verbose = 3;

  /* Si se define DI_FOR_CHAPMAN avisamos de que ya no se usa esta variable */
  if(getenv(PDI_ENV_CHAPMAN))
    _pdi_warning(NULL, NULL, "The environment variable " PDI_ENV_CHAPMAN
                 " is not used in pDI-Tools.");
}

/*****************************************************************************
 * int _pdi_loadConfiguration(void)
 *
 * Descripcin:
 *   Asigna los valores por defecto a los parmetros de funcionamiento, procesa
 *   las variables de entorno y carga la configuracin (si est disponible).
 *
 * Parmetros:
 *   -ninguno-
 *
 * Devuelve:
 *   0 en caso de xito, -1 en caso de error.
 *
 *****************************************************************************/

int _pdi_loadConfiguration(void)
{
  CONFIGPARAM *cp;
  char *fname;

  /* Establecemos la configuracin por defecto */
  PDICFG.allow_lib_as_be    = 0;      /* no permitimos usar libs como be's   */
  PDICFG.debug              = 0;      /* por defecto, el debug va apagado    */
  PDICFG.donttouch_backends = 1;      /* evidentemente, no dejamos que se    */
  PDICFG.donttouch_pdi      = 1;      /* instrumenten cosas raras            */
  PDICFG.log_filename       = NULL;   /* no hay ningn fichero de log        */
  PDICFG.logfile            = stderr; /* y todo se imprime por stderr        */
  PDICFG.verbose            = 1;      /* queremos slo warnings y errores    */
  PDICFG.max_objects        = 40;     /* nmero mximo de objetos en memoria */
  PDICFG.beconfig_files     = NULL;   /* Lista de ficheros de configuracin  */
  PDICFG.n_beconfig_files   = 0;
  PDICFG.be_path            = NULL;   /* Paths a los backends                */
  PDICFG.n_be_path          = 0;
  PDICFG.becfg_path         = NULL;   /* Paths a ficheros de configuracin   */
  PDICFG.n_becfg_path       = 0;
  PDICFG.lib_path           = NULL;   /* Path a las libreras                */
  PDICFG.n_lib_path         = 0;
  PDICFG.no_check_on_config = 0;      /* always check files if possible      */
#if PDI_CALLBACKS_ENABLED
  PDICFG.cb_max_stubs       = _pdi_arch_defaultMaxStubs();
  PDICFG.cb_allow_handler   = 0;      /* por defecto lo prohibimos           */
  PDICFG.cb_stack_size      = 1024;   /* parece ms que suficiente (o no...) */
  PDICFG.max_threads        = 100;    /* nmero mximo de threads            */
  PDICFG.num_threads        = -1;     /* asumimos nmero threads desconocido */
#else
  PDICFG.cb_max_stubs       = 0;
  PDICFG.cb_allow_handler   = 0;
  PDICFG.cb_stack_size      = 0;
  PDICFG.max_threads        = 0;
  PDICFG.num_threads        = 0;
#endif

  /* En 'be_path' y 'becfg_path' aadimos algunos directorios por defecto. */
  if(insertPath(&(PDICFG.be_path),
                &(PDICFG.n_be_path),
                ".:" PDI_PATH_RUNTIME ":" PDI_PATH_BACKENDS,
                1, ':'))
  {
    _pdi_error(__FILE__, NULL, "Adding a search path of backends.");
    return -1;
  }

  if(insertPath(&(PDICFG.becfg_path),
                &(PDICFG.n_becfg_path),
                ".:" PDI_PATH_BECFG ":" PDI_PATH_ETC,
                1, ':'))
  {
    _pdi_error(__FILE__, NULL, "Adding a search path of backend command files.");
    return -1;
  }

  if(getenv("LD_LIBRARY_PATH")
  && insertPath(&(PDICFG.lib_path),
                &(PDICFG.n_lib_path),
                getenv("LD_LIBRARY_PATH"),
                1, ':'))
  {
    _pdi_error(__FILE__, NULL, "Adding a search path of shared objects (libs).");
    return -1;
  }

  if(insertPath(&(PDICFG.lib_path),
                &(PDICFG.n_lib_path),
                "/lib:/usr/lib",
                1, ':'))
  {
    _pdi_error(__FILE__, NULL, "Adding a search path of shared objects (libs).");
    return -1;
  }

  /* inicializamos los contadores de los parmetros */
  for(cp = configparams; cp->name; cp++)
    cp->counter = 0;

  /* Interpretamos las variables de entorno */
  loadEnvVars();

  /* Buscamos el fichero de configuracin, si existe lo intentamos cargar */
  if((fname = findConfigurationFile()) != NULL)
  {
    _pdi_log(THIS, "Using config file '%s'.", fname);

    if(loadConfigurationFile(fname, NULL))
    {
      _pdi_error(__FILE__, NULL, "While processing config file '%s'",
                 fname);
      return -1;
    }
    free(fname);
  } else 
    _pdi_log(THIS, "Cannot find any configuration file.");

  /* Creamos el fichero log */
  if(PDICFG.log_filename && strlen(PDICFG.log_filename))
    if((PDICFG.logfile = fopen(PDICFG.log_filename, "w")) == NULL)
    {
      PDICFG.logfile = stderr;
      _pdi_error(THIS, "Cannot create log file '%s'.",
                 PDICFG.log_filename);
    }

  /* Emitimos warnings si la configuracin es muy alocada */
  if(!PDICFG.donttouch_pdi)
    _pdi_warning(THIS,
                 "Is dangerous to disable 'donttouch_pdi' flag. Luck!");

  return 0;
}

void _pdi_unloadConfiguration(void)
{
  /* Liberamos memoria */
  if(PDICFG.log_filename)
    free(PDICFG.log_filename);
  resetStack(&(PDICFG.beconfig_files), &(PDICFG.n_beconfig_files), 0);
  resetStack(&(PDICFG.be_path),        &(PDICFG.n_be_path),        0);
  resetStack(&(PDICFG.becfg_path),     &(PDICFG.n_becfg_path),     0);

  /* Cerramos los ficheros y restauramos la salida del log */
  /* a stderr, por si ocurriese un error de ltima hora    */
  if(PDICFG.logfile != stderr)
    fclose(PDICFG.logfile);
  PDICFG.logfile = stderr;
}

