/*****************************************************************************
 * $Id: objlist.c,v 1.1 2005/08/10 23:33:18 killabyte Exp $
 *
 * Functions to access and use lists of PDI_ELFOBJ structures.
 *
 * ---------------------------------------------------------------------------
 * pDI-Tools - portable Dynamic Instrumentation Tools
 *   (C) 2004, 2005 Gerardo Garca Pea
 *   Programmed by Gerardo Garca Pea - Inspired on CEPBA DItools
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301
 *   USA
 *
 *****************************************************************************/

#include<config.h>
#include<pdiconfig.h>
#include<ebeif.h>
#include<log.h>

/*****************************************************************************
 * Global variables
 *
 *   _pdi_objlist
 *     It is a "mirror" of the 'ldlinux.so' object list.
 *
 *   _pdi_libcobj
 *   _pdi_mainobj
 *   _pdi_pdiobj
 *     Pointers to famous and important objects.
 *
 * Module's Private Variables
 *
 *   freelist
 *     A list of free objects in the pool.
 *
 *   last_objlist
 *     The last assigned entry in the object pool.
 *
 *   objpool
 *     A static table of 'PDICFG.max_objects'. It contains two lists:
 *     '_pdi_objlist' and 'freelist'.
 * 
 *****************************************************************************/

PDI_ELFOBJ *_pdi_objlist;
PDI_ELFOBJ *_pdi_libcobj;
PDI_ELFOBJ *_pdi_mainobj;
PDI_ELFOBJ *_pdi_pdiobj;

static PDI_ELFOBJ *freelist, *last_objlist;
static PDI_ELFOBJ *objpool;

/*****************************************************************************
 * static PDI_ELFOBJ *_pdi_ebe_newElfObj(PDI_ELFOBJ *next)
 *
 * Descripcin:
 *   Inserta en la cadena de objetos '_pdi_objlist', justo delante del objeto
 *   'next' un nuevo objeto PDI_ELFOBJ vacio, y devuelve un puntero al mismo.
 *
 * Parmetros:
 *   next - Un puntero al objeto PDI_ELFOBJ delante del cual se quiere insertar
 *         el nuevo elemento. Si este parmetro se deja en NULL se aade al
 *         final de la lista de objetos '_pdi_objlist'.
 *
 * Devuelve:
 *   NULL si se agoto la memoria o hubo algn problema, y un puntero a una
 *   estructura PDI_ELFOBJ en caso de xito.
 *
 *****************************************************************************/

PDI_ELFOBJ *_pdi_ebe_newElfObj(PDI_ELFOBJ *next)
{
  PDI_ELFOBJ *eo, *freelist_next;

  /* Comprobamos que an haya espacio */
  if(!freelist)
  {
    _pdi_debug(THIS, "There is not any free entry in the object pool.");
    return NULL;
  }

  /* Pillamos el primero de la lista de libres y lo inicializamos a lo      */
  /* bestia: lo ponemos a cero y le cascamos una estructura PDI_ARCH_ELFOBJ */
  eo = freelist;
  freelist_next = freelist->o_next;
  memset(eo, 0, sizeof(PDI_ELFOBJ));
  if(_pdi_arch_newElfObj(eo, eo - objpool))
  {
    _pdi_debug(THIS, "Cannot initialize ELFOBJ arch dependant part.");
    return NULL;
  }

  /* Como ha ido todo bien lo extraemos de la lista de libres */
  freelist = freelist_next;

  /* y lo encadenamos la estructura a la lista de objetos (_pdi_objlist) */
  if(!next)
  {
    eo->o_prev = last_objlist;
    if(last_objlist)
      last_objlist->o_next = eo;
    last_objlist = eo;
    if(!_pdi_objlist)
      _pdi_objlist = eo;
  } else {
    /* eo apunta a next y al antiguo previo de next */
    eo->o_next = next;
    eo->o_prev = next->o_prev;
    if(eo->o_prev) eo->o_prev->o_next = eo; /* prev debera apuntar a eo */
    next->o_prev = eo;                      /* next debera apuntar a eo */
  }

  return eo;
}

/*****************************************************************************
 * void _pdi_ebe_freeElfObj(PDI_ELFOBJ *eo)
 *
 * Descripcin:
 *   Libera la memoria ocupada por una estructura PDI_ELFOBJ y la desencadena
 *   de la lista global.
 *
 * Parmetros:
 *   eo - Un puntero al objeto PDI_ELFOBJ que queremos destruir.
 *
 * Devuelve:
 *   Nada.
 *
 * TODO:
 *   - Destruir las interposiciones afectadas por la descarga.
 *
 *****************************************************************************/

void _pdi_ebe_freeElfObj(PDI_ELFOBJ *eo)
{
  /* Desencadenamos eo */
  if(eo->o_prev)
    eo->o_prev->o_next = eo->o_next;
  if(eo->o_next)
    eo->o_next->o_prev = eo->o_prev;

  /* liberamos las cosas de 'eo' que ocupen memoria */
  _pdi_ebe_setObjectAlias(eo, NULL);
  _pdi_arch_freeElfObj(eo, eo - objpool);

  /* Ponemos el eo en la lista de libres */
  eo->o_next = freelist;
  eo->o_prev = NULL;
  freelist = eo;
}

/*****************************************************************************
 * PDI_ELFOBJ *_pdi_ebe_searchObjectByPath(char *o_name)
 *
 * Descripcin:
 *   Busca en la lista de objetos en memoria un objeto cuyo path sea 'o_name'
 *
 * Parmetros:
 *   o_name - Path del objeto que buscamos.
 *
 * Devuelve:
 *   Un puntero a una estructura PDI_ELFOBJ si la bsqueda tuvo xito, NULL si
 *   no se encontr el objeto.
 *
 *****************************************************************************/

PDI_ELFOBJ *_pdi_ebe_searchObjectByPath(char *o_name)
{
  PDI_ELFOBJ *eo;

  /* Buscamos un objeto con ese nombre */
  for(eo = _pdi_objlist; eo; eo = eo->o_next)
    if(eo->o_name && !strcmp(eo->o_name, o_name))
      return eo;

  return NULL;
}

/*****************************************************************************
 * PDI_ELFOBJ *_pdi_ebe_searchObjectByAlias(char *o_alias)
 *
 * Descripcin:
 *   Busca en la lista de objetos en memoria un objeto cuyo alias sea 'o_alias'
 *
 * Parmetros:
 *   o_alias - Alias del objeto que buscamos.
 *
 * Devuelve:
 *   Un puntero a una estructura PDI_ELFOBJ si la bsqueda tuvo xito, NULL si
 *   no se encontr el objeto.
 *
 *****************************************************************************/

PDI_ELFOBJ *_pdi_ebe_searchObjectByAlias(char *o_alias)
{
  PDI_ELFOBJ *eo;

  /* Buscamos un objeto con ese alias */
  for(eo = _pdi_objlist; eo; eo = eo->o_next)
    if(eo->o_alias && !strcmp(eo->o_alias, o_alias))
      return eo;

  return NULL;
}

/*****************************************************************************
 * PDI_ELFOBJ *_pdi_ebe_searchObject(char *x)
 *
 * Descripcin:
 *   Busca en la lista de objetos en memoria un objeto cuyo alias sea 'x'. Si
 *   no se encuentra ningn objeto se busca un objeto cuyo path sea 'x'.
 *
 * Parmetros:
 *   x - Alias o path del objeto que buscamos.
 *
 * Devuelve:
 *   Un puntero a una estructura PDI_ELFOBJ si la bsqueda tuvo xito, NULL si
 *   no se encontr el objeto.
 *
 *****************************************************************************/

PDI_ELFOBJ *_pdi_ebe_searchObject(char *x)
{
  PDI_ELFOBJ *ret;

  if((ret = _pdi_ebe_searchObjectByAlias(x)) != NULL)
    return ret;

  return _pdi_ebe_searchObjectByPath(x);
}

/*****************************************************************************
 * int _pdi_ebe_setObjectAlias(PDI_ELFOBJ *obj, char *alias)
 *
 * Descripcin:
 *   Le pone al objeto 'obj' el alias 'alias'. Si 'alias' es NULL se le quita
 *   el alias actual al objeto.
 *
 * Parmetros:
 *   obj   - Objeto
 *   alias - Alias que le queremos poner
 *
 * Devuelve:
 *   0 si hubo xito, -1 en caso de error.
 *
 *****************************************************************************/

int _pdi_ebe_setObjectAlias(PDI_ELFOBJ *obj, char *alias)
{
  PDI_ELFOBJ *eo;
  char *new_alias;

  /* Primero buscamos que no haya ningn objeto con ese alias */
  if((eo = _pdi_ebe_searchObjectByAlias(alias)) != NULL)
  {
    _pdi_error(THIS, "Alias '%s' cannot be assigned to '%s' because it is "
               "used by object '%s'.", alias, _pdi_ebe_getObjectName(obj),
               _pdi_ebe_getObjectName(eo));
    return -1;
  }

  if(alias)
  {
    if((new_alias = malloc(strlen(alias) + 1)) == NULL)
    {
      _pdi_error(THIS, "No memory for new alias '%s' for object '%s'.",
                 _pdi_ebe_getObjectName(obj));
      return -1;
    }
    strcpy(new_alias, alias);
  } else
    new_alias = NULL;

  if(obj->o_alias)
  {
    _pdi_warning(THIS,
                 "Alias of object '%s' (%s) have %s%s%s.",
                 _pdi_ebe_getObjectName(obj), obj->o_alias,
                 new_alias ? "been substituted by '" : " been removed",
                 new_alias ? new_alias : "",
                 new_alias ? "'" : "");
    free(obj->o_alias);
  }
  obj->o_alias = new_alias;

  return 0;
}

/*****************************************************************************
 * char *_pdi_ebe_getObjectName(PDI_ELFOBJ *obj)
 *
 * Descripcin:
 *   Obtiene el path o el alias del ejecutable.
 *
 * Parmetros:
 *   obj - Objeto del que queremos averiguar el nombre.
 *
 * Devuelve:
 *   El nombre del objeto.
 *
 *****************************************************************************/

char *_pdi_ebe_getObjectName(PDI_ELFOBJ *obj)
{
  return strlen(obj->o_name)
           ? obj->o_name
           : (obj->o_alias
               ? obj->o_alias
               : "--object wo name--");
}

/*****************************************************************************
 * int _pdi_ebe_initObjectList(void)
 *
 * Descripcin:
 *   Inicializa la lista de objetos.
 *
 * Parmetros:
 *   ninguno
 *
 * Devuelve:
 *   0 si hubo xito, otro valor en caso de error.
 *
 *****************************************************************************/

int _pdi_ebe_initObjectList(void)
{
  int i;

  /* Reservamos memoria para el pool de estructuras PDI_ELFOBJ */
  if(PDICFG.max_objects <= 0)
  {
    _pdi_error(THIS, "Parameter 'max_objects' must be a positive value.");
    return -1;
  }

  if((objpool = malloc(sizeof(PDI_ELFOBJ) * PDICFG.max_objects)) == NULL)
  {
    _pdi_error(THIS, "Cannot alloc %d bytes for the object pool.",
               sizeof(PDI_ELFOBJ) * PDICFG.max_objects);
    return -1;
  }

  _pdi_log(THIS, "Reserved %d bytes for an object pool of %d entries.",
           sizeof(PDI_ELFOBJ) * PDICFG.max_objects, PDICFG.max_objects);

  /* Ponemos a cero de golpe a toda la estructura */
  memset(objpool, 0, sizeof(PDI_ELFOBJ) * PDICFG.max_objects);

  /* Ponemos todos los objetos en la lista de libres y _pdi_objlist vacio */
  for(i = 0; i < PDICFG.max_objects - 1; i++)
    (objpool + i)->o_next = objpool + i + 1;
  freelist = objpool;

  _pdi_objlist = last_objlist = NULL;

  /* Inicializamos la parte dependiente de la arquitectura */
  /* (es decir, construmos la lista inicial...)           */
  if(_pdi_arch_initObjectList())
  {
    _pdi_error(THIS, "Cannot initialize arch dependant object list.");
    return -1;
  }

  return 0;
}

/*****************************************************************************
 * int _pdi_ebe_finiObjectList(void)
 *
 * Descripcin:
 *   Destruye la lista de objetos y libera la memoria usada.
 *
 * Parmetros:
 *   ninguno
 *
 * Devuelve:
 *   nada.
 *
 *****************************************************************************/

void _pdi_ebe_finiObjectList(void)
{
  /* Finalizamos la parte dependiente de la arquitectura */
  _pdi_arch_finiObjectList();

  /* Ponemos a NULL todos los punteros pblicos */
  _pdi_objlist = NULL;
  _pdi_libcobj = NULL;
  _pdi_mainobj = NULL;
  _pdi_pdiobj  = NULL;

  /* Liberamos memoria usada */
  free(objpool);
  freelist     = NULL;
  last_objlist = NULL;
  objpool      = NULL;
}

