/*****************************************************************************
 * $Id: lx-symtab.c,v 1.1 2005/08/28 19:06:41 killabyte Exp $
 *
 * Functions to work on symbol table of objects in memory.
 *
 * ---------------------------------------------------------------------------
 * pDI-Tools - portable Dynamic Instrumentation Tools
 *   (C) 2004, 2005 Gerardo Garca Pea
 *   Programmed by Gerardo Garca Pea - Inspired on CEPBA DItools
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301
 *   USA
 *
 *****************************************************************************/

#include<config.h>
#include<ebeif.h>
#include<linux/elfops.h>

/*****************************************************************************
 * static inline unsigned int elf_hash(const unsigned char *name)
 *
 * Description:
 *   This function computes the hash of a symbol name to search it in the
 *   section DT_HASH of an ELF object. This function is an optimized version of
 *   the function described in the ELF ABI. It has been extracted from GNU
 *   libc.
 *
 * Parameters:
 *   name - symbol name
 *
 * Returns:
 *   A 32 bit unsigned integer.
 *
 *****************************************************************************/

static inline unsigned int elf_hash(const unsigned char *name)
{
  unsigned long int hash = 0;

  if(*name != '\0')
  {
    hash = *name++;
    if(*name != '\0')
    {
      hash = (hash << 4) + *name++;
      if(*name != '\0')
      {
        hash = (hash << 4) + *name++;
        if(*name != '\0')
        {
          hash = (hash << 4) + *name++;
          if(*name != '\0')
          {
            hash = (hash << 4) + *name++;
            while(*name != '\0')
            {
              unsigned long int hi;
              hash = (hash << 4) + *name++;
              hi = hash & 0xf0000000;

              /* The algorithm specified in the ELF ABI is as
               * follows:
               *
               *   if(hi != 0)
               *     hash ^= hi >> 24;
               *
               *   hash &= ~hi;
               *
               * But the following is equivalent and a lot
               * faster, especially on modern processors.  */

              hash ^= hi;
              hash ^= hi >> 24;
            }
          }
        }
      }
    }
  }

  return hash;
}

/*****************************************************************************
 * ElfW(Sym) *_pdi_linux_getSymbolTableEntry(PDI_LINUX_ELFOBJ *obj,
 *                                           char *name,
 *                                           unsigned int *symndx)
 *
 * Description:
 *   Search symbol 'name' in 'obj' symbol table. It returns a pointer to the
 *   symbol information and its index in symbol table.
 *
 * Parameters:
 *   obj    - object
 *   name   - searched symbol
 *   symndx - (out). Symbol index (if it exists). If this parameter is NULL
 *            nothing will be returned through it.
 *
 * Returns:
 *   A pointer to symbol info, or NULL if the symbol does not exist.
 *
 *****************************************************************************/

ElfW(Sym) *_pdi_linux_getSymbolTableEntry(PDI_LINUX_ELFOBJ *obj,
                                          char *name,
                                          Elf_Symndx *symndx)
{
  unsigned int hash;
  Elf_Symndx si;
  ElfW(Sym) *symptr;

  /* Get the hash of symbol name */
  hash = elf_hash((unsigned char *) name);

  /* Search symbol in hash section */
  for(si = obj->l_bucket[hash % obj->n_bucket];
      si != STN_UNDEF;
      si = obj->l_chain[si])
  {
    symptr = (ElfW(Sym) *) (((char *) obj->symbolTable) + (si * obj->symbolSize));
    if(!strcmp(name, ((char *) obj->stringTable) + symptr->st_name))
    {
      /* Symbol found! Returning a pointer to the symbol */
      if(symndx) *symndx = si;
      return symptr;
    }
  }

  /* The symbol does not exist :'( */
  if(symndx) *symndx = STN_UNDEF;
  return NULL;
}

