/*****************************************************************************
 * $Id: lx-rel.c,v 1.1 2005/08/28 17:34:36 killabyte Exp $
 *
 * This file implements some function to search REL/A structures.
 *
 * ---------------------------------------------------------------------------
 * pDI-Tools - portable Dynamic Instrumentation Tools
 *   (C) 2004, 2005 Gerardo Garca Pea
 *   Programmed by Gerardo Garca Pea - Inspired on CEPBA DItools
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301
 *   USA
 *
 *****************************************************************************/

#include<config.h>
#include<stdlib.h>
#include<log.h>
#include<linux/elfops.h>

/*****************************************************************************
 * static LINUX_PLTREL *dicoSearchJmpRel(PDI_LINUX_ELFOBJ *obj,
 *                                       Elf_Symndx symndx)
 * static LINUX_PLTREL *seqSearchJmpRel(PDI_LINUX_ELFOBJ *obj,
 *                                      Elf_Symndx symndx)
 *
 * Description:
 *   This function search in section DT_JMPREL of object 'obj' the REL/A
 *   structure of the symbol with index 'symndx' in the symbol table, if it
 *   exists of course.
 *
 *   dicoSearchJmpRel() makes a dicotomic search in DT_JMPREL (only if it is
 *   sorted by symbol table index, of course).
 *   seqSearchJmpRel() makes a linear search in DT_JMPREL. It is only used when
 *   DT_JMPREL is not sorted.
 *
 * Parameters:
 *   obj    - object where we are looking for the relocation structure
 *   symndx - index of the searched symbol in symbol table
 *
 * Returns:
 *   A pointer to the REL/A structure or NULL if it doesn't exist.
 *
 *****************************************************************************/

static LINUX_PLTREL *dicoSearchJmpRel(PDI_LINUX_ELFOBJ *obj, Elf_Symndx symndx)
{
  int inf, sup, pivot;
  LINUX_PLTREL *rel;

  /* If DT_JMPREL is void terminate now ! */
  if(!obj->jmpRelTableSize)
    return NULL;

  /* dicotomic search */
  inf = 0;
  sup = (obj->jmpRelTableSize / obj->pltRelEntSize) - 1;
  while(inf + 1 < sup)
  {
    pivot = (inf + sup) >> 1;
    rel = PDI_JMPREL_ELEM(obj, pivot);
    if(ELFW_R_SYM(rel->r_info) == symndx)
      return rel;
    else
      if(ELFW_R_SYM(rel->r_info) < symndx)
        inf = pivot;
      else
        sup = pivot;
  }
  
  if(ELFW_R_SYM(PDI_JMPREL_ELEM(obj, sup)->r_info) == symndx)
    return PDI_JMPREL_ELEM(obj, sup);
  if(ELFW_R_SYM(PDI_JMPREL_ELEM(obj, inf)->r_info) == symndx)
    return PDI_JMPREL_ELEM(obj, inf);

  return NULL;
}

static LINUX_PLTREL *seqSearchJmpRel(PDI_LINUX_ELFOBJ *obj, Elf_Symndx symndx)
{
  ElfW(Addr) pos;

  for(pos = obj->jmpRelTable;
      pos < obj->jmpRelTable + obj->jmpRelTableSize;
      pos += obj->pltRelEntSize)
    if(ELFW_R_SYM(((LINUX_PLTREL *) pos)->r_info) == symndx)
      return (LINUX_PLTREL *) pos;

  return NULL;
}

/*****************************************************************************
 * LINUX_PLTREL *_pdi_linux_getFunctionRel(PDI_LINUX_ELFOBJ *obj,
 *                                         Elf_Symndx symndx)
 *
 * Description:
 *   This function search in object 'obj' the REL/A structure of the symbol
 *   with index 'symndx' in the symbol table, if it exists of course.
 *
 * Parameters:
 *   obj    - target object
 *   symndx - index of the searched symbol in symbol table
 *
 * Returns:
 *   Un puntero a la estructura REL/A o en caso de no encontrarse un NULL.
 *
 *****************************************************************************/

LINUX_PLTREL *_pdi_linux_getFunctionRel(PDI_LINUX_ELFOBJ *obj,
                                        Elf_Symndx symndx)
{
  LINUX_PLTREL *rel;

  /* If DT_JMPREL is sorted, then make a dicotomic search, or a */
  /* linear search otherwise                                    */
  rel = obj->jmpRelSortedBySymndx
          ? dicoSearchJmpRel(obj, symndx)
          : seqSearchJmpRel(obj, symndx);

  /* if it have not found this function exits quietly returning NULL */
  if(!rel)
    return NULL;

  /* Check that is is a LINUX_R_JMPREL_TYPE relocation structure.            */
  /* NOTE: The value of LINUX_R_JMPREL_TYPE depends on platform and it could */
  /*       be ugly things like:  R_386_JMP_SLOT, R_IA64_IPLTLSB, etc...      */
  if(ELFW(R_TYPE)(rel->r_info) != LINUX_R_JMPREL_TYPE)
  {
    _pdi_debug(THIS, "Symbol '%s' does not have a " LINUX_R_JMPREL_TYPE_STR 
               " relocation structure.", PDI_GET_SYMBOL_NAME(obj, symndx));
    return NULL;
  }

  return rel;
}

