/*****************************************************************************
 * $Id: lx-objlist.c,v 1.2 2005/08/28 19:25:39 killabyte Exp $
 *
 * This module contains subroutines which examine the objects in memory and
 * extract important info about them. They also mantain a list of the objects
 * in memory which is used to iterate on these objects in a non-arch-dependant
 * way.
 *
 * ---------------------------------------------------------------------------
 * pDI-Tools - portable Dynamic Instrumentation Tools
 *   (C) 2004, 2005 Gerardo Garca Pea
 *   Programmed by Gerardo Garca Pea - Inspired on CEPBA DItools
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301
 *   USA
 *
 *****************************************************************************/

#include<config.h>
#include<pdiconfig.h>
#include<ebeif.h>
#include<files.h>
#include<log.h>

#define LMNAME(x)            (((x)->l_name && strlen((x)->l_name) > 0) \
                               ? (x)->l_name : "<null name>")

/*****************************************************************************
 * Module's private variables
 *
 *   archobjlist
 *     A pool of arch-dependant structures, where each object is directly
 *     associated with a PDI_ELFOBJ structure in the no arch-dependant pool.
 *
 *****************************************************************************/

static PDI_LINUX_ELFOBJ *archobjlist;

/*****************************************************************************
 * int _pdi_linux_newElfObj(PDI_ELFOBJ *eo, int objpool_index)
 *
 * Description:
 *   Alloc memory for a new PDI_LINUX_ELFOBJ structure and initialize the field
 *   'o_arch' in 'eo' with a pointer to it.
 *
 * Parameters:
 *   eo            - object associated with the new PDI_LINUX_ELFOBJ structure.
 *   objpool_index - pDI-Tools uses a static pool of PDI_ELFOBJ structures.
 *                   This index is the position of the structure 'eo' in this
 *                   pool.
 *
 * Returns:
 *   0 if succesful, -1 otherwise.
 *
 *****************************************************************************/

int _pdi_linux_newElfObj(PDI_ELFOBJ *eo, int objpool_index)
{
  eo->o_arch = &(archobjlist[objpool_index]);

  return 0;
}

/*****************************************************************************
 * void _pdi_linux_freeElfObj(PDI_ELFOBJ *eo, int objpool_index)
 *
 * Description:
 *   Free memory used by a PDI_LINUX_ELFOBJ structure.
 *
 * Parameters:
 *   eo            - object
 *   objpool_index - Index of structure 'eo' in PDI_ELFOBJ pool.
 *
 * Returns:
 *   nothing.
 *
 *****************************************************************************/

void _pdi_linux_freeElfObj(PDI_ELFOBJ *eo, int objpool_index)
{
  free(eo->o_name);
  eo->o_name = NULL;
}

/*****************************************************************************
 * static void initObject(PDI_ELFOBJ *eo, struct link_map *lm)
 *
 * Description:
 *   Check a lot of arch-dependant information of object 'lm', and caches some
 *   values in structure 'eo'.
 *
 * Parameters:
 *   eo - A pointer to the PDI_ELFOBJ structure to initialize
 *   lm - link_map (specific info about object)
 *
 * Returns:
 *   nothing.
 *
 *****************************************************************************/

#define IO_GET_PT(x,y)   case DT_##x: flags[f_##x]++; eo->o_arch->y = dyn->d_un.d_ptr; break;
#define IO_GET_VL(x,y)   case DT_##x: flags[f_##x]++; eo->o_arch->y = dyn->d_un.d_val; break;

static int initObject(PDI_ELFOBJ *eo, struct link_map *lm)
{
  ElfW(Dyn) *dyn;
  enum { f_STRTAB,  f_STRSZ,  f_SYMTAB,   f_SYMENT, f_PLTGOT,
         f_REL,     f_RELSZ,  f_RELENT,   f_RELA,   f_RELASZ,
         f_RELAENT, f_JMPREL, f_PLTRELSZ, f_PLTREL, f_HASH };
  int flags[15];
  char tmp[PATH_MAX];

  /* Save a pointer to the linkmap structure and get the DSO filename */
  eo->o_arch->lm = lm;
  if(lm->l_name && strlen(lm->l_name) > 0)
  {
    if(_pdi_getCanonicalPath(lm->l_name, tmp, 0))
    {
      _pdi_error(THIS, "Cannot canonicalize path '%s'", lm->l_name);
      return -1;
    }
    if(!strlen(tmp))
      strcpy(tmp, lm->l_name);
  } else
    *tmp = '\0';
  if((eo->o_name = malloc(strlen(tmp)+1)) == NULL)
  {
    _pdi_error(THIS, "No memory for string '%s'", tmp);
    return -1;
  }
  strcpy(eo->o_name, tmp);
  eo->o_alias = NULL;

  /* Optimistically assume this object is ok */
  eo->o_arch->broken_dso = 0;

  /* Search and cache some required sections                                 */
  /* NOTE: Macros IO_GET_PT & IO_GET_VL get and fix the pointer/value stored */
  /*       in a dynamic tag and save it in some field of structure 'eo'.     */
  /*       IO_GET_PT is used to get a pointer, while IO_GET_VL is used to    */
  /*       read an ElfW(Word).                                               */
  memset(flags, 0, sizeof(flags));
  for(dyn = lm->l_ld; dyn->d_tag != DT_NULL; dyn++)
    switch(dyn->d_tag)
    {
      IO_GET_PT(STRTAB,   stringTable)     /* mandatory */
      IO_GET_VL(STRSZ,    stringTableSize) /* mandatory */
      IO_GET_PT(SYMTAB,   symbolTable)     /* mandatory */
      IO_GET_VL(SYMENT,   symbolSize)      /* mandatory */
      IO_GET_PT(PLTGOT,   gotTable)        /* optional  */
      IO_GET_PT(REL,      relTable)        /* mandatory (if !exist DT_RELA) */
      IO_GET_VL(RELSZ,    relTableSize)    /* mandatory (if !exist DT_RELA) */
      IO_GET_VL(RELENT,   relEntSize)      /* mandatory (if !exist DT_RELA) */
      IO_GET_PT(RELA,     relaTable)       /* mandatory (if !exist DT_REL) */
      IO_GET_VL(RELASZ,   relaTableSize)   /* mandatory (if !exist DT_REL) */
      IO_GET_VL(RELAENT,  relaEntSize)     /* mandatory (if !exist DT_REL) */
      IO_GET_PT(JMPREL,   jmpRelTable)
      IO_GET_VL(PLTRELSZ, jmpRelTableSize)
      IO_GET_VL(PLTREL,   pltRel)
      case DT_HASH: /* mandatory */
        flags[f_HASH]++;
        eo->o_arch->hashTable = _pdi_linux_vlibc < LINUX_LIBC_VABS(2,3,0)
                                  ? dyn->d_un.d_ptr + lm->l_addr
                                  : dyn->d_un.d_ptr;
        break; /* mandatory */
    }

  if(!flags[f_STRTAB] || !flags[f_STRSZ]
  || !flags[f_SYMTAB] || !flags[f_SYMENT]
  || !flags[f_HASH])
  {
    _pdi_error(THIS, "In object '%s' cannot find sections:", LMNAME(lm));
    if(!flags[f_STRTAB])  _pdi_error(THIS, "  - DT_STRTAB");
    if(!flags[f_STRSZ])   _pdi_error(THIS, "  - DT_STRSZ");
    if(!flags[f_SYMTAB])  _pdi_error(THIS, "  - DT_SYMTAB");
    if(!flags[f_SYMENT])  _pdi_error(THIS, "  - DT_SYMENT");
    if(!flags[f_HASH])    _pdi_error(THIS, "  - DT_HASH");
    exit(1);
  }
 
  if(!(flags[f_REL]  && flags[f_RELSZ]  && flags[f_RELENT]) &&
      !(flags[f_RELA] && flags[f_RELASZ] && flags[f_RELAENT]))
  {
    _pdi_debug(THIS, "Object '%s' is broken.", LMNAME(lm));
#if 0
    if(!flags[f_REL])     _pdi_debug(THIS, "  - DT_REL");
    if(!flags[f_RELSZ])   _pdi_debug(THIS, "  - DT_RELSZ");
    if(!flags[f_RELENT])  _pdi_debug(THIS, "  - DT_RELENT");
    if(!flags[f_RELA])    _pdi_debug(THIS, "  - DT_RELA");
    if(!flags[f_RELASZ])  _pdi_debug(THIS, "  - DT_RELASZ");
    if(!flags[f_RELAENT]) _pdi_debug(THIS, "  - DT_RELAENT");
#endif
    eo->o_arch->broken_dso    = -1;
    eo->o_arch->relaTable     = (ElfW(Addr)) NULL;
    eo->o_arch->relaTableSize = 0;
    eo->o_arch->relaEntSize   = 0;
    eo->o_arch->pltRel        = 0;
    eo->o_arch->relTable      = (ElfW(Addr)) NULL;
    eo->o_arch->relTableSize  = 0;
    eo->o_arch->relEntSize    = 0;
  } else {
    if(!flags[f_JMPREL])
      _pdi_debug(THIS, "Cannot find section DT_JMPREL in '%s'. Strange.",
                 LMNAME(lm));

    if(!flags[f_PLTGOT])
      _pdi_debug(THIS, "Cannot find section DT_PLTGOT in '%s'. Strange.",
                 LMNAME(lm));
  }

  if(flags[f_JMPREL])
  {
    /* Check that all mandantory sections are there */
    if(!flags[f_PLTRELSZ] || !flags[f_PLTREL])
    {
      _pdi_error(THIS, "Section DT_JMPREL in '%s', but without:", LMNAME(lm));
      if(!flags[f_PLTRELSZ]) _pdi_error(THIS, "  - DT_PLTRELSZ");
      if(!flags[f_PLTREL])   _pdi_error(THIS, "  - DT_PLTREL");
    }

    /* Guess entry size in DT_JMPREL */
    eo->o_arch->pltRelEntSize = (eo->o_arch->pltRel == DT_REL)
                                  ? eo->o_arch->relEntSize
                                  : eo->o_arch->relaEntSize;
  } else
    _pdi_debug(THIS, "Cannot find section DT_JMPREL in '%s'. Strange.",
               LMNAME(lm));

  if(!flags[f_PLTGOT])
    _pdi_debug(THIS, "Cannot find section DT_PLTGOT '%s'. Strange.",
               LMNAME(lm));

  /* Extract info about the hash table */
  eo->o_arch->n_bucket = *(((Elf_Symndx *) eo->o_arch->hashTable) + 0);
  eo->o_arch->n_chain  = *(((Elf_Symndx *) eo->o_arch->hashTable) + 1);
  eo->o_arch->l_bucket =   ((Elf_Symndx *) eo->o_arch->hashTable) + 2 + 0;
  eo->o_arch->l_chain  =   ((Elf_Symndx *) eo->o_arch->hashTable) + 2 + eo->o_arch->n_bucket;

  /* Check if the elements in DT_JMPREL are sorted by its index in the  */
  /* symbol table (DT_SYMTAB). If they are correctly sorted we can make */
  /* dicotomic searches in DT_JMPREL section.                           */
  if(flags[f_JMPREL])
  {
    ElfW(Addr) pos;
    Elf_Symndx last_symndx = 0;

    /* In the begining all was an ideal world where all were beautiful... */
    eo->o_arch->jmpRelSortedBySymndx = -1;

    /* Check that is actually a beautiful world... */
    for(pos = eo->o_arch->jmpRelTable;
        pos < eo->o_arch->jmpRelTable + eo->o_arch->jmpRelTableSize && eo->o_arch->jmpRelSortedBySymndx;
        pos += eo->o_arch->pltRelEntSize)
    {
      /* TODO XXX: Type Rel/Rela should depend on the value in DT_PLTREL
       * (DT_REL o DT_RELA). */
      eo->o_arch->jmpRelSortedBySymndx =
        !last_symndx || last_symndx < ELFW_R_SYM(((LINUX_PLTREL *) pos)->r_info);
      last_symndx = ELFW_R_SYM(((LINUX_PLTREL *) pos)->r_info);
    }
    if(!eo->o_arch->jmpRelSortedBySymndx)
      _pdi_debug(THIS,
                 "Section DT_JMPREL in '%s' is not sorted by symtab index.",
                 LMNAME(lm));
  }

  return 0;
}

/*****************************************************************************
 * static int buildObjectList(void)
 *
 * Description:
 *   Builds the list of PDI_ELFOBJ objects in memory.
 *
 * Parameters:
 *   none.
 *
 * Returns:
 *   0 if succesful, != 0 if any error.
 *
 *****************************************************************************/

static int buildObjectList(void)
{
  struct link_map *lm;
  PDI_ELFOBJ *eo;

  /* for each linkmap, create a new PDI_ELFOBJ object */
  for(lm = _r_debug.r_map; lm; lm = lm->l_next)
  {
    /* Alloc a PDI_ELFOBJ structure */
    if((eo = _pdi_ebe_newElfObj(NULL)) == NULL)
    {
      _pdi_error(THIS, "Cannot alloc memory for a new PDI_ELFOBJ.");
      return -1;
    }

    _pdi_debug(THIS, "Add object '%s' to the object list.", LMNAME(lm));
      
    /* Initialize it */
    if(initObject(eo, lm))
    {
      _pdi_error(THIS, "Cannot initialize object '%s'.", lm->l_name);
      return -1;
    }
  }

  return 0;
}

/*****************************************************************************
 * static int initMainObject(void)
 *
 * Description:
 *   Search, initialize and set alias PDI_ALIAS_MAIN to the executable binary.
 *
 * Parameters:
 *   none.
 *
 * Returns:
 *   0 if succesful, != 0 if any error.
 *
 *****************************************************************************/

static int initMainObject(void)
{
  /* In Linux, the first element in the list of mapped objects in memory, */
  /* it is always the main program (the binary executable).               */
  /* NOTE: There is also something like a bug: the main program is shown  */
  /* in the object list like a DSO without name. It is a worrying and     */
  /* curios curiosity.                                                    */

  /* Check that '_pdi_objlist' has been built and its first element is */
  /* the binary executable (see previous comentary)                    */
  if(!_pdi_objlist || !_pdi_objlist->o_name || strcmp(_pdi_objlist->o_name, ""))
  {
    _pdi_debug(THIS, "Object " PDI_ALIAS_MAIN
               " was expected on first position in the object list.");
    return -1;
  }

  /* Initialize '_pdi_mainobj' and assign it alias 'PDI_ALIAS_MAIN' */
  _pdi_mainobj = _pdi_objlist;
  if(_pdi_ebe_setObjectAlias(_pdi_mainobj, PDI_ALIAS_MAIN))
  {
    _pdi_error(THIS, "Cannot assign alias " PDI_ALIAS_MAIN " to the first object.");
    return -1;
  }

  return 0;
}

/*****************************************************************************
 * static int initLIBCObject(void)
 *
 * Description:
 *   Search, initialize and set alias PDI_ALIAS_LIBC to the C Shared Library.
 *
 *   NOTE: If cannot find 'libc.so.*' in the object list this function prints a
 *         warning, libc ausence is not an error.
 *
 * Parameters:
 *   none.
 *
 * Returns:
 *   0 if succesful, != 0 if any error.
 *
 *****************************************************************************/

static int initLIBCObject(void)
{
  PDI_ELFOBJ *eo;
  char *fname;
  
  /* Search in the object list this library */
  for(eo = _pdi_objlist; eo; eo = eo->o_next)
  {
    /* From the path select only the filename */
    for(fname = eo->o_name + strlen(eo->o_name);
        fname > eo->o_name && *(fname-1) != '/';
        fname--)
      ;
    if(!strncmp("libc.so", fname, strlen("libc.so")))
      break;
  }

  if(!eo)
  {
    _pdi_warning(THIS, "Cannot find 'libc.so.*' in the object list.");
    _pdi_libcobj = NULL;
    return 0;
  }

  /* Make a Direct Access (patent pending) to this object */
  _pdi_libcobj = eo;
  if(_pdi_ebe_setObjectAlias(_pdi_libcobj, PDI_ALIAS_LIBC))
  {
    _pdi_error(THIS, "Cannot assign alias " PDI_ALIAS_LIBC " to '%s'.",
               _pdi_pdiobj->o_name);
    return -1;
  }

  return 0;
}

/*****************************************************************************
 * static int initPDIObject(void)
 *
 * Description:
 *   Search and set alias PDI_ALIAS_PDI to pDI-Tools DSO.
 *
 *   NOTE: pDI-Tools can be linked statically, so is not an error to not find
 *         it. If pDI-Tools DSO (libpdi.so) cannot be found then '_pdi_mainobj'
 *         and '_pdi_pdiobj' are the same and a warning is printed.
 *
 * Parameters:
 *   none.
 *
 * Returns:
 *   0 if succesful, != 0 if any error.
 *
 *****************************************************************************/

static int initPDIObject(void)
{
  PDI_ELFOBJ *eo;
  char *fname;
  
  /* Search in the object list 'libpdi.so' (pDI-Tools DSO) */
  for(eo = _pdi_objlist; eo; eo = eo->o_next)
  {
    /* From the path select only the filename */
    for(fname = eo->o_name + strlen(eo->o_name);
        fname > eo->o_name && *(fname-1) != '/';
        fname--)
      ;
    if(!strcmp("libpdi.so", fname))
      break;
  }

  if(!eo)
  {
    _pdi_warning(THIS, "Cannot find 'libpdi.so' in the object list.");
    _pdi_warning(THIS, "Assuming '%s' as object '%s'.",
                 PDI_ALIAS_MAIN, PDI_ALIAS_PDI);
    _pdi_pdiobj = _pdi_mainobj;
    return 0;
  }

  /* Create a Direct Link (patent pending) to this object and aliase (TM) it */
  _pdi_pdiobj = eo;
  if(_pdi_ebe_setObjectAlias(_pdi_pdiobj, PDI_ALIAS_PDI))
  {
    _pdi_error(THIS, "Cannot assign alias " PDI_ALIAS_PDI " to '%s'.",
               _pdi_pdiobj->o_name);
    return -1;
  }

  return 0;
}

/*****************************************************************************
 * int _pdi_linux_initObjectList(void)
 *
 * Description:
 *   Initializes the arch dependant part of the object list, which is
 *   practically all the object list.
 *
 * Parameters:
 *   none.
 *
 * Returns:
 *   0 if succesful, != 0 if any error.
 *
 *****************************************************************************/

int _pdi_linux_initObjectList(void)
{
  /* Alloc memory for 'archobjlist' pool. It must have the same number */
  /* of entries as the 'objpool' pool of '/src/link/gestobj.c'.        */
  if((archobjlist = malloc(sizeof(PDI_LINUX_ELFOBJ) * PDICFG.max_objects)) == NULL)
  {
    _pdi_debug(THIS, "Cannot alloc memory for 'archobjlist' pool (%d bytes).",
               sizeof(PDI_LINUX_ELFOBJ) * PDICFG.max_objects);
    return -1;
  }
  _pdi_log(THIS, "Reserved %d bytes for 'archobjlist' table.",
           sizeof(PDI_LINUX_ELFOBJ) * PDICFG.max_objects);

  /* Build the object list (initializing it) */
  if(buildObjectList())
  {
    _pdi_debug(THIS, "Cannot initialize object list");
    return -1;
  }

  /* Search basic objects */
  if(initMainObject()) _pdi_debug(THIS, "Cannot find object " PDI_ALIAS_MAIN);
  if(initLIBCObject()) _pdi_debug(THIS, "Cannot find object " PDI_ALIAS_LIBC);
  if(initPDIObject())  _pdi_debug(THIS, "Cannot find object " PDI_ALIAS_PDI);

  return 0;
}

/*****************************************************************************
 * void _pdi_linux_finiObjectList(void)
 *
 * Description:
 *   Free all memory used by the arch dependant part of the object list.
 *
 * Parameters:
 *   none.
 *
 * Returns:
 *   nothing.
 *
 *****************************************************************************/

void _pdi_linux_finiObjectList(void)
{
  free(archobjlist);
  archobjlist = NULL;
}

/*****************************************************************************
 * int _pdi_linux_refreshObjectList(void)
 *
 * Description:
 *   Refreshes object list comparing the current object list with the runtime
 *   linker object list.
 *
 * Parameters:
 *   none.
 *
 * Returns:
 *   0 if succesful, != 0 if any error.
 *
 *****************************************************************************/

int _pdi_linux_refreshObjectList(void)
{
  struct link_map *lm;
  PDI_ELFOBJ *eo, *next, *neo;

  _pdi_debug(THIS, "Removing DSO's not in memory.");
  /* check that all objects in our list are currently in memory */
  for(eo = _pdi_objlist; eo; eo = next)
  {
    /* remember the next 'eo' */
    next = eo->o_next;

    /* search in the list of linkmaps the object 'eo' */
    for(lm = _r_debug.r_map; lm && eo->o_arch->lm != lm; lm = lm->l_next)
      ;

    /* if this object is not in memory send it to go jump in the digital river */
    if(!lm)
    {
      _pdi_debug(THIS, "Remove '%s' from the object list.",
                 _pdi_ebe_getObjectName(eo));
      _pdi_ebe_freeElfObj(eo);
    }
  }

  _pdi_debug(THIS, "Adding new objects.");
  for(lm = _r_debug.r_map, eo = _pdi_objlist; lm; lm = lm->l_next)
  {
    if(!eo || lm != eo->o_arch->lm)
    {
      _pdi_debug(THIS, "Adding '%s' to the object list.", LMNAME(lm));
      
      if((neo = _pdi_ebe_newElfObj(eo)) == NULL)
      {
        _pdi_error(THIS, "Cannot alloc memory for a new PDI_ELFOBJ.");
        return -1;
      }
      
      initObject(neo, lm);
    } else
      eo = eo->o_next;
  }

  return 0;
}

