/*****************************************************************************
 * $Id: lx-callbck.c,v 1.1 2005/08/28 13:41:21 killabyte Exp $
 *
 * This module implements the mechanism used to install callbacks and part of
 * the mechanism to manage callback interpositions in runtime.
 *
 * ---------------------------------------------------------------------------
 * pDI-Tools - portable Dynamic Instrumentation Tools
 *   (C) 2004, 2005 Gerardo Garca Pea
 *   Programmed by Gerardo Garca Pea - Inspired on CEPBA DItools
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301
 *   USA
 *
 *****************************************************************************/

#include<config.h>
#include<ebeif.h>
#include<log.h>
#include<pdiconfig.h>
#include<linux/elfops.h>

/* This is a stub template. The stubs generated from this template will      */
/* allow to jump (literally: JMP, not CALL) to the generic wrapper (function */
/* '_pdi_arch_genericWrapper()'), passing to it some self-identifying info.  */
/* The stub pushes in stack this info (in this order):                       */
/*                                                                           */
/*    - address of the intercepted function (@wrapped_func)                  */
/*    - a pointer to the name of the intercepted function (@func name)       */
/*    - a pointer to the interposition PDI_ARCH_INTERCEPT (@intercept)       */
/*                                                                           */
/* Then jump to the genwrapper with a relative JMP. The offset is calculated */
/* in this way:                                                              */
/*                                                                           */
/*    - GW = absolute address of the genwrapper                              */
/*    - JA = absolute address of the JMP instruction                         */
/*    - TO = absolute destination address                                    */
/*    - TO = (GW - JA + 4) & 0xFFFFFFFF                                      */

static unsigned char template_stub[] = {
  0x68, 0x44, 0x55, 0x66, 0x77,         /* pushl $0x77665544 - @wrapped func */
  0x68, 0x00, 0x11, 0x22, 0x33,         /* pushl $0x33221100 - @func name    */
  0x68, 0x88, 0x99, 0xaa, 0xbb,         /* pushl $0xbbaa9988 - @arch int     */
  0xe9, 0xcc, 0xdd, 0xee, 0xff          /* jmp   0xffeeddcc  - jmp TO        */
};

#define TEMPLATE_STUB_WRAPPED(x)        ((void **) (((char *) (x)) + 1))
#define TEMPLATE_STUB_FUNCNAME(x)       ((void **) (((char *) (x)) + 6))
#define TEMPLATE_STUB_INTERCEPT(x)      ((void **) (((char *) (x)) + 11))
#define TEMPLATE_STUB_GENWRAPPER(x)     ((void **) (((char *) (x)) + 16))
#define CALC_JMP_DESP(stub, to)            \
                ((void *)                  \
                 (((unsigned long) (to)) - \
                  ((unsigned long) TEMPLATE_STUB_GENWRAPPER(stub)) - 4))

/* Declare, but with a very very false prototype, the function      */
/* _pdi_linux_genericWrapper, which is our hardcore generic wrapper */
void _pdi_linux_genericWrapper(void);

/* Module's private variables */
static char *stubs_buffer;
static char *stubs_first_free;
static int stubs_buffer_size;
static int stubs_free;

/*****************************************************************************
 * static void *getNewCallback(PDI_INTERCEPT *i,
 *                             char *func_name,
 *                             void *genericWrapper,
 *                             void *wrapped_func)
 *
 * Description:
 *   Alloc memory and initializes a stub. It returns a pointer to the new stub.
 *
 * Parameters:
 *   i            - interposition that will use this stub
 *   func_name    - a pointer to a string with the name of the function that
 *                  will be intercepted by the new stub
 *   wrapped_func - a pointer to the function that will be intercepted by the
 *                  stub.
 *
 * Returns:
 *   A pointer to the new stub if all was ok, NULL if an error happened.
 *
 *****************************************************************************/

static void *getNewCallback(PDI_INTERCEPT *i,
                            char *func_name,
                            void *genericWrapper,
                            void *wrapped_func)
{
  char *stub;

  for(stub = stubs_first_free;
      stub < stubs_buffer + stubs_buffer_size;
      stub += sizeof(template_stub))
    if(!*stub)
    {
      /* Copy the template to the new stub */
      memcpy(stub, template_stub, sizeof(template_stub));

      /* Apply changes to the stub */
      *TEMPLATE_STUB_WRAPPED(stub)    = wrapped_func;
      *TEMPLATE_STUB_FUNCNAME(stub)   = func_name;
      *TEMPLATE_STUB_INTERCEPT(stub)  = i->i_arch;
      *TEMPLATE_STUB_GENWRAPPER(stub) = CALC_JMP_DESP(stub, genericWrapper);

      /* Extract this stub from the free list */
      stubs_first_free = stub + sizeof(template_stub);
      stubs_free--;

      return stub;
    }

  return NULL;
}

/*****************************************************************************
 * static void freeCallback(void *cb)
 *
 * Description:
 *   Free memory used by an stub.
 *
 * Parameters:
 *   cb - pointer to the stub
 *
 * Returns:
 *   Nothing.
 *
 *****************************************************************************/

static void freeCallback(void *cb)
{
  memset(cb, 0, sizeof(template_stub));
  if((char *) cb < stubs_first_free)
    stubs_first_free = cb;
}

/*****************************************************************************
 * int _pdi_linux_defaultMaxStubs(void)
 *
 * Description:
 *   This function calculates a ideal maximum number for the current platform.
 *
 *   The result of this implementation is the number of stubs that fit in a
 *   memory page.
 *
 * Parameters:
 *   none.
 *
 * Returns:
 *   A positive number different from zero.
 *
 *****************************************************************************/

int _pdi_linux_defaultMaxStubs(void)
{
  return ((PAGESIZE) / sizeof(template_stub));
}

/*****************************************************************************
 * int _pdi_linux_initCallback(void)
 *
 * Description:
 *   Initialize callback management data structures.
 *
 * Parameters:
 *   none.
 *
 * Returns:
 *   0 if all ok, a value different of zero otherwise.
 *
 *****************************************************************************/

int _pdi_linux_initCallback(void)
{
  if(PDICFG.cb_max_stubs < 0)
  {
    _pdi_error(THIS, "Parameter 'cb_max_stubs' has an illegal value (%d).",
               PDICFG.cb_max_stubs);
    return -1;
  }

  if(PDICFG.cb_max_stubs > 0)
  {
    /* Calculate the size of the stubs pool */
    stubs_buffer_size = PDICFG.cb_max_stubs * sizeof(template_stub);

    /* Alloc memory */
    if((stubs_buffer = malloc(stubs_buffer_size)) == NULL)
    {
      stubs_buffer_size = 0;
      _pdi_error(THIS, "Cannot alloc memory (%d bytes) for 'stubs_buffer'.",
                 stubs_buffer_size);
      return -1;
    }
    stubs_free = PDICFG.cb_max_stubs;

    /* Set write, read and execution permissions to the stubs buffer(RWX) */
    if(mprotect(
         (void *)
           ((unsigned long) stubs_buffer & (unsigned long) ~(PAGESIZE - 1)),
         PAGESIZE, PROT_READ | PROT_WRITE | PROT_EXEC))
    {
      _pdi_error(THIS, "Cannot assign RWX permissions to the stubs pool.");
      return -1;
    }
  } else {
    stubs_buffer = NULL;
    stubs_buffer_size = 0;
    stubs_free = 0;
  }

  stubs_first_free = stubs_buffer;

  return 0;
}

/*****************************************************************************
 * void _pdi_linux_finiCallback(void)
 *
 * Description:
 *   Finalizes the callback manager and frees memory.
 *
 * Parameters:
 *   none.
 *
 * Returns:
 *   nothing.
 *
 *****************************************************************************/

void _pdi_linux_finiCallback(void)
{
  if(PDICFG.cb_max_stubs > 0)
    free(stubs_buffer);
  stubs_buffer = NULL;
  stubs_first_free = NULL;
  stubs_buffer_size = 0;
  stubs_free = 0;
}

/*****************************************************************************
 * int _pdi_linux_callback(PDI_ELFOBJ *eo, PDI_INTERCEPT *i)
 *
 * Description:
 *   Install a callback interposition specified by 'i' on object 'eo'.
 *
 * Parameters:
 *   eo - target object
 *   i  - callback parameters
 *
 * Returns:
 *   0 if installation was succesful, -1 otherwise.
 *
 *****************************************************************************/

int _pdi_linux_callback(PDI_ELFOBJ *eo, PDI_INTERCEPT *i)
{
  LINUX_PLTREL *jmpRel;
  void **undo_table,
       *stub,
       *wrapped_func,
       *genwrapper;
  char *name;
  int nrels;
  PDI_LINUX_PLT_UNDO_DATA undo_data;

  /* Calculate how many REL/A structures there are in section DT_JMPREL */
  /* Deducimos cuantos REL/A encontraremos en la seccin DT_JMPREL */
  nrels = eo->o_arch->jmpRelTableSize / eo->o_arch->pltRelEntSize;

  /* First check that there are the sufficient free stubs */
  if(nrels > stubs_free)
  {
    _pdi_error(THIS,
               "There are not so many free stubs "
               "(%d are needed to instrument '%s').",
               nrels, _pdi_ebe_getObjectName(eo));
    return -1;
  }

  /* Alloc memory for the undo table */
  if((i->i_arch->u.callback.undo_table = malloc(nrels * sizeof(void *))) == NULL)
  {
    _pdi_error(THIS,
               "Cannot alloc memory for the callback undo table for '%s'.",
               _pdi_ebe_getObjectName(eo));
    return -1;
  }

  /* Intiliaze the arch dependant part of the structure */
  i->i_arch->u.callback.beCallbackRequired  = i->i_backend->o_beCallbackRequired;
  i->i_arch->u.callback.bePreEventCallback  = i->i_backend->o_bePreEventCallback;
  i->i_arch->u.callback.bePostEventCallback = i->i_backend->o_bePostEventCallback;

  /* Get a pointer to the generic wrapper */
  genwrapper = i->i_wrapper
                 ? i->i_wrapper
                 : _pdi_linux_genericWrapper;

  /* For each entry REL/A in DT_JMPREL install a stub that jumps to the      */
  /* generic wrapper. The stubs only will be installed on entries of type    */
  /* LINUX_R_JMPREL_TYPE. The original references will be stored in the      */
  /* table 'i->i_arch->u.callback.undo_table'. If one reference is not saved */
  /* (because it is not modified), it is saved as NULL in the undo table.    */
  undo_table = i->i_arch->u.callback.undo_table;
  for(jmpRel = (LINUX_PLTREL *) eo->o_arch->jmpRelTable;
      jmpRel < (LINUX_PLTREL *) (eo->o_arch->jmpRelTable + eo->o_arch->jmpRelTableSize);
      jmpRel++)
  {
    if(ELFW_R_TYPE(jmpRel->r_info) == LINUX_R_JMPREL_TYPE)
    {
      /* Get the name of the function relocated by this entry */
      name = PDI_GET_SYMBOL_NAME(eo->o_arch, ELFW_R_SYM(jmpRel->r_info));

      if((wrapped_func = _pdi_linux_resolveSymbol(name)) == NULL)
      {
        _pdi_error(THIS, "Cannot resolve function '%s'.", name);
        _pdi_error(THIS, "Cannot recover from this error. Abort!");
        abort();
      }

      /* Create a stub for this entry */
      if((stub = getNewCallback(i, name, genwrapper, wrapped_func)) == NULL)
      {
        /* Argh! Some weird error that cannot happen have been happened   */
        /* (fuck! I checked for memory before doing anything!). This is a */
        /* so nasty situation that the best we can do is abort pDI-Tools! */
        _pdi_error(THIS,
                   "Cannot create a stub while installing a callback on '%s'.",
                   _pdi_ebe_getObjectName(eo));
        _pdi_error(THIS, "Cannot recover from this error. Abort!");
        abort();
      }

      /* Relink this function to the new stub saving the original address */
      if(_pdi_linux_alterPlt(eo, jmpRel, stub, &undo_data))
      {
        _pdi_error(THIS, "Cannot relink symbol '%s::%s()' to a stub.",
                   _pdi_ebe_getObjectName(eo), name);
        _pdi_error(THIS, "Cannot recover from this error. Abort!");
        abort();
      } else
        *undo_table = undo_data.undo_addr;
    } else
      *undo_table = NULL;
    undo_table++;
  }

  return 0;
}

/*****************************************************************************
 * int _pdi_linux_undoCallback(PDI_ELFOBJ *eo, PDI_INTERCEPT *i)
 *
 * Description:
 *   Uninstall callback 'i' from object 'eo'.
 *
 * Parameters:
 *   eo - object
 *   i  - callback parameters
 *
 * Returns:
 *   0 if all ok, -1 otherwise.
 *
 *****************************************************************************/

int _pdi_linux_undoCallback(PDI_ELFOBJ *eo, PDI_INTERCEPT *i)
{
  LINUX_PLTREL *jmpRel;
  void **undo_table;

  undo_table = i->i_arch->u.callback.undo_table;
  for(jmpRel = (LINUX_PLTREL *) eo->o_arch->jmpRelTable;
      jmpRel < (LINUX_PLTREL *) (eo->o_arch->jmpRelTable + eo->o_arch->jmpRelTableSize);
      jmpRel++)
  {
    if(ELFW_R_TYPE(jmpRel->r_info) == LINUX_R_JMPREL_TYPE)
    {
      freeCallback(*((void **) jmpRel->r_offset));
      if(_pdi_linux_alterPlt(eo, jmpRel, *undo_table, NULL))
      {
        _pdi_error(THIS, "An error happened while restoring a callback relink.");
        return -1;
      }
    }

    undo_table++;
  }

  free(i->i_arch->u.callback.undo_table);
  i->i_arch->u.callback.undo_table = NULL;

  return 0;
}

