/*****************************************************************************
 * $Id: lx-alterplt.c,v 1.1 2005/08/25 13:07:19 killabyte Exp $
 *
 * This file implements a mechanism used to alter easily the content of the
 * GOT/PLT tables.
 *
 * ---------------------------------------------------------------------------
 * pDI-Tools - portable Dynamic Instrumentation Tools
 *   (C) 2004, 2005 Gerardo Garca Pea
 *   Programmed by Gerardo Garca Pea - Inspired on CEPBA DItools
 *
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Lesser General Public
 *   License as published by the Free Software Foundation; either
 *   version 2.1 of the License, or (at your option) any later version.
 *
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public
 *   License along with this library; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301
 *   USA
 *
 *****************************************************************************/

#include<config.h>
#include<ebeif.h>
#include<log.h>
#include<linux/elfops.h>

/*****************************************************************************
 * int _pdi_linux_alterPlt(PDI_ELFOBJ *eo, LINUX_PLTREL *rel, void *target,
 *                         PDI_LINUX_PLT_UNDO_DATA *undo_data)
 *
 * Description:
 *   Modify the PLT/GOT entry selected by 'rel' in the object 'eo',
 *   substituting the current pointer by 'target', and saving to 'undo_data'
 *   (if it is not NULL) the current data.
 *
 * Parameters:
 *   eo        - target object
 *   rel       - relocation structure
 *   target    - a pointer to the new definition
 *   undo_data - A pointer to a buffer where save the original PLT/GOT data. If
 *               it is NULL the current PLT/GOT info is discarded.
 *
 * Devuelve:
 *   0 if all ok, other value otherwise.
 *
 *****************************************************************************/

int _pdi_linux_alterPlt(PDI_ELFOBJ *eo, LINUX_PLTREL *rel, void *target,
                        PDI_LINUX_PLT_UNDO_DATA *undo_data)
{
  ElfW(Addr) plt_address;
#if PLATFORM_ASM == PDI_ASM_PPC && PDI_LINUX_WORDSIZE == 32
  int rel_index;
  ElfW(Addr) jmpval;
#endif

  /***************************************************************************
   * IA32 - 386, 486, Pentium, AMD K6, K7, XP ...
   ***************************************************************************/
#if PLATFORM_ASM == PDI_ASM_IA32
  plt_address = rel->r_offset + eo->o_arch->lm->l_addr;
  if(undo_data)
  {
    undo_data->addr_addr = (void **) plt_address;
    undo_data->undo_addr = *((void **) plt_address);
  }
  *((void **) plt_address) = target;

  /***************************************************************************
   * IA64 - Itanium, Itanium II
   ***************************************************************************/
#elif PLATFORM_ASM == PDI_ASM_IA64
#error No se como modificar el PLT en IA64

  /***************************************************************************
   * PowerPC 32 Bits - G3, G4, G5...
   ***************************************************************************/
#elif PLATFORM_ASM == PDI_ASM_PPC && PDI_LINUX_WORDSIZE == 32
  /* 1 - Calculate the index of this REL structure in DT_JMPREL table */
  rel_index = (((ElfW(Addr)) rel) - eo->o_arch->jmpRelTable)
            / eo->o_arch->pltRelEntSize;

  /* 2 - Calculate pointer to symbol position in PLT:          */
  /*     18*4 + 8 * N + 4 * i                                  */
  /*       where N is the number of entries in DT_JMPREL       */
  /*         == DT_PLTRELSZ / DT_RELAENT                       */
  /*       and 'i' is the REL index of the symbol in DT_JMPREL */
  plt_address = eo->o_arch->gotTable + 18*4
              + 8 * (eo->o_arch->jmpRelTableSize / eo->o_arch->pltRelEntSize);

  /* 3 - Load the new pointer in PLT and, if demanded, */
  /*     save the original value                       */
  if(undo_data)
  {
    undo_data->addr_addr = ((ElfW(Addr) *) plt_address) + rel_index;
    undo_data->undo_addr = *(undo_data->addr_addr);
  }
  ((ElfW(Addr) *) plt_address)[rel_index] = (ElfW(Addr)) target;

  /* 4 - Insert one or two instructions in PLT in position 'rel_index'. If   */
  /*     the symbol is near, insert a branch instruction to jump to it       */
  /*     directly. If the symbol is too far for a simple branch, then insert */
  /*     two instructions to make an indirect jump with GOT and the code at  */
  /*     the begining of PLT.                                                */
  /*     NOTE: THIS IMPLEMENTATION OF PLT/GOT IS SLIGHTLY DIFFERENT FROM THE */
  /*     "POWERPC ELF ABI SUPPLEMENT" SPECIFICATION (the indirect jump code  */
  /*     should be in offset @(DT_JMPREL) + 6*4, and it doesn't explain      */
  /*     nothing about near direct jumps)                                    */
  plt_address = rel->r_offset + eo->o_arch->lm->l_addr;
  if(undo_data)
  {
    undo_data->addr_data = (void *) plt_address;
    memcpy(undo_data->undo_data, (void *) plt_address, 8);
  }

  jmpval = (signed) (((ElfW(Addr)) target) - plt_address) >> 2;

  if((jmpval & 0xff000000) != 0x00000000
  && (jmpval & 0xff000000) != 0xff000000)
  {
    /*    ==> addi r11, r0, 4*rel_index */
    ((unsigned *) plt_address)[0] =       /* bits  => valor     */
      ((14              & 0x003f) << 26)  /* 26-31 => addi      */
    | ((11              & 0x001f) << 21)  /* 21-25 => r11       */
    | ((0               & 0x001f) << 16)  /* 16-20 => r0        */
    | (((rel_index * 4) & 0xffff) <<  0); /* 16-31 => immediate */
    /*    ==> b .PLT */
    jmpval = ((eo->o_arch->lm->l_addr + eo->o_arch->gotTable) - (plt_address + 4)) >> 2;
    ((unsigned *) plt_address)[1] =
      ((18     & 0x00003f) << 26)  /* 26-31 => b         */
    | ((jmpval & 0xffffff) <<  2)  /* 30-02 => @salto    */
    | ((0      & 0x000001) <<  1)  /*   1   => r0        */
    | ((0      & 0x000001) <<  0); /*   0   => immediate */
  } else  {
    /*    ==> b .wrapper */
    ((unsigned *) plt_address)[0] =
      ((18     & 0x00003f) << 26)  /* 26-31 => b         */
    | ((jmpval & 0xffffff) <<  2)  /* 30-02 => @salto    */
    | ((0      & 0x000001) <<  1)  /*   1   => r0        */
    | ((0      & 0x000001) <<  0); /*   0   => immediate */
  }

  /***************************************************************************
   * PowerPC 64 Bits - G5, POWER3, POWER4, POWER4+, POWER5, ...
   ***************************************************************************/
#elif PLATFORM_ASM == PDI_ASM_PPC && PDI_LINUX_WORDSIZE == 64
  /* Calculate the address of the function descriptor */
  plt_address = eo->o_arch->lm->l_addr + rel->r_offset;

  /* Make a copy of this function descriptor (it is the undo data) */
  if(undo_data)
  {
    undo_data->addr_fd = (void *) plt_address;
    memcpy(undo_data->func_desc, (void *) plt_address, sizeof(undo_data->func_desc));
  }

  /* Install the interposition */
  memcpy((void *) plt_address, target, sizeof(ElfW(Addr)) * 3);

  /***************************************************************************
   * Unknown platform
   ***************************************************************************/
#else
#error "I don't know how to alter PLT/GOT in this platform"
#endif

  return 0;
}

/*****************************************************************************
 * void _pdi_linux_restorePlt(PDI_LINUX_PLT_UNDO_DATA *undo_data)
 *
 * Description:
 *   Restore a GOT slot with info stored in 'undo_data'.
 *
 * Parameters:
 *   undo_data - Undo data :)
 *
 * Devuelve:
 *   -nothing-
 *
 *****************************************************************************/

void _pdi_linux_restorePlt(PDI_LINUX_PLT_UNDO_DATA *undo_data)
{
  /***************************************************************************
   * IA32 - 386, 486, Pentium, AMD K6, K7, XP ...
   ***************************************************************************/
#if PLATFORM_ASM == PDI_ASM_IA32
  *(undo_data->addr_addr) = undo_data->undo_addr;

  /***************************************************************************
   * PowerPC 32 - POWER3, POWER4+, G3, G4, G5...
   ***************************************************************************/
#elif PLATFORM_ASM == PDI_ASM_PPC && PDI_LINUX_WORDSIZE == 32
  memcpy(undo_data->undo_data, undo_data->addr_data, 8);
  *(undo_data->addr_addr) = undo_data->undo_addr;

  /***************************************************************************
   * PowerPC 64 - G5, POWER3, POWER4+, POWER5, ...
   ***************************************************************************/
#elif PLATFORM_ASM == PDI_ASM_PPC && PDI_LINUX_WORDSIZE == 64
  memcpy(undo_data->addr_fd, undo_data->func_desc, sizeof(undo_data->func_desc));

  /***************************************************************************
   * Plataformas desconocidas
   ***************************************************************************/
#else
#error "I don't know how to undo PLT/GOT modifications in this platform"
#endif

  /* Set structure contents to zero */
  memset(undo_data, 0, sizeof(PDI_LINUX_PLT_UNDO_DATA));
}

